package com.ibm.ulc.ui.lists;

/*
 * Instances of this class hold a table of UiRows. They are the items of a <code>IListModel</code>
 * that have been loaded into the UI
 */
import java.util.Vector;
import java.util.Enumeration;
import javax.swing.event.EventListenerList;
import com.ibm.ulc.util.UlcHashtable;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.UlcObject;
import com.ibm.ulc.comm.Common;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.ui.base.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.base.IDefaults;
public class UiItemCache extends UlcObject implements IDefaults {
	protected UlcHashtable fData = new UlcHashtable();
	protected IRow[] fEditCache = new IRow[0];
	private UiInvalidRow fInvalidRow = UiInvalidRow.getInstance();
	protected IRow fCurrentlyEditedRow = fInvalidRow;
	protected UITableModel fModel = null;
	protected EventListenerList fListeners = new EventListenerList();
	protected int fRequestsStarted = 0;
	protected Vector fRequestedRows = new Vector();
	protected Vector fRequestedAttributes = new Vector();
	protected Vector fPreloadAttributes = new Vector();
/**
 * handle the case of the missing row
 */
public UiItemCache(UITableModel model) {
	fModel = model;
}
/**
 * Return true if the component for which I am an enabler should be enabled.
 *
 */
protected void addListener(IItemCacheListener listener) {
	fListeners.add(IItemCacheListener.class, listener);
}
/**
 * handle the case of the missing row
 */
private void addPendingRow(int oid) {
	if (fRequestsStarted > 0 && (!fRequestedRows.contains(new Integer(oid)))) {
		fRequestedRows.addElement(new Integer(oid));
	}
}
/**
 * handle the case of the missing row
 */
private void allChanged() {
	fEditCache = new IRow[0];
	Enumeration elements = fData.elements();
	while (elements.hasMoreElements()) {
		IRow row = (IRow) elements.nextElement();
		row.resetValues(this);
	}
}
protected void cancelInput() {
	IRow[] resetRows = resetInput(fEditCache);
	fCurrentlyEditedRow = fInvalidRow;
	if (resetRows.length > 0)
		notifyRowsSet(resetRows);
}
protected void cancelInput(IRow[] rows) {
	IRow[] resetRows = resetInput(rows);
	if (resetRows.length > 0)
		notifyRowsSet(resetRows);
}
private void cancelInput(Anything rowId) {
	if (rowId != null) {
		if (!rowId.isNull()) {
			UIRowModel model = (UIRowModel) fModel.getManaged(UIRowModel.class, fModel.getConnection(), rowId);
			if (model != null && model.fRowId > -1) {
				IRow row = getRow(model.fRowId);
				if (row != null) {
					cancelInput(new IRow[] {row});
					return;
				}
			}
		}
	}
	cancelInput();
}
/**
 * Invalidate cell range.
 * Typically called from ULC via "change".
 */
private void change(Anything args) {
	// change notification from ULC

	int type = args.get("type", TABLE_MODEL_CONTENTS_CHANGED);
	switch (type) {
		case TABLE_MODEL_CONTENTS_CHANGED :
			//changedContents(args);
			break;
		case TABLE_MODEL_ROWS_CHANGED :
			//rowsChanged(extractRowIds(args));
			break;
		case TABLE_MODEL_CELL_CHANGED :
			//cellsChanged(args);
			break;
		case TABLE_MODEL_ROWS_ADDED :
			//rowsAdded(args, true);
			break;
		case TABLE_MODEL_ROWS_REMOVED :
			//int newSize = args.get("rows", 0);
			//if (newSize == 0)
			//	contentsCleared();
			//else
			//	rowsRemoved(extractRowIds(args));
			break;
	}
}
/**
 * Invalidate cell range.
 * Typically called from ULC via "change".
 */
private void changedContents(Anything args) {
	cancelInput();
	fData.clear();
	setData(args, false);
}
private int currentlyEditedOid() {
	return fCurrentlyEditedRow.getOid();
}
/**
 * Invalidate cell range.
 * Typically called from ULC via "change".
 */
protected Anything extractRowIds(Anything args, String identifier) {
	Anything answer = args.get(identifier);
	if (answer == null || answer.isNull()) {
		answer = new Anything();
		if (args.isDefined("ot") && args.isDefined("of")) {
			int from = args.get("of", -1);
			int to = args.get("ot", -1);
			if (from < to) {
				for (int i = from; i <= to; i++) {
					answer.append(new Anything(i));
				}
			}
			else {
				for (int i = from; i >= to; i--) {
					answer.append(new Anything(i));
				}
			}
		}
		else {
			if (args.isDefined("it") && args.isDefined("if")) {
				int from = args.get("if", -1);
				int to = args.get("it", -1);
				if (from > -1 && to > -1)
					for (int i = from; i <= to; i++) {
						answer.append(new Anything(i));
					}
			}
			else { //trouble("UITableModel.change", "received null rowIds");
			}
		}
	}
	return answer;
}
/**
 * Return true if the component for which I am an enabler should be enabled.
 *
 */
protected void free() {
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
public Object getExistingValue(int oid, String attributeName) {
	IRow row;
	Object object = fData.get((Integer) new Integer(oid));
	if (object == null) {
		row = fInvalidRow;
	}
	else {
		row = (IRow) object;
	}
	return row.getExistingValueAt(attributeName, this);
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
IRow getInsertedRow(int oid) {
	IRow answer;
	Object object = fData.get((Integer) new Integer(oid));
	if (object == null) {
		answer = new UiPendingRow(oid);
		fData.put(new Integer(oid), answer);
	} else {
		answer = (IRow) object;
	}
	return answer;
}
public UITableModel getModel() {
	return fModel;
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
private IRow getNewRow(int oid) {
	IRow answer;
	Object object = fData.get((Integer) new Integer(oid));
	if (object == null) {
		answer = new UiRow(oid);
	}
	else {
		answer = ((IRow) object).asNonPendingRow(this, oid);
	}
	fData.put((Integer) new Integer(oid), answer);
	return answer;
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
private IRow getNonPendingRow(int oid) {
	IRow answer;
	Object object = fData.get((Integer) new Integer(oid));
	if (object == null) {
		answer = handleMissingRow(oid);
	}
	else {
		answer = ((IRow) object).asNonPendingRow(this, oid);
		fData.put((Integer) new Integer(oid), answer);
	}
	return answer;
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
public IRow getRow(int oid) {
	IRow answer;
	Object object = fData.get((Integer) new Integer(oid));
	if (object == null) {
		answer = handleMissingRow(oid);
	} else {
		answer = (IRow) object;
	}
	return answer;
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
public Object getValue(int oid, String attributeName) {
	if (fRequestsStarted <= 0) {
		return getExistingValue(oid, attributeName);
	}
	else {
		IRow row = getRow(oid);
		return row.getValueAt(attributeName, this);
	}
}
/**
 * handle the case of the missing row
 */
private IRow handleMissingRow(int oid) {
	IRow answer = null;
	if (oid < 0) {
		answer = fInvalidRow;
	} else {
		answer = new UiPendingRow(oid);
		fData.put(new Integer(oid), answer);
		addPendingRow(oid);
	}
	return answer;
}
/*
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
protected boolean handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("setAttributes")) {
		setAttributes(args);
		return true;
	}
	if (request.equals("setData")) {
		if (args.get("data") == null || (args.get("data").size() == 0)) {
			trouble("setData", "received an update without rowIds");
			return true;
		}
		setData(args, true);
		return true;
	}
	if (request.equals("change")) { // change notifications from ULC
		change(args);
		return true;
	}
	if (request.equals("removeRows")) {
		rowsRemoved(args);
		return true;
	}
	if (request.equals("rowsChanged")) {
		rowsChanged(args);
		return true;
	}
	if (request.equals("allChanged")) {
		allChanged();
		return true;
	}
	if (request.equals("contentsChanged")) {
		changedContents(args);
		return true;
	}
	if (request.equals("setRows")) {
		setRows(args, true);
		return true;
	}
	if (request.equals("setAttributes")) {
		setAttributes(args);
		return true;
	}
	if (request.equals("cancelInput")) {
		cancelInput(args);
		return true;
	}
	if (request.equals("flush")) {
		saveInput(args);
		return true;
	}
	return false;
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
public boolean hasChanges() {
	return fEditCache.length > 0;
}
/**
 * handle the case of the missing row
 */
private void notifyRowsSet(IRow[] setRows) {
	Object[] listeners = fListeners.getListenerList();
	for (int i = listeners.length - 2; i >= 0; i -= 2) {
		if (listeners[i] == IItemCacheListener.class) {
			((IItemCacheListener) listeners[i + 1]).notifyRowsSet(setRows);
		}
	}
}
protected void removeEdited(IRow row) {
	int index = -1;
	for (int i = 0; i < fEditCache.length; i++) {
		if (fEditCache[i] == row) {
			index = i;
			break;
		}
	}
	if (index > -1) {
		int newLength = fEditCache.length - 1;
		if (newLength > 0) {
			IRow[] newCache = new IRow[newLength];
			if (index > 0)
				System.arraycopy(fEditCache, 0, newCache, 0, index );
			if (index < fEditCache.length)
				System.arraycopy(fEditCache, index + 1, newCache, index, (newCache.length - index));
			fEditCache = newCache;
		}
		else {
			fEditCache = new IRow[0];
			fModel.updateEmptyStateListeners(hasChanges());
		}
	}
}
/**
 * Return true if the component for which I am an enabler should be enabled.
 *
 */
protected void removeListener(IItemCacheListener listener) {
	fListeners.remove(IItemCacheListener.class, listener);
}
/**
 * handle the case of the missing row
 */
protected Object requestAttribute(int oid, String attributeName) {
	getRow(oid); // ensure the object is also requested
	addPendingRow(oid);
	return requestAttribute(attributeName);
}
/**
 * handle the case of the missing row
 */
protected Object requestAttribute(String attributeName) {
	if (!fRequestedAttributes.contains(attributeName)) {
		fRequestedAttributes.addElement(attributeName);
	}
	return PendingCell.thePendingCell();
}
private IRow[] resetInput(IRow[] rows) {
	if (rows.length > 0) {
		for (int i = 0; i < rows.length; i++) {
			rows[i].cancelInput(this);
			if (fCurrentlyEditedRow == rows[i])
				fCurrentlyEditedRow = fInvalidRow;
		}
	}
	fModel.updateEmptyStateListeners(hasChanges());
	return rows;
}
/**
 * handle the case of the missing row
 */
private void restoreRow(IRow row, String[] attributeNames, Anything rowAny) {
	Anything attributeAny = rowAny.get("a");
	if (attributeAny != null && !attributeAny.isNull()) {
		for (int i = 0; i < attributeNames.length; i++) {
			String attribute = attributeNames[i];
			if (attribute != null) {
				row.setValue(this, attribute, fModel.convert(fModel.getConnection(), attributeAny.get(i)));
			}
		}
	}
}
/**
 * handle the case of the missing row
 */
private void restoreRow(IRow row, Anything rowAny) {
	Anything attributeAny = rowAny.get("a");
	if (attributeAny != null && !attributeAny.isNull()) {
		Enumeration attributes = attributeAny.keys();
		while (attributes.hasMoreElements()) {
			String attribute = (String) attributes.nextElement();
			if (attribute != null) {
				row.setValue(this, attribute, fModel.convert(fModel.getConnection(), attributeAny.get(attribute)));
			}
		}
	}
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(Anything args) {
	Anything cols = args.get("preloadColumns");
	if (cols != null) {
		for (int col = 0; col < cols.size(); col++) {
			String name = (cols.get(col)).asString("n/a");
			if (name != "n/a") {
				fPreloadAttributes.addElement(name);
			}
		}
	}
	setData(args, false);
}
/**
 * Invalidate cell range.
 * Typically called from ULC via "change".
 */
private void rowsChanged(Anything args) {
	Anything rowIds = extractRowIds(args, "rowids");
	boolean clearAll = args.get("clearAll", false);
	Anything attributes = args.get("a");
	String[] attributeNames = new String[0];
	if (attributes != null && !attributes.isNull()) {
		attributeNames = new String[attributes.size()];
		for (int i = 0; i < attributes.size(); i++) {
			attributeNames[i] = attributes.get(i).toString();
		}
	}
	IRow[] rowsChanged = new IRow[rowIds.size()];
	for (int i = 0; i < rowIds.size(); i++) {
		IRow row = getRow(rowIds.get(i).asInt(-1));
		rowsChanged[i] = row;
		if (clearAll)
			row.changedAttributes(new String[0]);
		else
			row.changedAttributes(attributeNames);
	}
	Anything data = args.get("data");
	for (int i = 0; i < data.size(); i++) {
		Anything rowAny = data.get(i);
		if (rowAny != null) {
			int oid = rowAny.get("oid", -1);
			IRow row = getRow(oid);
			restoreRow(row, attributeNames, rowAny);
		}
	}
	notifyRowsSet(rowsChanged);
}
/**
 * Invalidate cell range.
 * Typically called from ULC via "change".
 */
private void rowsRemoved(Anything args) {
	Anything rowIds = extractRowIds(args, "rowids");
	// change notification from ULC
	IRow[] rowsRemoved = new IRow[rowIds.size()];
	for (int i = 0; i < rowIds.size(); i++) {
		IRow row = getRow(rowIds.get(i).asInt(-1));
		//fData.remove(new Integer(rowIds.get(i).asInt(-1)));
		rowsRemoved[i] = row;
	}
	Object[] listeners = fListeners.getListenerList();
	for (int i = listeners.length - 2; i >= 0; i -= 2) {
		if (listeners[i] == IItemCacheListener.class) {
			((IItemCacheListener) listeners[i + 1]).notifyRowsRemoved(rowsRemoved);
		}
	}
	for (int i = 0; i < rowsRemoved.length; i++) {
		fData.remove(new Integer(rowsRemoved[i].getOid()));
	}
}
protected void saveInput() {
	saveInput(-1, fEditCache);
}
public void saveInput(int saveInputId, IRow[] rows) {
	IRow[] workRows = new IRow[rows.length];
	System.arraycopy(rows, 0, workRows, 0, rows.length);
	Anything anything = new Anything();
	Anything rowIds = new Anything();
	Anything changes = new Anything();
	for (int i = 0; i < workRows.length; i++) {
		workRows[i].prepareChanges(fModel, rowIds, changes);
	}
	resetInput(workRows);
	anything.put("siid", saveInputId);
	anything.put("rowids", rowIds);
	anything.put("changes", changes);
	sendULC("setData", anything);
	fModel.updateEmptyStateListeners(hasChanges());
}
private void saveInput(Anything args) {
	int siid = args.get("siid", -1);
	Anything modelAny = args.get("oid");
	if (modelAny != null) {
		if (!modelAny.isNull()) {
			UIRowModel model = (UIRowModel) fModel.getManaged(UIRowModel.class, fModel.getConnection(), modelAny);
			if (model != null && model.fRowId > -1) {
				IRow row = getRow(model.fRowId);
				if (row != null) {
					saveInput(siid, new IRow[] {row});
					return;
				}
			}
		}
	}
	saveInput(siid, fEditCache);
}
/**
 * Return true if the component for which I am an enabler should be enabled.
 *
 */
private void sendULC(String command, Anything anything) {
	fModel.sendULC(command, anything);
}
/**
 * handle the update of a sparse collection of rows and attribute values
 */
protected void setAttributes(Anything args) {
	Anything update = args.get("update");
	IRow[] changedRows = new IRow[update.size()];
	for (int i = 0; i < update.size(); i++) {
		Anything entry = update.get(i);
		int oid = entry.get("oid").asInt(-1);
		Anything attributeNames = entry.get("an");
		Anything attributeValues = entry.get("av");
		IRow row = getNewRow(oid);
		changedRows[i] = row;
		if (attributeNames != null && attributeValues != null)
			row.setData(fModel, attributeNames, attributeValues);
	}
	notifyRowsSet(changedRows);
}
/**
 * Set the data directly without trashing the edit cache.
 * This method handles the initial getData requests.
 */
protected void setData(Anything args, boolean notify) {
	Anything attributeNames = args.get("an");
	Anything data = args.get("data");
	Vector changedRows = new Vector();
	if (data != null) {
		for (int di = 0; di < data.size(); di++) {
			Anything range = data.get(di);
			Anything rowIds = extractRowIds(range, "rowids");
			if (rowIds != null) {
				Anything attributeValues = range.get("av");
				for (int i = 0; i < rowIds.size(); i++) {
					int oid = rowIds.get(i).asInt(-1);
					if (oid >= 0) {
						IRow row = getNewRow(oid);
						changedRows.addElement(row);
						if (attributeNames != null && attributeValues != null)
							row.setData(fModel, attributeNames, attributeValues.get(i));
					}
				}
			}
		}
	}
	if (notify) {
		IRow[] rows = new IRow[changedRows.size()];
		changedRows.copyInto(rows);
		notifyRowsSet(rows);
	}
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
private IRow setRow(int oid, IRow row) {
	IRow answer = row;
	if (row == null) {
		answer = handleMissingRow(oid);
	}
	else {
		Object object = fData.put((Integer) new Integer(oid), row);
	}
	return answer;
}
/**
 * handle the case of the missing row
 */
private void setRows(Anything args, boolean notify) {
	Anything anything = args.get("data");
	int rows = (anything == null ? 0 : anything.size());
	IRow[] addedRows = new IRow[rows];
	for (int i = 0; i < rows; i++) {
		Anything rowAny = anything.get(i);
		if (rowAny != null) {
			int oid = rowAny.get("oid", -1);
			IRow row = getNewRow(oid);
			addedRows[i] = row;
			restoreRow(row, rowAny);
		}
	}
	if (notify) {
		notifyRowsSet(addedRows);
	}
}
public void setRowToBeEdited(int oid) {
	IRow rowToBeEdited = getRow(oid);
	if (fCurrentlyEditedRow != rowToBeEdited) {
		if (fModel.getNotificationPolicy() == TABLE_EDIT_UPDATE_ON_ROW_CHANGE)
			saveInput();
		fCurrentlyEditedRow = rowToBeEdited;
		//rowToBeEdited.startEditing(this);
	}
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
public void setValue(int oid, String attributeName, Object value, int notificationPolicy) {
	setValue(oid, attributeName, value, null, notificationPolicy);
}
/**
 * answer the instance of <code>UiRow</code> that is defined by <code>oid</code>. answer
 * the PendingItem if the row is not already present, and request the data from ULC
 */
public void setValue(int oid, String attributeName, Object value, IFormModelListener sender, int notificationPolicy) {
	IRow row = fCurrentlyEditedRow;
	if (currentlyEditedOid() != oid)
		row = getRow(oid);
	if (row.setInput(this, attributeName, value)) {
		Object[] listeners = fListeners.getListenerList();
		for (int i = listeners.length - 2; i >= 0; i -= 2) {
			if (listeners[i] == IItemCacheListener.class) {
				IItemCacheListener l = (IItemCacheListener) listeners[i + 1];
				if (l != sender)
					l.notifyCellChanged(row, attributeName);
			}
		}
		if ((notificationPolicy == TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE) && row.hasUnsavedChanges())
			saveInput();
	}
}
public void startedEditing(IRow row) {
	int newLength = fEditCache.length + 1;
	IRow[] newCache = new IRow[newLength];
	System.arraycopy(fEditCache, 0, newCache, 1, fEditCache.length);
	fEditCache = newCache;
	fEditCache[0] = row;
	if (fEditCache.length == 1)
		fModel.updateEmptyStateListeners(true);
}
/**
 * Returns the value at the specified index.  
 */
public void startRequest() {
	if (fRequestsStarted == 0) {
		if (fRequestedAttributes == null)
			fRequestedAttributes = new Vector();
		if (fRequestedAttributes.size() > 0 || fRequestedRows.size() > 0) {
			trouble("startRequest", "should have empty requestedAttributes and requestedRows");
			fRequestedAttributes = new Vector();
			fRequestedRows = new Vector();
		}
	}
	fRequestsStarted++;
}
/**
 * Returns the value at the specified index.  
 */
public void stopRequest() {
	fRequestsStarted--;
	if (fRequestsStarted == 0) {
		if ((fRequestedRows.size() > 0) && fRequestedAttributes.size() > 0) {
			Anything anything = new Anything();
			Vector ranges = UlcRange.createFromVector(fRequestedRows);
			if (ranges.size() == 1) {
				UlcRange range = (UlcRange) ranges.elementAt(0);
				anything.put("of", range.fStartIndex);
				anything.put("ot", range.fEndIndex);
			} else {
				anything.put("rowids", new Anything(fRequestedRows));
			}
			Vector attributes = fRequestedAttributes;
			anything.put("an", new Anything(attributes));
			sendULC("requestAttributes", anything);
		}
		fRequestedRows = new Vector();
		fRequestedAttributes = new Vector();
	}
}
}
