package com.ibm.ulc.ui.dataTypes;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 *
 * This class is used for ensuring that the input to a text field
 * a value that can be represented as a percentage.
 */
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.ui.UIFormComponent;
public class UIPercentValidator extends UIDataType {
	/**
	 * The number of digits allowed after the decimal point.
	 * @serial
	 */
	protected int fFractDigits = 2;
	/**
	 * The list of valid characters allowed as input
	 * @serial
	 */
	protected static String validChars = "0123456789.% ";
/**
 * UIPercentValidator constructor comment.
 */
public UIPercentValidator() {
	super();
}
/**
 * Convert the input string to a percentage value. Throw an exception on error.
 *
 * @param phase int	The current validation phase.
 * The phase can be one of the following:
 * <pre>
 *	FORM_NOTIFICATION_IMMEADIATE : Typically no validation is performed in this case the input string is returned as is.
 *	FORM_NOTIFICATION_ON_FOCUS_CHANGE 	 
 * 	FORM_NOTIFICATION_ON_REQUEST
 *	</pre>
 * @param newString The String to convert.
 * @return An object representing the converted String or null.
 *
 * @see IDataType#convertToObject(int, String, Object)
 */

public Object convertToObject(int phase, String newString, Object previousValue) throws DataTypeConversionException {
	if (phase == FORM_NOTIFICATION_IMMEDIATE)
		return newString; // no checking
	if (newString != null && newString.length() > 0) {
		Double d = parse(newString);
		if (d == null)
			throw new DataTypeConversionException("format error", previousValue);
		return d;
	}
	return null;
}
/**
 * Return the String represenation of the object or the empty string if object is null.
 * The default implementation calls the toString() to return a string representation of the object.
 * Subclasses should override this method to return the formatted data type as a String.
 *
 * @see IDataType#convertToString(Object, boolean)
 */
public String convertToString(Object object, boolean forEditing) {
	if (object == null)
		return "";
	if (object instanceof String) {
		// If we receive a String we have to convert it into a number first
		object = parse(object.toString());
		if (object == null)
			object = "";
	}
	if (object instanceof Double || object instanceof Integer) {
		if (!forEditing)
			return limitFractionDigits(object.toString()) + " %";
	}
	if (object != null)
		return limitFractionDigits(object.toString());
	return limitFractionDigits(getDefaultValue("").toString()) + " %";
}
/**
 * The default implementation returns original string without modification.
 *
 * @see IDataType#filterInput(String)
 */
public String filterInput(String newString) {
	if (hasValidCharacters(newString))
		return newString;
	return null;
}
public int getFractionLength(String s) {
	int i = s.indexOf('.');
	if (i < 0)
		return 0;
	return s.length() - (i + 1);
}
public boolean hasValidCharacters(String s) {
	for (int i= 0; i < s.length(); i++)
		if (validChars.indexOf(s.charAt(i)) < 0)
			return false;
	return true;
}
public String limitFractionDigits(String s) {
	int i = s.indexOf('.');
	String r;
	if (i >= 0) {
		if (fFractDigits > 0)
			r = s.substring(0, i + 1);
		else
			r = s.substring(0, i);
		int fractLen = getFractionLength(s);
		int pad = fFractDigits - fractLen;
		if (pad > 0) {
			r= r.concat(s.substring(i + 1, i + 1 + fractLen));
			for (int j = 0; j < pad; j++)
				r= r.concat("0");
			return r;
		} else {
			return r.concat(s.substring(i + 1, i + 1 + fFractDigits));
		}
	} else
		return s;
}
public Double parse(String s) {
	if (s.endsWith("%"))
		s = s.substring(0, s.length() - 1);
	try {
		Double d = Double.valueOf(s);
		return d;
	} catch (NumberFormatException e) {
	}
	return null;
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	super.restoreState(conn, args);
	fFractDigits = args.get("fracts", 2);
}
}
