package com.ibm.ulc.ui;

import java.util.*;
import java.awt.*;
import java.beans.*;
import javax.swing.*;
import javax.swing.text.View;
import javax.swing.plaf.*;
import javax.swing.plaf.basic.BasicLabelUI;

/**
 * To add multiline label support to your swing applications, just add this
 * static call to your main method.  Note, you only need to do this once, even
 * if you change LookAndFeel as the UIManager knows not to overwrite the user
 * defaults.  Moreover, it uses the current L&F foreground/background colors
 * <p><pre>
 *        UiJMultiLineLabelUI.initialize();
 * </pre><p>
 * There are two alignment modes, BLOCK and INDIVIDUAL, which can be set with
 * the MULTI_ALIGNMENT_MODE property.  If not specified or set to null, the
 * default is INDIVIDUAL.
 * <p><pre>
 *     label.putClientProperty(MULTI_ALIGNMENT_MODE,BLOCK);
 *     label.putClientProperty(MULTI_ALIGNMENT_MODE,INDIVIDUAL);
 * </pre></p>
 * By default, text is clipped.  This can be changed using the CLIPPED_MODE
 * property.
 * <p><pre>
 *     label.putClientProperty(CLIPPED_MODE,CLIPPED);
 *     label.putClientProperty(CLIPPED_MODE,NOT_CLIPPED);
 * </pre></p>
 * You can also specify if the label should be drawn vertically, by using the
 * ORIENTATION_MODE property.
 * <p><pre>
 *     label.putClientProperty(ORIENTATION_MODE,HORIZONTAL);
 *     label.putClientProperty(ORIENTATION_MODE,VERTICAL);
 * </pre></p>
 * In vertical mode, MULTI_ALIGNMENT_MODE can be used for each column of text.
 * However, the CLIPPED_MODE property is ignored.
 */
public class UiJMultiLineLabelUI extends BasicLabelUI implements SwingConstants {
	public static String MULTI_ALIGNMENT_MODE = "multiAlignmentMode";
	public static String BLOCK = "block";
	public static String INDIVIDUAL = "individual";
	public static String CLIPPED_MODE = "clippedMode";
	public static String CLIPPED = "clipped";
	public static String NOT_CLIPPED = "notClipped";
	public static String ORIENTATION_MODE = "orientationMode";
	public static String HORIZONTAL = "horizontal";
	public static String VERTICAL = "vertical";
	public static String VERTICAL_SPACE = "verticalSpace";
	public static String VERTICAL_WIDTH = "verticalWidth";
	//
	protected static UiJMultiLineLabelUI SINGLETON = new UiJMultiLineLabelUI();
	//
	private static String DEFAULT_ALIGNMENT_MODE = INDIVIDUAL;
	private static String DEFAULT_CLIPPED_MODE = CLIPPED;
	private static String DEFAULT_VERTICAL_WIDTH = "W";
	private static String DEFAULT_VERTICAL_SPACE = " ";

	// don't make these final, since the value is different on each platform
	private static String LINE_SEPARATOR = System.getProperty("line.separator");
	private static int LINE_SEPARATOR_LEN = LINE_SEPARATOR.length();

	/* These are shared between instances in paint() below, as allocating
	 * Rectangles doubled the time that method took to run. 
	 */
	private Rectangle fIconR = new Rectangle();
	private Rectangle fTextR = new Rectangle();
	private Rectangle fViewR = new Rectangle();
	private static Insets fgViewInsets = new Insets(0, 0, 0, 0);

	/* These rectangles/insets are allocated once for this shared LabelUI
	 * implementation.  Re-using rectangles rather than allocating
	 * them in each paint call halved the time it took paint to run.
	 */
	private static Rectangle fgPaintIconR = new Rectangle();
	private static Rectangle fgPaintTextR = new Rectangle();
	private static Rectangle fgPaintViewR = new Rectangle();
	private static Insets fgPaintViewInsets = new Insets(0, 0, 0, 0);
	private static Rectangle fgOPaintIconR = new Rectangle();
	private static Rectangle fgOPaintTextR = new Rectangle();
/**
 * StringTokenizer(text,"\n") really does a "\n+" which is not what we want.
 * We also want it be based on the line.separator property.  So create our
 * own version.  We first attempt to divide by line.separator, then by "\n".
 * Ideally, we'd prefer to just break up by line.separator, but we need to
 * handle text that was defined in a properties file, with embedded \n.
 */
public static String[] breakupLines(String text) {
	int len = text.length();
	if (len == 0) {
		return new String[] {""};
	} else {
		Vector data = new Vector(10);
		int start = 0;
		int i = 0;
		while (i < len) {
			if (text.startsWith(LINE_SEPARATOR, i)) {
				data.addElement(text.substring(start, i));
				start = i + LINE_SEPARATOR_LEN;
				i = start;
			} else
				if (text.charAt(i) == '\n') {
					data.addElement(text.substring(start, i));
					start = i + 1;
					i = start;
				} else {
					i++;
				}
		}
		if (start != len) {
			data.addElement(text.substring(start));
		}
		int numlines = data.size();
		String lines[] = new String[numlines];
		data.copyInto(lines);
		return lines;
	}
}
public static ComponentUI createUI(JComponent c) {
	return SINGLETON;
}
/**
 *  copied and modified from BasicGraphicsUtils
 *
 *  Draw a string with the graphics g at location (x,y) just like
 *  g.drawString() would.  The first occurence of underlineChar in text will
 *  be underlined. The matching is not case sensitive.  Returns true if the
 *  underlined character was drawn.  
 */
public static boolean drawString(Graphics g, String text, int underlinedChar, int x, int y) {
	char lc, uc;
	int index = -1, lci, uci;
	if (underlinedChar != '\0') {
		uc = Character.toUpperCase((char) underlinedChar);
		lc = Character.toLowerCase((char) underlinedChar);
		uci = text.indexOf(uc);
		lci = text.indexOf(lc);
		if (uci == -1)
			index = lci;
		else
			if (lci == -1)
				index = uci;
			else
				index = (lci < uci) ? lci : uci;
	}
	g.drawString(text, x, y);
	if (index != -1) {
		FontMetrics fm = g.getFontMetrics();
		int underlineRectX = x + fm.stringWidth(text.substring(0, index));
		int underlineRectY = y;
		int underlineRectWidth = fm.charWidth(text.charAt(index));
		int underlineRectHeight = 1;
		g.fillRect(underlineRectX, underlineRectY + fm.getDescent() - 1, underlineRectWidth, underlineRectHeight);
	}
	return (index != -1);
}
public static String getClippedText(String text, FontMetrics fm, int availTextWidth) {
	String clipString = "...";
	int totalWidth = SwingUtilities.computeStringWidth(fm, clipString);
	int nChars;
	for (nChars = 0; nChars < text.length(); nChars++) {
		totalWidth += fm.charWidth(text.charAt(nChars));
		if (totalWidth > availTextWidth) {
			break;
		}
	}
	return text.substring(0, nChars) + clipString;
}
public Dimension getPreferredSize(JComponent c) {
	JLabel label = (JLabel) c;
	String text = label.getText();
	Icon icon = label.getIcon();
	Insets insets = label.getInsets(fgViewInsets);
	Font font = label.getFont();
	int dx = insets.left + insets.right;
	int dy = insets.top + insets.bottom;
	if ((icon == null) && ((text == null) || ((text != null) && (font == null)))) {
		return new Dimension(dx, dy);
	} else
		if ((text == null) || ((icon != null) && (font == null))) {
			return new Dimension(icon.getIconWidth() + dx, icon.getIconHeight() + dy);
		} else {
			//++
			FontMetrics fm = label.getFontMetrics(font);
			int fontHeight = fm.getHeight();
			String lines[] = breakupLines(text);
			int numLines = lines.length;
			String maxline = "";
			int maxwidth = 0;
			boolean clippedMode = isClippedMode(label);
			boolean vertical = isVerticalMode(label);
			for (int i = 0; i < numLines; i++) {
				int w = fm.stringWidth(lines[i]);
				if (w > maxwidth) {
					maxline = lines[i];
					maxwidth = w;
				}
			}
			//--
			fIconR.x = fIconR.y = fIconR.width = fIconR.height = 0;
			fTextR.x = fTextR.y = fTextR.width = fTextR.height = 0;
			fViewR.x = dx;
			fViewR.y = dy;
			fViewR.width = fViewR.height = Short.MAX_VALUE;

			//++
			if (vertical) {
				String totalWidth = getTotalVerticalWidth(label, numLines);
				clippedMode = false;
				layoutCL(label, fm, totalWidth, icon, fViewR, fIconR, fTextR, maxline.length() * fontHeight, clippedMode);
			} else {
				layoutCL(label, fm, maxline, icon, fViewR, fIconR, fTextR, numLines * fontHeight, clippedMode);
			}
			//--
			int x1 = Math.min(fIconR.x, fTextR.x);
			int x2 = Math.max(fIconR.x + fIconR.width, fTextR.x + fTextR.width);
			int y1 = Math.min(fIconR.y, fTextR.y);
			int y2 = Math.max(fIconR.y + fIconR.height, fTextR.y + fTextR.height);
			Dimension rv = new Dimension(x2 - x1, y2 - y1);
			rv.width += dx;
			rv.height += dy;
			return rv;
		}
}
protected String getTotalVerticalWidth(JLabel label, int numLines) {
	String space = getVerticalSpace(label);
	char[] spaceArr = space.toCharArray();
	int spaceLen = spaceArr.length;
	String width = getVerticalWidth(label);
	char[] widthArr = width.toCharArray();
	int widthLen = widthArr.length;
	char[] totalWidth = new char[spaceLen * numLines + widthLen * (numLines - 1)];
	for (int i = 0; i < numLines; i++) {
		System.arraycopy(widthArr, 0, totalWidth, i * (spaceLen + widthLen), widthLen);
		if (i != numLines - 1) {
			System.arraycopy(spaceArr, 0, totalWidth, i * (spaceLen + widthLen) + widthLen, spaceLen);
		}
	}
	return new String(totalWidth);
}
protected String getVerticalSpace(JLabel label) {
	String val = (String) label.getClientProperty(VERTICAL_SPACE);
	if (val == null) {
		return DEFAULT_VERTICAL_SPACE;
	} else {
		return val;
	}
}
protected String getVerticalWidth(JLabel label) {
	String val = (String) label.getClientProperty(VERTICAL_WIDTH);
	if (val == null) {
		return DEFAULT_VERTICAL_WIDTH;
	} else {
		return val;
	}
}
public static void initialize() {
	String key = "LabelUI";
	Class cls = SINGLETON.getClass();
	String name = cls.getName();
	UIManager.put(key, name);
	UIManager.put(name, cls);
}
protected boolean isClippedMode(JLabel label) {
	String val = (String) label.getClientProperty(CLIPPED_MODE);
	if (val == null) {
		return DEFAULT_CLIPPED_MODE.equals(CLIPPED);
	} else {
		return val.equals(CLIPPED);
	}
}
protected boolean isIndividualAlign(JLabel label) {
	String val = (String) label.getClientProperty(MULTI_ALIGNMENT_MODE);
	if (val == null) {
		return DEFAULT_ALIGNMENT_MODE.equals(INDIVIDUAL);
	} else {
		return val.equals(INDIVIDUAL);
	}
}
protected boolean isVerticalMode(JLabel label) {
	String val = (String) label.getClientProperty(ORIENTATION_MODE);
	if (val == null) {
		return false;
	} else {
		return val.equals(VERTICAL);
	}
}
/**
 * override BasicLableUI version, since we need to pass textHeight
 * forwards the call to layoutCompoundLabel().
 * This method is here to shorten the method call a little.
 */
protected String layoutCL(JLabel label, FontMetrics fontMetrics, String text, Icon icon, Rectangle viewR, Rectangle iconR, Rectangle textR, int textHeight, boolean clippedMode) {
	return layoutCompoundLabel(label, fontMetrics, text, icon, label.getVerticalAlignment(), label.getHorizontalAlignment(), label.getVerticalTextPosition(), label.getHorizontalTextPosition(), viewR, iconR, textR, label.getIconTextGap(), textHeight, clippedMode);
}
  /**
   * copied and modified from SwingUtilities, we define what the
   * textHeight is so that we can handle multiline text
   *
   * Compute and return the location of the icons origin, the
   * location of origin of the text baseline, and a possibly clipped
   * version of the compound labels string.  Locations are computed
   * relative to the viewR rectangle.
   */
  public static String layoutCompoundLabel(
	JComponent c,
	FontMetrics fm,
	String text,
	Icon icon,
	int verticalAlignment,
	int horizontalAlignment,
	int verticalTextPosition,
	int horizontalTextPosition,
	Rectangle viewR,
	Rectangle iconR,
	Rectangle textR,
	int textIconGap,
	//++ 
	// add two more parameters
	int textHeight,
	boolean clipIt
	//-- 
	)
  {
	/* Initialize the icon bounds rectangle iconR.
	 */

	if (icon != null) {
	  iconR.width = icon.getIconWidth();
	  iconR.height = icon.getIconHeight();
	}
	else {
	  iconR.width = iconR.height = 0;
	}

	/* Initialize the text bounds rectangle textR.  If a null
	 * or and empty String was specified we substitute "" here
	 * and use 0,0,0,0 for textR.
	 */

	boolean textIsEmpty = (text == null) || text.equals("");

	View v = null;
	if (textIsEmpty) {
	  textR.width = textR.height = 0;
	  text = "";
	}
	else {
	  v = (c != null) ? (View) c.getClientProperty("html") : null;
	  if (v != null) {
	textR.width = (int) v.getPreferredSpan(View.X_AXIS);
	textR.height = (int) v.getPreferredSpan(View.Y_AXIS);
	  } else {
	textR.width = SwingUtilities.computeStringWidth(fm,text);
	//++  
	// store the text height rather than calling fm.getHeight()
	textR.height = textHeight;
	//--  
	  }
	}

	/* Unless both text and icon are non-null, we effectively ignore
	 * the value of textIconGap.  The code that follows uses the
	 * value of gap instead of textIconGap.
	 */

	int gap = (textIsEmpty || (icon == null)) ? 0 : textIconGap;

	if (!textIsEmpty) {

	  /* If the label text string is too wide to fit within the available
	   * space "..." and as many characters as will fit will be
	   * displayed instead.
	   */

	  int availTextWidth;

	  if (horizontalTextPosition == CENTER) {
	availTextWidth = viewR.width;
	  }
	  else {
	availTextWidth = viewR.width - (iconR.width + gap);
	  }


	  //++ 
	  //clip the text (if asked and required)
	  if (clipIt && textR.width > availTextWidth) {
	if (v != null) {
	  textR.width = availTextWidth;
	} else {
	  text = getClippedText(text,fm,availTextWidth);
	  textR.width = SwingUtilities.computeStringWidth(fm,text);
	}
	  }
	  //-- 
	}

	/* Compute textR.x,y given the verticalTextPosition and
	 * horizontalTextPosition properties
	 */

	if (verticalTextPosition == TOP) {
	  if (horizontalTextPosition != CENTER) {
	textR.y = 0;
	  }
	  else {
	textR.y = -(textR.height + gap);
	  }
	}
	else if (verticalTextPosition == CENTER) {
	  textR.y = (iconR.height / 2) - (textR.height / 2);
	}
	else { // (verticalTextPosition == BOTTOM)
	  if (horizontalTextPosition != CENTER) {
	textR.y = iconR.height - textR.height;
	  }
	  else {
	textR.y = (iconR.height + gap);
	  }
	}

	if (horizontalTextPosition == LEFT) {
	  textR.x = -(textR.width + gap);
	}
	else if (horizontalTextPosition == CENTER) {
	  textR.x = (iconR.width / 2) - (textR.width / 2);
	}
	else { // (horizontalTextPosition == RIGHT)
	  textR.x = (iconR.width + gap);
	}

	/* labelR is the rectangle that contains iconR and textR.
	 * Move it to its proper position given the labelAlignment
	 * properties.
	 *
	 * To avoid actually allocating a Rectangle, Rectangle.union
	 * has been inlined below.
	 */
	int labelR_x = Math.min(iconR.x, textR.x);
	int labelR_width = Math.max(iconR.x + iconR.width,
				textR.x + textR.width) - labelR_x;
	int labelR_y = Math.min(iconR.y, textR.y);
	int labelR_height = Math.max(iconR.y + iconR.height,
				 textR.y + textR.height) - labelR_y;

	int dx, dy;

	if (verticalAlignment == TOP) {
	  dy = viewR.y - labelR_y;
	}
	else if (verticalAlignment == CENTER) {
	  dy = (viewR.y + (viewR.height / 2)) - (labelR_y + (labelR_height / 2));
	}
	else { // (verticalAlignment == BOTTOM)
	  dy = (viewR.y + viewR.height) - (labelR_y + labelR_height);
	}

	if (horizontalAlignment == LEFT) {
	  dx = viewR.x - labelR_x;
	}
	else if (horizontalAlignment == RIGHT) {
	  dx = (viewR.x + viewR.width) - (labelR_x + labelR_width);
	}
	else { // (horizontalAlignment == CENTER)
	  dx = (viewR.x + (viewR.width / 2)) -
	  (labelR_x + (labelR_width / 2));
	}

	/* Translate textR and glypyR by dx,dy.
	 */

	textR.x += dx;
	textR.y += dy;

	iconR.x += dx;
	iconR.y += dy;

	return text;
  }      
public void paint(Graphics g, JComponent c) {
	JLabel label = (JLabel) c;
	String text = label.getText();
	Icon icon = label.isEnabled() ? label.getIcon() : label.getDisabledIcon();
	if ((icon == null) && (text == null)) {
		return;
	}
	FontMetrics fm = g.getFontMetrics();
	int fontHeight = fm.getHeight();
	int fontAscent = fm.getAscent();
	int numLines = 1;
	int maxline = 0;
	String lines[];
	if (text != null) {
		int maxlength = 0;
		lines = breakupLines(text);
		numLines = lines.length;
		for (int i = 0; i < numLines; i++) {
			int len = SwingUtilities.computeStringWidth(fm, lines[i]);
			if (len > maxlength) {
				maxlength = len;
				maxline = i;
			}
		}
	} else {
		maxline = 0;
		lines = new String[1];
		lines[0] = null;
	}
	boolean clippedMode = isClippedMode(label);
	boolean vertical = isVerticalMode(label);
	boolean individualAlign = isIndividualAlign(label);
	fgPaintViewInsets = c.getInsets(fgPaintViewInsets);
	fgPaintViewR.x = fgPaintViewInsets.left;
	fgPaintViewR.y = fgPaintViewInsets.top;
	fgPaintViewR.width = c.getWidth() - (fgPaintViewInsets.left + fgPaintViewInsets.right);
	fgPaintViewR.height = c.getHeight() - (fgPaintViewInsets.top + fgPaintViewInsets.bottom);
	fgPaintIconR.x = fgPaintIconR.y = fgPaintIconR.width = fgPaintIconR.height = 0;
	fgPaintTextR.x = fgPaintTextR.y = fgPaintTextR.width = fgPaintTextR.height = 0;
	// now get layout, for the maximum line with the total text height
	if (vertical) {
		String totalWidth = getTotalVerticalWidth(label, numLines);
		clippedMode = false;
		layoutCL(label, fm, totalWidth, icon, fgPaintViewR, fgPaintIconR, fgPaintTextR, lines[maxline].length() * fontHeight, clippedMode);
	} else {
		layoutCL(label, fm, lines[maxline], icon, fgPaintViewR, fgPaintIconR, fgPaintTextR, fontHeight * numLines, clippedMode);
	}
	if (icon != null) {
		icon.paintIcon(c, g, fgPaintIconR.x, fgPaintIconR.y);
	}
	if (text != null) {
		int accChar = label.getDisplayedMnemonic();
		int textX = fgPaintTextR.x;
		int textY = fgPaintTextR.y + fontAscent;
		int textWidth = fgPaintTextR.width;
		View v = (View) c.getClientProperty("html");
		if (v != null) {
			// use View renderer
			v.paint(g, fgPaintTextR);
		} else
			if (vertical) {
				// vertical text
				String space = getVerticalSpace(label);
				String width = getVerticalWidth(label);
				int pixelSpace = fm.stringWidth(space);
				int pixelWidth = fm.stringWidth(width);
				int pixelColumnWidth = pixelSpace + pixelWidth;
				char[] oneChar = new char[1];
				int i, j, k, strlen, offset;
				String oneCharStr;
				String line;
				for (i = 0; i < numLines; i++, textX += pixelColumnWidth) {
					line = lines[i];
					strlen = line.length();
					if (individualAlign) {
						fgOPaintIconR.x = fgOPaintIconR.y = fgOPaintIconR.width = fgOPaintIconR.height = 0;
						fgOPaintTextR.x = fgOPaintTextR.y = fgOPaintTextR.width = fgOPaintTextR.height = 0;
						/* relayout each line. Note, we're only concerned about y location.
						     * so it doesn't matter what text we use to do the layout
						     */
						line = layoutCL(label, fm, space, icon, fgPaintViewR, fgOPaintIconR, fgOPaintTextR, strlen * fontHeight, clippedMode);
						textY = fgOPaintTextR.y + fontAscent;
					}
					for (j = 0, k = textY; j < strlen; j++, k += fontHeight) {
						oneChar[0] = lines[i].charAt(j);
						oneCharStr = new String(oneChar);
						// now center the character in the column
						offset = (pixelWidth - fm.stringWidth(oneCharStr)) / 2;
						if (label.isEnabled()) {
							if (paintEnabledText(label, g, oneCharStr, textX + offset, k, accChar)) {
								accChar = '\0';
							}
						} else {
							if (paintDisabledText(label, g, oneCharStr, textX + offset, k, accChar)) {
								accChar = '\0';
							}
						}
					}
				}
			} else {
				// horizontal text
				int textHeight = fontHeight * numLines;
				for (int i = 0; i < numLines; i++, textY += fontHeight) {
					String clipped = lines[i];
					if (individualAlign) {
						fgOPaintIconR.x = fgOPaintIconR.y = fgOPaintIconR.width = fgOPaintIconR.height = 0;
						fgOPaintTextR.x = fgOPaintTextR.y = fgOPaintTextR.width = fgOPaintTextR.height = 0;
						// now get layout, for the maximum line with the total text height
						clipped = layoutCL(label, fm, clipped, icon, fgPaintViewR, fgOPaintIconR, fgOPaintTextR, textHeight, clippedMode);
						textX = fgOPaintTextR.x;
					} else
						if (clippedMode) {
							/* block mode, now clip the text, if necessary */
							int w = SwingUtilities.computeStringWidth(fm, clipped);
							if (w > textWidth) {
								clipped = getClippedText(clipped, fm, textWidth);
							}
						}
					if (label.isEnabled()) {
						if (paintEnabledText(label, g, clipped, textX, textY, accChar)) {
							accChar = '\0';
						}
					} else {
						if (paintDisabledText(label, g, clipped, textX, textY, accChar)) {
							accChar = '\0';
						}
					}
				}
			}
	}
}
/**
 * Paint clippedText at textX, textY with background.lighter() and then
 * shifted down and to the right by one pixel with background.darker().
 * Return true if the accelerator Character was printed
 *
 * @see #paint
 * @see #paintEnabledText
 */
protected boolean paintDisabledText(JLabel l, Graphics g, String s, int textX, int textY, int accChar) {
	Color background = l.getBackground();
	g.setColor(background.brighter());
	drawString(g, s, accChar, textX, textY);
	g.setColor(background.darker());
	return drawString(g, s, accChar, textX + 1, textY + 1);
}
/**
 * Paint clippedText at textX, textY with the labels foreground color.
 * Return true if the accelerator Character was printed
 *
 * @see #paint
 * @see #paintDisabledText
 */
protected boolean paintEnabledText(JLabel l, Graphics g, String s, int textX, int textY, int accChar) {
	g.setColor(l.getForeground());
	return drawString(g, s, accChar, textX, textY);
}
  public void propertyChange(PropertyChangeEvent e) {
	if (e.getPropertyName().equals(MULTI_ALIGNMENT_MODE) ||
	e.getPropertyName().equals(CLIPPED_MODE) ||
	e.getPropertyName().equals(VERTICAL_SPACE) ||
	e.getPropertyName().equals(VERTICAL_WIDTH)) {
	  JLabel label = (JLabel) e.getSource();
	  label.repaint();
	} else {
	  super.propertyChange(e);
	}
  }    
/**
 * Change the default alignment mode
 */
public static void setDefaultAlignmentMode(String mode) {
	DEFAULT_ALIGNMENT_MODE = mode;
}
/**
 * Change the default clipped mode
 */
public static void setDefaultClippedMode(String mode) {
	DEFAULT_CLIPPED_MODE = mode;
}
/**
 * Change the default vertical space.  Ideally, we would like to specify in
 * pixel sizes the space for each column of text.  That would require more
 * changes to SwingUtilities.layoutCompoundLabel than we'd like.  Instead,
 * we specify the space with a string template.
 */
public static void setDefaultVerticalSpace(String space) {
	DEFAULT_VERTICAL_SPACE = space;
}
/**
 * Change the default vertical width
 */
public static void setDefaultVerticalWidth(String width) {
	DEFAULT_VERTICAL_WIDTH = width;
}
}
