package com.ibm.ulc.ui;

import com.ibm.ulc.util.UlcObject;
import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.print.*;

/**
 * This class wraps a <code>BufferedImage</code> for being
 * printed on a single page. If it doesn't fit on the page
 * the image is scaled.
 *
 * @since:	R3.1
 */
public class UiBufferedImagePrinter extends UlcObject implements Printable {

	/**
	 * Holds the buffered image that should be printed
	 * @ since	R3.1
	 * @ serial
	 */
	private BufferedImage fImage= null;
/**
 * Create a new UiImagePrinter for the
 * given <code>image</code>
 *
 * @param	image	java.awt.Image
 * @since	R3.1
 */
public UiBufferedImagePrinter(BufferedImage image) {
	fImage= image;
}
/**
 * Print method used to print the image
 * This method will be called by the Java
 * print API several times for one
 * page (the frame is scanned).
 *
 * @since	R3.1
 */
public int print(Graphics g, PageFormat pf, int pi) throws PrinterException {
	if (pi > 0)
		return Printable.NO_SUCH_PAGE;

	Graphics2D g2 = (Graphics2D) g;

	// Shift defined page border
 	g2.translate(pf.getImageableX(), pf.getImageableY());

 	// Scale it to fit to page
 	double scaleFactor= 1.0d;
 	if (pf.getImageableWidth() < fImage.getWidth())
 		scaleFactor= pf.getImageableWidth() / fImage.getWidth();
 	if (pf.getImageableHeight() < fImage.getHeight()) {
	 	double yScaleFactor= pf.getImageableHeight() / fImage.getHeight();
	 	if (yScaleFactor < scaleFactor)
	 		scaleFactor= yScaleFactor;
 	}
 	g2.scale(scaleFactor, scaleFactor);
 	
	g2.drawImage(fImage, 0, 0, null);
	return Printable.PAGE_EXISTS;
}
/**
 * Print this shell on a printer.
 * 
 * @param	showPrintDialog		Show dialog to select printer and other print properties
 * @param	showPageDialog		Show dialog to select page properties
 * @since	R3.1
 */
public void print(boolean showPrintDialog, boolean showPageDialog) {
	PrinterJob pjob = PrinterJob.getPrinterJob();
	pjob.setJobName("ULC PrintScreen");
	PageFormat dPf= pjob.defaultPage();
	PageFormat pf= dPf;	
	if (showPageDialog) {
		pf= pjob.pageDialog(dPf);
		if (pf == dPf) // user pressed [cancel]
			return;
	}
	pjob.setPrintable(this, pf);
	if (!showPrintDialog || pjob.printDialog())
		try {
			pjob.print();
		} catch (Exception e) {
			troubleErr("printScreen(boolean, boolean)", "unexpected exception");
			e.printStackTrace(System.err);
			pjob.cancel();
		} catch (OutOfMemoryError e) {
			troubleErr("printScreen(boolean, boolean)", "not enough memory");
			e.printStackTrace(System.err);
			pjob.cancel();
		}
}
}
