package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997-1999 Object Technology International Inc.
 */
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.base.IDefaults;
import java.util.*;
/**
 * The UIProxy class is the root of the hierarchy for all UI proxy widgets.
 * This class is an Abstract class and implements the infrastructure for
 * ULC and UI widgets to communicate.
 *
 */
abstract public class UIProxy extends UlcObject implements ICallable, IDefaults {
	/**
	 * Every proxy widget knows its active UlcConnection
	 */
	protected ORBConnection fConnection = null;
	/**
	 * The unique identifier within the connection for this widget.
	 * The id is allocated by the ULCProxy.
	 */
	protected int fId = 0;
	static private int fgInstanceCount = 0;
	protected Vector fList = new Vector();
	protected UlcHashtable fOptionalEvents = null;
	protected UIProxy fProxyParent;
public UIProxy() {
	fgInstanceCount++;
}
/**
 * Generic method to add a child widget to this widget.
 * Subclasses will typically override the <code>add(Managed)</code> method
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> optional arguments 
 */
public void add(ORBConnection conn, Anything args) {
	UIProxy m = getManaged(UIProxy.class, conn, args);
	if (m != null) {
		add(m);
		internalAddToChildren(m);
	}
}
/**
 * Generic method to add a UIProxy widget to this widget.
 * Subclasses that support child widgets must override this method
 *
 * @param managed 		the UI proxy widget that is being added as a child
 */
public void add(UIProxy managed) {
	trouble("add(UIProxy)", "abstract method");
}
protected static synchronized void addInstance() {
	fgInstanceCount++;
	Runtime.getRuntime().gc();
	System.out.println("+Managed: " + fgInstanceCount);
}
/**
 * Generic method to add a collection of child widgets to this widget.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param list		the <code>Anything</code> containing the list of children
 */
public void addMany(ORBConnection conn, Anything list) {
	if (list != null)
		for (int i = 0; i < list.size(); i++)
			add(conn, list.get(i));
}
public UIResourceCacheAbstract cachedResources() {
	UI ui = getUI();
	if (ui != null)
		return ui.getResourceCache();
	else
		return null;
}
/**
 * Converts an Anything into an Object.
 * Typically called when sending data from ULC to UI.
 * If a <code>IProxy</code> object is sent it must be sent using the following rules.
 * 1) The Anything type must be a vector type.
 * 2) The key 'oid' must contain the identifier for the IProxy
 * 3) The key 'type' will be sent ONLY when the IProxy must be created as well.
 * 4) If 'oid' does not exist then the object is not a IProxy and the object is created using
 *  the Common#convertFromAnything method.
 *
 */
public Object convert(ORBConnection conn, Anything anything) {
	if (anything.getType() == Anything.VECTOR) {
		if (anything.get("oid", null) != null) {
			if (anything.get("type", null) != null) {
				ICallable ic = find(conn, anything, "Managed.convert");
				if (ic != null)
					return ic;
			} else {
				if (anything.size() == 1) {
					ICallable ic = find(conn, anything.get(0), "Managed.convert");
					if (ic != null)
						return ic;
				}
			}
		}
	}
	Object o = Common.convertFromAnything(anything);
	return o;
}
/**
 * Converts an Object into an Anything.
 * Typically called from setValueAt when sending data from UI to ULC.
 */
public Anything convert(Object object) {
	Anything a = Common.convertToAnything(object);
	if (a != null)
		return a;
	trouble("convert(Object o)", "cannot convert");
	return new Anything(object.toString());
}
protected void enableOptionalEvents(Anything optionalEvents) {
	if (optionalEvents == null) 
		return;
	if (fOptionalEvents == null) 
		fOptionalEvents = new UlcHashtable();
	for (int i = 0; i < optionalEvents.size(); i++) 
		fOptionalEvents.put(optionalEvents.get(i).asString(null), "1");
}
public void finalize() throws Throwable {
	fgInstanceCount--;
	super.finalize();
}
/**
 * Convenience method to find an object that is wrapped by an Anything.
 *
 * @param arg 	the <code>Anything</code> storing the object id
 * @param where	the <code>String</code> method location of the caller
 */
public static ICallable find(ORBConnection conn, Anything arg, String where) {
	ICallable c = null;
	if (arg != null) {
		if (arg.getType() == Anything.LONG) { // seems to be an ID
			int oid = arg.asInt(0);
			if (conn == null) {
				trouble2("UIProxy.find", " connection is null");
				return null;
			}
			if (oid > 0)
				c = conn.find(oid);
			if (c == null && where != null) {
				trouble2("UIProxy.find", where + " couldn't find object " + oid);
			}
		} else {
			// seems to be a full object description; try to create it
			UI ui = (UI) conn.find(Common.FACTORY_ID);
			if (ui != null)
				c = ui.newInstance(conn, arg);
		}
	}
	return c;
}
/**
 * hook for disposing stuff.
 * All subclasses that need to perform some cleanUp when they are disposed 
 * should override this method.
 */
public void free() {
	for (int i = 0; i < getChildren().size(); i++) {
		UIProxy p = (UIProxy) getChildren().elementAt(i);
		p.free();
		p.setProxyParent(null);
	}
	getChildren().removeAllElements();
	if (fConnection != null) {
		fConnection.unregister(fId);
		fConnection = null;
	}
}
public UIApplication getApplication() {
	if (getProxyParent() == null) {
		Registry r = fConnection.getRegistry();
		Hashtable ht = r.getRegistry();
		Enumeration e = ht.elements();
		while (e.hasMoreElements()) {
			ICallable c = (ICallable) e.nextElement();
			if (c instanceof UIApplication) {
				UIApplication a = (UIApplication) c;
				if (a.isChildWidget(this))
					return a;
			}
		}
		return null;
	}
	else
		return getProxyParent().getApplication();
}
/**
 * Return all child widgets of this widget.
 *
 */
protected Vector getChildren() {
	return fList;
}
public ORBConnection getConnection() {
	return fConnection;
}
public int getId() {
	return fId;
}
/**
 * Return an instance of the object based on the information in the arguments.
 * The returned instance will be registered within the UlcConnection registry.
 *
 * @param uiClass 	the <code>Class</code> of the object to instantiate
 * @param conn 		the <code>UlcConnection</code> in which the Class should be instantiated
 * @param args		the <code>Anything</code> arguments that contain information of the uiClass
 */
public static UIProxy getManaged(Class uiClass, ORBConnection conn, Anything args) {
	return getManaged(uiClass, conn, args, true);
}
/**
 * Return an instance of the object based on the information in the arguments.
 * The returned instance will be registered within the UlcConnection registry.
 *
 * @param uiClass 		the <code>Class</code> of the object to instantiate
 * @param conn 			the <code>UlcConnection</code> in which the Class should be instantiated
 * @param args			the <code>Anything</code> arguments that contain information of the uiClass
 * @param reportErrors	If true writes error messages on instantiation failure
 */
public static UIProxy getManaged(Class uiClass, ORBConnection conn, Anything args, boolean reportErrors) {
	UIProxy m = null;
	if (uiClass == null) {
		System.out.println("UIProxy.find: Class is null");
		return null;
	}
	if (conn == null) {
		System.out.println("UIProxy.find: conn is null");
		return null;
	}
	if (args != null) {
		if (args.get("type", null) != null) {
			m = (UIProxy) find(conn, args, "UIProxy.getManaged");
			if (uiClass.isInstance(m))
				return m;
		} else {
			if (args.size() == 1) {
				m = (UIProxy) find(conn, args.get(0), "UIProxy.getManaged");
				if (uiClass.isInstance(m))
					return m;
			}
		}
		String s = ((m == null) ? "null" : m.toString());
		if (reportErrors)
			trouble2("getManaged", "object [" + s + "] is not an instanceof " + uiClass.getName());
	}
	return null;
}
/**
 * Insert the method's description here.
 * Creation date: (10/15/99 10:58:27 AM)
 * @return com.ibm.ulc.ui.UIProxy
 */
public UIProxy getProxyParent() {
	return fProxyParent;
}
public UI getUI() {
	if (fConnection == null)
		return null;
	UI ui = (UI) fConnection.find(Common.FACTORY_ID);
	return ui;
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request write a message out on the console.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("add")) {
		add(conn, args);
		return;
	}
	if (request.equals("remove")) {
		remove(conn, args);
		return;
	}
	if (request.equals("synch")) {
		sendULC(request, args);
		return;
	}
	if (request.equals("addMany")) {
		addMany(conn, args);
		return;
	}
	if (request.equals("removeMany")) {
		removeMany(conn, args);
		return;
	}
	if (request.equals("release")) {
		release();
		return;
	}
	if (request.equals("setOptionalEvents")) {
		setOptionalEvents(args);
		return;
	}
	System.out.println(this.getClass().toString() + "  unknown request: " + request);
}
/**
 * Keep track of the children added to this widget.
 *
 */
protected void internalAddToChildren(UIProxy proxy) {
	if (!getChildren().contains(proxy)) {
		proxy.setProxyParent(this);
		getChildren().addElement(proxy);
	}
}
/**
 * Keep track of the children added to this widget.
 *
 */
protected void internalRemoveFromChildren(UIProxy proxy) {
	if (getChildren().contains(proxy)) {
		proxy.setProxyParent(null);
		getChildren().removeElement(proxy);
	}
}
/**
 * Answer true if the receiver is a child of the specified object.

 */
public boolean isChildOf(UIProxy parent) {
	return (getProxyParent() == parent);
}
/**
 * Return true if the proxy is a child widget of this widget or any of its children.
 *
 */
protected boolean isChildWidget(UIProxy proxy) {
	boolean found = false;
	for (int i = 0;(i < getChildren().size() && !found); i++) {
		UIProxy p = (UIProxy) getChildren().elementAt(i);
		if (proxy == p)
			return true;
		found = p.isChildWidget(proxy);
	}
	return found;
}
/**
 * Answer boolean whether the receiver should send
 * the given event type across to the application side.
 * This API is provided for traffic-optimization, so that only
 * events which have some listeners on the application side, 
 * need be sent.
 *
 * @see ULCProxy.enableOptionalEvent(key)
 */
public boolean isOptionalEventEnabled(String type) {
	if (fOptionalEvents == null)
		return false;
	return fOptionalEvents.containsKey(type);
}
/**
 * Answer true if the receiver is a child of the specified object.

 */
public boolean isParentOf(UIProxy child) {
	return (getChildren().contains(child));
}
protected CachedResourceUpdate newUpdate(Object component, String methodName, Object userData) {
	if (component != null) {
		return new CachedResourceUpdate(component, methodName, userData);
	}
	else {
		return null;
	}
}
public static void printInstances(String msg) {
	System.out.println("Managed instances at " + msg + ": " + fgInstanceCount);
}
protected void release() {
	free();
	removeFromParent();
}
/**
 * Generic method to remove a child widget from this widget.
 * Subclasses will typically override the <code>remove(Managed)</code> method
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> optional arguments 
 */
public void remove(ORBConnection conn, Anything args) {
	UIProxy m = getManaged(UIProxy.class, conn, args, false);
	if (m != null) {
		remove(m);
		internalRemoveFromChildren(m);
	}
}
/**
 * Generic method to remove a UI proxy child widget from this widget.
 * Subclasses that support child widgets must override this method
 *
 * @param uiproxy 		the UI proxy widget that is being removed 
 */
public void remove(UIProxy uiproxy) {
	trouble("remove(UIProxy)", "abstract method");
}
/**
 * Generic method to remove a child widget from his parent widget.
 */
protected void removeFromParent() {
	UIProxy parent = getProxyParent();
	if (parent != null) {
		parent.remove(this);
		parent.internalRemoveFromChildren(this);
	}
}
protected static synchronized void removeInstance() {
	fgInstanceCount--;
}
/**
 * Generic method to remove a collection of child widgets from this widget.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param list		the <code>Anything</code> containing the list of children
 */
public void removeMany(ORBConnection conn, Anything list) {
	if (list != null)
		for (int i = 0; i < list.size(); i++)
			remove(conn, list.get(i));
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	if (args.isDefined("list"))
		addMany(conn, args.get("list"));
	enableOptionalEvents(args.get("oe"));
}
/**
 * Send an event to this widgets ULC proxy.
 *
 * @param type		The identifier for this event
 */
public void sendEventULC(String type) {
	Anything a = new Anything();
	a.put("type", type);
	sendULC("event", a);
}
/**
 * Send an event to this widgets ULC proxy.
 *
 * @param type		The identifier for this event
 * @param anything	The <code>Anything</code> that contains the arguments for this event
 */
public void sendEventULC(String type, Anything anything) {
	anything.put("type", type);
	sendULC("event", anything);
}
/**
 * Send an event to this widgets ULC proxy.
 *
 * @param type	The identifier for this event
 * @param key	The attribute that has changed
 * @param value	The value of the changed attribute
 */
public void sendEventULC(String type, String key, Anything value) {
	Anything a = new Anything();
	a.put("type", type);
	a.put(key, value);
	sendULC("event", a);
}
/**
 * Send an optional event to this widgets ULC proxy. This event will be sent
 * only if the ULC Proxy has explicitly requested this event if not the send 
 * request is ignored.
 *
 * @param type	The identifier for this event
 */
public void sendOptionalEventULC(String type) {
	if (isOptionalEventEnabled(type))
		sendEventULC(type);
}
/**
 * Send an optional event to this widgets ULC proxy. This event will be sent
 * only if the ULC Proxy has explicitly requested this event if not the send 
 * request is ignored.
 *
 * @param type		The identifier for this event
 * @param anything	The <code>Anything</code> that contains the key/value that has changed
 */
public void sendOptionalEventULC(String type, Anything anything) {
	if (isOptionalEventEnabled(type))
		sendEventULC(type, anything);
}
/**
 * Send an optional event to this widgets ULC proxy. This event will be sent
 * only if the ULC Proxy has explicitly requested this event if not the send 
 * request is ignored.
 *
 * @param type	The identifier for this event
 * @param key	The attribute that has changed
 * @param value	The value of the changed attribute
 */
public void sendOptionalEventULC(String type, String key, Anything value) {
	if (isOptionalEventEnabled(type))
		sendEventULC(type, key, value);
}
/**
 * Send a request to this widgets ULC proxy.
 *
 * @param request	The identifier for this request
 */
public void sendULC(String request) {
	sendULC(request, new Anything());
}
/**
 * Send a request to this widgets ULC proxy.
 *
 * @param request	The identifier for this request
 * @param args		The <code>Anything</code> containing the arguments for this request
 */
public void sendULC(String request, Anything args) {
	if (fConnection != null)
		fConnection.send(fId, request, args);
}
public void setConnectionAndId(ORBConnection connection, Anything args) {
	fConnection = connection;
	fId = args.get("oid", 0);
	fConnection.register(fId, this);
}
protected void setOptionalEvents(Anything optionalEvents) {
	if (optionalEvents == null)
		return;
	fOptionalEvents = new UlcHashtable();
	for (int i = 0; i < optionalEvents.size(); i++)
		fOptionalEvents.put(optionalEvents.get(i).asString(null), "1");
}
public void setOptionalEventsEnabled(boolean enable) {
}
/**
 * Insert the method's description here.
 * Creation date: (10/15/99 10:58:27 AM)
 * @param newProxyParent com.ibm.ulc.ui.UIProxy
 */
public void setProxyParent(UIProxy newProxyParent) {
	fProxyParent = newProxyParent;
}
/**
 * Convenience method to write an error message to the current error log
 *
 * @param where		The <code>String</code> identifier of the calling method
 * @param what		The <code>String</code> description of the error
 */
public void trouble(String where, String what) {
	String className = getClass().getName();
	System.out.println("Error in " + className + "." + where + "(" + fId + "): " + what);
}
public void updateCachedResource(String methodName, ICachedResource resource, Object userData) {
	trouble("update Cached Resource", "unknown resource" + methodName);
}
}
