package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997-2000 Object Technology International Inc.
 */
import java.util.*;
import java.io.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;

/**
 * UI proxy for ULCProgramStarter
 *
 * @since	R3.1
 */
public class UIProgramStarter extends UIProxy {
/**
 * Execute the specified <code>commandLine</code> in a separate process with the 
 * specified environment variables. The arguments are directly passed to exec.
 *
 * @param		commandLine	a commandLine (can also be only a file).
 * @param		environment	array of strings, each element of which 
 *							has environment variable settings in format
 *							<i>name</i>=<i>value</i>.
 * @exception	SecurityException	if not enough permission
 * @see			Runtime#exec(String, String[])
 * @since		R3.1
 */
protected void exec(String command, Vector environment) {
	if (command == null)
		return;
	String [] envStrings= null;
	if (environment != null) {
		envStrings= new String[environment.size()];
		int i= 0;
		for (Enumeration enum= environment.elements(); enum.hasMoreElements();) {
			envStrings[i]= (String) enum.nextElement();
			i++;
		}
	}
	try {
		Runtime.getRuntime().exec(command, envStrings);
	}
	catch (Exception e) {
		troubleErr("exec(String, Vector)", "Exception executing: " + command);
		if (isOptionalEventEnabled("error")) {
			Anything args= new Anything();
			args.put("method", "exec");			
			args.put("cmdLine", command);
			args.put("error", e.toString());
			sendEventULC("error", args);
		}
	}
}
protected String getExecEnvironment() {
	String os = System.getProperty("os.name").toUpperCase();
	String execEnv= null;
	try {
		execEnv= System.getProperty("ulc.programStarter.environment");
	} catch (SecurityException e) {
		trouble("getExecEnvironment()", "SecurityException getting property: ulc.programStarter.environment");};

	if (execEnv == null) {
		if (os.startsWith("WINDOWS")) {
			if (os.endsWith("2000") || os.endsWith("NT"))
				execEnv= "cmd /c start " + '"' + '"';
			// Other Windows OSs eg 95, 97, 98
			else
				execEnv= "start";
		}
		else if (os.endsWith("OS2"))
			execEnv= "start";
		else
			execEnv= "";
	}

	// Ensure trailing blank
	if (execEnv.equals("") || execEnv.endsWith(" "))
		return execEnv;
	else
		return execEnv + ' ';
}
/**
 * Return the command line patched for specific OS
 *
 * @param   cmdLine   an absolute URL giving the location of the document.
 * @since	R3.1
 */
protected String getOSSpecificCmdLine(String cmdLine) {
	String os= System.getProperty("os.name").toUpperCase();
	if (os.startsWith("WINDOWS"))
		// Quote url for Windows OSs
		return '"' + cmdLine + '"';
	else
		return cmdLine;
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request write a message out on the console.
 *
 * @param	conn		ORBConnection	The connection on which the reply should be sent.
 * @param	request 	String			The string that identifies this request.
 * @param	args		Anything		The arguments associated with this request.
 * @since	R3.1
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	String nullString= null;

	if (request.equals("exec") && args.isDefined("cmdLine")) {
		if (args.isDefined("env"))
			exec(args.get("cmdLine", nullString), args.get("env").toCollection());
		else
			exec(args.get("cmdLine", nullString), null);
		return;
	}
	if (request.equals("start") && args.isDefined("cmdLine")) {
		if (args.isDefined("env"))
			start(args.get("cmdLine", nullString), args.get("env").toCollection());
		else
			start(args.get("cmdLine", nullString), null);
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Execute the specified <code>commandLine</code> in a separate process with the 
 * specified environment variables. If supported by the OS, the command line can
 * also be a single file. In this case it is up to the OS to find the program
 * (eg Windows uses file extension).
 *
 * @param		commandLine	a commandLine (can also be only a file).
 * @param		environment	array of strings, each element of which 
 *							has environment variable settings in format
 *							<i>name</i>=<i>value</i>.
 * @exception	SecurityException	if not enough permission
 * @see			Runtime#exec(String, String[])
 * @since		R3.1
 */
protected void start(String commandLine, Vector environment) {
	if (commandLine == null)
		return;
	String [] envStrings= null;
	if (environment != null) {
		envStrings= new String[environment.size()];
		int i= 0;
		for (Enumeration enum= environment.elements(); enum.hasMoreElements();) {
			envStrings[i]= (String) enum.nextElement();
			i++;
		}
	}
	try {
		Runtime.getRuntime().exec(getExecEnvironment() + getOSSpecificCmdLine(commandLine), envStrings);
	}
	catch (Exception e) {
		troubleErr("start(String, Vector)", "Exception starting: " + commandLine);
		if (isOptionalEventEnabled("error")) {		
			Anything args = new Anything();
			args.put("method", "start");
			args.put("cmdLine", commandLine);
			args.put("error", e.toString());
			sendOptionalEventULC("error", args);
		}
	}
}
}
