package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.ui.base.*;

/**
 * This is the abstract superclass for form-based widgets. All such widgets
 * can be associated with a FormModel :
 * @see UIFormModel
 * As a result, widgets in this hierarchy can get and set their data on the
 * form model, which in turn communicates with the application in a manner
 * so as to minimize traffic.
 */
abstract public class UIFormComponent extends UIComponent implements IFormModelListener {
	protected IFormModel fFormModel = null;
	protected String fKey = null;
	protected int fNotificationPolicy = FORM_NOTIFICATION_ON_FOCUS_CHANGE;
/**
 * The application may set my value to be an invalid value for the configured Validator.
 * Detect this condition and if so update the application with the new value.
 */
public void checkForInvalidFormValueUpdate() {
}
/**
 * Send the data contained in the receiver, to the
 * application side. This would happen via the receiver's
 * form model.
 */
protected void flush() {
	trouble("flush", "abstract method");
}
/**
 * The form model is about to send its contents to the ULC Server.
 * Widgets that may have unsaved changes must save their changes to the formModel now.
 * This method will be triggered only when a saveInput is requested explicitly from the ULC Server.
 */
protected void formModelAboutToSaveInput() {
}
/**
 * The receiver is being destroyed. Release all the 
 * associated resources.
 */
public void free() {
	if (fFormModel != null) {
		fFormModel.removeModelListener(this);
		fFormModel = null;
	}
	fKey = null;
	super.free();
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("flush")) {
		flush();
		return;
	}
	if (request.equals("setNotificationPolicy")) {
		fNotificationPolicy = args.asInt(FORM_NOTIFICATION_ON_FOCUS_CHANGE);
		return;
	}
	if (request.equals("setFormModel")) {
		setFormModel(conn, args);
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Notify the receiver that it's form model has
 * changed.
 * @param type	: Type of change in the UIFormModel. Values possible :
 *					FORM_MODEL_UNSPECIFIED 
 *					FORM_MODEL_INVALID 
 *					FORM_MODEL_CHANGED 
 *					FORM_MODEL_ABOUT_TO_SAVE
 * @param key 	: The id (slot) which has changed it's contents.
 */
public void notify(int type, String key) {
	if (fFormModel != null) { // we only have to update if we have a formmodel
		if (type == FORM_MODEL_ABOUT_TO_SAVE) {
			formModelAboutToSaveInput();
			return;
		}
		if (key == null || key.equals(fKey)) {
			resetPreviousValue(type, key);
			updateFromModel();
		}
	}
}
/**
 * The FormComponent has been asked to throw away its changes.
 * Discard any previous values that this component may be keeping.
 */
protected void resetPreviousValue(int type, String key) {
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	fNotificationPolicy = args.get("notificationPolicy", FORM_NOTIFICATION_ON_FOCUS_CHANGE);
	fKey = args.get("key", "key?");
	super.restoreState(conn, args);
	setFormModel(conn, args.get("formModel"));
}
/**
 * Set the FormModel for the receiver from the given
 * arguments.
 */
public void setFormModel(ORBConnection conn, Anything args) {
	fFormModel = (IFormModel) getManaged(IFormModel.class, conn, args);
	if (fFormModel != null) {
		fFormModel.addModelListener(fKey, this);
		updateFromModel();
	}
}
/**
 * Update the data in the receiver, from the
 * values it it's form model.
 */
abstract protected void updateFromModel();
}
