package com.ibm.ulc.ui;

/*
 * Copyright (c) 1997 Object Technology International Inc.
 *
 * This class is used for displaying and entering text in the UIEngine.
 * Validators can be associated with this widget, to perform input-checks 
 * in the UI without going to the application.
 * Also, the UIField can be associated with a formModel, in which
 * case it gets and sets it's values via this form model (rather than
 * directly from the application side).
 */
import java.util.*;
import java.awt.*;
import java.awt.event.*;
import javax.swing.*;
import javax.swing.text.*;
import javax.swing.event.*;
import javax.swing.border.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.ui.base.*;
import com.ibm.ulc.ui.dataTypes.*;
public class UIField extends UIFormComponent implements DocumentListener, FocusListener, IEnableListener, ActionListener {
	protected JTextComponent fTextComponent = null;
	protected JScrollPane fScrollPane = null;
	protected int fCols = 10, fRows = 1;
	protected Object fValue = null;
	protected String fLastInput = null;
	protected Vector fEmptyStateListeners = null;
	protected boolean fInFocus = false;
	protected IDataType fDataType = null;
	protected boolean fEditable;
	protected UITrigger fTrigger;
	protected Color	fReadOnlyBackgroundColor;
	protected Color	fEditableBackgroundColor;
	protected boolean fLastEnableState= false;
	protected boolean fEnableSentToListeners= false;
	protected boolean fFirstTime= true;
/**
 * Notification that an action has been performed, i.e. the 
 * 'Enter' key has been pressed on this field.
 * 
 * @param event : The notification event.
 */
public void actionPerformed(ActionEvent event) {
	updateToModel(FORM_NOTIFICATION_ON_FOCUS_CHANGE);
	if (isOptionalEventEnabled("action")) {
		sendOptionalEventULC("action");
		if (fTrigger != null)
			fTrigger.trigger(fConnection, TRIGGER_ON_ACTION, this, null);
	} else {
		// If not explicitly handled by the application, 
		// press the default button
		AbstractButton defaultButton = ((JComponent) getComponent()).getRootPane().getDefaultButton();
		if (defaultButton instanceof AbstractButton) {
			defaultButton.doClick();
		}
	}
}
/**
 * Add the given component as listener, who will be informed
 * when the state of the receiver changes from enabled to disabled,
 * or vice-versa.
 *
 * @param component : The Component which will act as listener.
 */
public void addEnableListener(IEnableListenerTarget component) {
	Assert.isNotNull(component);
	if (fEmptyStateListeners == null)
		fEmptyStateListeners = new Vector();
	fEmptyStateListeners.addElement(component);
	component.setEnabled(shouldEnableListener());
}
/**
 * Do the necessary updates when the receiver document
 * has changed.
 *
 * @param event : DocumentEvent describing the change.
 */
public void changedUpdate(DocumentEvent event) {
}
/**
 * The application may set my value to be an invalid value for the configured Validator.
 * Detect this condition and if so update the application with the new value.
 */
public void checkForInvalidFormValueUpdate() {
	if (fDataType == null)
		return;
	Object value = getValue();
	if ((value instanceof PendingCell) || (value instanceof InvalidCell))
		return;
	if ((convertToString(value) != convertToString(fValue)) || ((value != null) && (fValue == null)))
		updateToModel(FORM_NOTIFICATION_ON_REQUEST);
}
String convertToString(Object o) {
	return convertToString(o, fInFocus);
}
String convertToString(Object o, boolean inFocus) {
	if (fDataType != null)
		return fDataType.convertToString(o, inFocus);
	if (o != null)
		return o.toString();
	return ""; // a good idea?
}
/**
 * Copy the receiver's text to the system clipboard.
 */
public void copy() {
	fTextComponent.copy();
}
/**
 * Cut the receiver's text to the system clipboard.
 */
public void cut() {
	fTextComponent.cut();
	setText(fTextComponent.getText());
	updateToModel(FORM_NOTIFICATION_ON_FOCUS_CHANGE);
}
void enableListeners(boolean state) {
	if (!fEnableSentToListeners || fLastEnableState != state) {
		fLastEnableState= state;
		fEnableSentToListeners= true;
		if (fEmptyStateListeners != null) {
			Enumeration e = fEmptyStateListeners.elements();
			while (e.hasMoreElements()) {
				IEnableListenerTarget c = (IEnableListenerTarget) e.nextElement();
				c.setEnabled(state);
			}
		}
	}
}
/**
 * This method is called from the UIPlainDocument class to filter
 * the text entered (pasted) into a field.
 * It may return the original string, a modified string, or null to indicate
 * invalid input.
 *
 * @see UIPlainDocument
 */
public String filterInput(int position, String inputString, AttributeSet attribute) {
	String newString = inputString;
	if (fDataType != null) {
		newString = fDataType.filterInput(position, newString, fTextComponent.getText(), attribute);
		if (newString == null)
			UI.beep();
	}
	return newString;
}
/**
 * Send all the changes in the receiver to the application.
 */
public void flush() {
	updateToModel(FORM_NOTIFICATION_ON_REQUEST);
}
/**
 * The receiver widget has gained focus. Do the
 * needful.
 *
 * @param event : the FocusEvent describing the focus change.
 */
public void focusGained(FocusEvent event) {
	fInFocus = true;
	updateFromModel(fInFocus);
	if (fFormModel != null)
		fFormModel.focusGained(event);
}
/**
 * The receiver widget has lost focus. Do the needful.
 *
 * @param event : the FocusEvent describing the focus change.
 */
public void focusLost(FocusEvent event) {
	fInFocus = false;
	updateToModel(FORM_NOTIFICATION_ON_FOCUS_CHANGE);
	if (fFormModel != null)
		fFormModel.focusLost(event);
}
/**
 * Set the receiver as an editable field.
 *
 * For Windows LAF we change the background
 * color to be identical to its parent (or grey
 * if no parent can be found)
 */
protected void forceSetEditable() {
	if (fTextComponent != null) {
		fTextComponent.setEditable(fEditable);
		switchBackgroundColor();
	}
}
/**
 * The form model is about to send its contents to the ULC Server.
 * Widgets that may have unsaved changes must save their changes to the formModel now.
 * This method will be triggered only when a saveInput is requested explicitly from the ULC Server.
 */
protected void formModelAboutToSaveInput() {
	updateToModel(FORM_NOTIFICATION_ON_REQUEST);
}
/**
 * The receiver is being destroyed. Release all the 
 * associated resources.
 */
public void free() {
	if (fTextComponent != null) {
		Document d = fTextComponent.getDocument();
		if (d != null) {
			d.removeDocumentListener(this);
			if (d instanceof UiPlainDocument) {
				UiPlainDocument pd = (UiPlainDocument) d;
				pd.free();
			}
		}
		fTextComponent.removeFocusListener(this);
		fTextComponent.removeAll();
		fTextComponent = null;
	}
	if (fScrollPane != null) {
		fScrollPane.removeFocusListener(this);
		fScrollPane.removeAll();
		fScrollPane = null;
	}
	fValue = null;
	fEmptyStateListeners = null;
	fDataType = null;
	super.free();
}
/**
 * Return the background of the receiver
 */
public Color getBackgroundColor() {
	if (!UIManager.getLookAndFeel().getName().equals("Windows"))
		return fEditableBackgroundColor;

	if (isEditable())
		return fEditableBackgroundColor;
	else
		return fReadOnlyBackgroundColor;
}
/**
 * Return the Component whose background can be set to show the mandatory property.
 * @return java.awt.Component
 */
protected Component getBackgroundComponent() {
	return getBasicComponent();
}
/**
 * Answer the basic component associated with the
 * receiver.
 */
public Component getBasicComponent() {
	return fTextComponent;
}
/**
 * Answer the component associated with the
 * receiver.
 */
 
public Component getComponent() {
	if (fScrollPane != null)
		return fScrollPane;
	return fTextComponent;
}
/**
 * Answer the validator object for the receiver
 */
public IDataType getDataType() {
	return fDataType;
}
/**
 * Returns the current value of this field.
 * A FormModel takes precedence over the fObject.
 */
protected Object getValue() {
	Object o = null;
	if (fFormModel != null) {
		o = fFormModel.getValueAt(fKey);
	} else {
		o = fValue;
	}
	return o;
}
/**
 * Handle LAF change
 */
protected void handleLAFChanged() {
	if (UIManager.getLookAndFeel().getName().equals("Windows")) {
		if (isEditable() && shouldEnableListener())
			fTextComponent.setBackground(fEditableBackgroundColor);
		else if (!isEditable())
			fTextComponent.setBackground(fReadOnlyBackgroundColor);
	}
}
/**
 * The ULC application has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("setText")) {
		setValue(args.asString("???"));
		return;
	}
	if (request.equals("setValue")) {
		if (args != null)
			setValue(convert(conn, args));
		return;
	}
	if (request.equals("setSelection")) {
		setSelection(args);
		return;
	}
	if (request.equals("setDataType")) {
		setDataType(conn, args);
		return;
	}
	if (request.equals("setEditable")) {
		setEditable(args.asBoolean(false));
		return;
	}
	if (request.equals("setLineWrap")) {
		setLineWrap(args.asBoolean(true));
		return;
	}
	if (request.equals("setWordWrap")) {
		setWordWrap(args.asBoolean(true));
		return;
	}
	if (request.equals("cut")) {
		cut();
		return;
	}
	if (request.equals("copy")) {
		copy();
		return;
	}
	if (request.equals("paste")) {
		paste();
		return;
	}
	if (request.equals("setSelectionForeground")) {
		setSelectionForegroundColor(args.get("r", -1), args.get("g", -1), args.get("b", -1), true);
		return;
	}
	if (request.equals("setSelectionBackground")) {
		setSelectionBackgroundColor(args.get("r", -1), args.get("g", -1), args.get("b", -1), true);
		return;
	}
	if (request.equals("setHorizontalAlignment")) {
		setHorizontalAlignment(args.asInt(BOX_LEFT));
		return;
	}
	if (request.equals("setTrigger")) {
		if (args.isNull())
			fTrigger = null;
		else
			fTrigger = (UITrigger) getManaged(UITrigger.class, conn, args);
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Do the necessary updates when the receiver document
 * has got some data inserted into it.
 *
 * @param event : DocumentEvent describing the insert.
 */
public void insertUpdate(DocumentEvent event) {
	enableListeners(shouldEnableListener());
	updateToModel(FORM_NOTIFICATION_IMMEDIATE);
}
/**
 * Answer boolean whether the receiver can be edited.
 */
protected boolean isEditable() {
	return fEditable;
}
/**
 * Answer boolean whether the receiver is enabled.
 */
public boolean isEnabled() {
	return fTextComponent.isEnabled();
}
/**
 * Paste the receiver's text from the system clipboard.
 */
public void paste() {
	fTextComponent.paste();
	setText(fTextComponent.getText());
	updateToModel(FORM_NOTIFICATION_ON_FOCUS_CHANGE);
}
/**
 * remove the given component as listener from being informed
 * when the state of the receiver changes from enabled to disabled,
 * or vice-versa.
 *
 * @param component : The Component which will act as listener.
 */
public void removeEnableListener(IEnableListenerTarget component) {
	Assert.isNotNull(component);
	if (fEmptyStateListeners == null)
		return;
	fEmptyStateListeners.removeElement(component);
}
/**
 * Do the necessary updates when the receiver document
 * has got some data deleted from it.
 *
 * @param event : DocumentEvent describing the deletion.
 */
public void removeUpdate(DocumentEvent event) {
	enableListeners(shouldEnableListener());
	updateToModel(FORM_NOTIFICATION_IMMEDIATE);
}
/**
 * Get the focus onto the receiver.
 */
public void requestFocus() {
	if (fTextComponent != null) {
		fTextComponent.requestFocus();
	}
}
/**
 * The FormComponent has been asked to throw away its changes.
 * Discard any previous values that this component may be keeping.
 */
protected void resetPreviousValue(int type, String key) {
	fValue= null;
}
/**
 * This method is the first method called after this widget is instantiated.
 * All widget specific initialization must take place in this method.
 * All the parameters necessary to initialize this widget are specified in the arguments.
 * Subclasses implementing this method must call the superclass implementation as well.
 *
 * @param conn 		the <code>UlcConnection</code> in which this operation is performed
 * @param args		the <code>Anything</code> containing the optional initialization parameters
 */
public void restoreState(ORBConnection conn, Anything args) {
	setDataType(conn, args.get("dataType"));
	fCols = args.get("cols", (fDataType != null) ? fDataType.getCols() : 10);
	fRows = args.get("rows", 1);
	setType(args.get("password", false));
	setHorizontalAlignment(args.get("ha", BOX_LEFT));
	Document d = fTextComponent.getDocument();
	if (d != null)
		d.addDocumentListener(this);
	setLineWrap(args.get("linewrap", true));
	setWordWrap(args.get("wordwrap", true));
	setInitialEditableColor();
	fEditable= args.get("editable", true);	
	super.restoreState(conn, args);
	DeferredRequest r = new DeferredRequest(conn) {
		public void safeDispatch() {
			fFirstTime= false;
			forceSetEditable();
		}
	};
	conn.postRequest(r);


	
	if (fFormModel != null)
		fNotificationPolicy = FORM_NOTIFICATION_ON_REQUEST;
	else {
		fValue = args.get("text", null);
		if (fValue == null) {
			if (args.isDefined("value"))
				fValue = convert(conn, args.get("value"));
		}
	}
	fTextComponent.addFocusListener(this);
	setDecoration(args, true);
	final Anything a = args;
	r = new DeferredRequest(conn) { //1F5SD2S
		public void safeDispatch() {
			updateFromModel();
			setSelection(a.get("selection"));
			if (fDataType != null)
				updateToModel(FORM_NOTIFICATION_ON_REQUEST); //1F8GSXR
		}
	};
	conn.postRequest(r);
	fTrigger = (UITrigger) getManaged(UITrigger.class, conn, args.get("trigger"));

}
/**
 * Set the background of the receiver to the color defined
 * by the RGB int values <code>red, green, and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
public void setBackgroundColor(int red, int green, int blue, boolean refresh) {
	if (! UIManager.getLookAndFeel().getName().equals("Windows"))
		super.setBackgroundColor(red, green, blue, refresh);
	else if (isEditable() || fFirstTime) {
		fFirstTime= false;
		super.setBackgroundColor(red, green, blue, refresh);
	}
	if (red == -1 || green == -1 || blue == -1)
		fEditableBackgroundColor= null;
	else
		fEditableBackgroundColor= new Color(red, green, blue);
}
/**
 * Set the validator for the receiver, as defined in the incoming
 * arguments.
 */
protected void setDataType(ORBConnection conn, Anything args) {
	if (args != null)
		fDataType = (IDataType) getManaged(IDataType.class, conn, args);
}
/**
 * Set the color and font decorations of the receiver to the colors / fonts defined
 * as Anythings in <code>args</code>.
 * <br>By default, UIComponents support fore- and background colors, and a font.
 * In addition, the receiver supports selection colors.
 * <br>update the receiver's UI exactly once if both <code>refresh</code> and any decoration
 * has been reset
 * <br>Answer true if the component color should be refreshed. If refresh is true, the answer
 * is invariably false, because the refresh is done if needed in this method
 * 
 * @param args Anything holds the anything specifications of all decorations
 * @param refresh boolean indicates whether the receiver should refresh its UI
 */
protected boolean setDecoration(Anything args, boolean refreshLocal) {
	Anything ob = args.get("obc");
	Anything of = args.get("ofc");
	boolean localRefresh = (ob != null || of != null);
	boolean refreshNeeded = super.setDecoration(args, !localRefresh);
	if (localRefresh) {
		if (of != null) {
			setSelectionForegroundColor(of.get("r", -1), of.get("g", -1), of.get("b", -1), (refreshLocal && (ob == null)));
		}
		if (ob != null) {
			setSelectionBackgroundColor(ob.get("r", -1), ob.get("g", -1), ob.get("b", -1), refreshLocal);
		}
	}
	return (localRefresh && !refreshLocal);
}
/**
 * Set the receiver as an editable field.
 *
 * For Windows LAF we change the background
 * color to be identical to its parent (or grey
 * if no parent can be found)
 */
public void setEditable(boolean editable) {
	if (fTextComponent != null) {
		fEditable= editable;
		if (fTextComponent.isEditable() != fEditable) {
			fTextComponent.setEditable(editable);
			switchBackgroundColor();
		}
	}
}
/**
 * Set the receiver as an enabled/disabled field.
 */
public void setEnabled(boolean state) {
	if (fTextComponent != null && fTextComponent.isEnabled() != state) {
		fTextComponent.setEnabled(state);
		enableListeners(shouldEnableListener());
	}
}
/**
 * Sets the alignment of the label's contents along the X axis.
 * <p>
 *
 * @param alignment  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>BOX_LEFT</code> (the default for text-only labels),
 *           <code>BOX_CENTER</code> (the default for image-only labels), or
 *           <code>BOX_RIGHT</code>.
 *
 * @see IDefaults 
 */
public void setHorizontalAlignment(int alignment) {
	if (fTextComponent instanceof JTextField) {
		JTextField tf = (JTextField) fTextComponent;
		tf.setHorizontalAlignment(alignment);
	}
}
protected void setInitialEditableColor() {
	fEditableBackgroundColor = UIManager.getDefaults().getColor("TextField.background");
	fFirstTime= true;
}
/**
 * Sets the line-wrapping policy of the text area.  If set
 * to true the lines will be wrapped if they are too long
 * to fit within the allocated width.  If set to false,
 * the lines will always be unwrapped. 
 *
 * @param wrap indicates if lines should be wrapped.
 */
public void setLineWrap(boolean wrap) {
	if (fTextComponent instanceof JTextArea) {
		JTextArea ta = (JTextArea) fTextComponent;
		ta.setLineWrap(wrap);
		ta.revalidate();
		ta.repaint();
	}
}
/**
 * Set the selection for the receiver based on the incoming
 * arguments.
 */
void setSelection(Anything args) {
	if (args != null && fTextComponent != null) {
		int doclen = fTextComponent.getDocument().getLength();
		int type = args.get("t", 0);
		int offset = args.get("o", 0);
		int length = args.get("l", 0);
		//System.out.println("Field.setSelection: " + type + " " + offset + " " + length);
		switch (type) {
			default :
			case 0 : // from start
				offset = 0 + offset;
				break;
			case 1 : // from current position
				offset = fTextComponent.getSelectionEnd() + offset;
				break;
			case 2 : // from end
				offset = doclen - offset;
				break;
		}
		if (offset > doclen)
			offset = doclen;
		if (offset < 0)
			offset = 0;
		fTextComponent.setCaretPosition(offset);
		if (length > 0)
			fTextComponent.moveCaretPosition(offset + length);
		else
			if (length < 0)
				fTextComponent.moveCaretPosition(doclen);
	}
}
/**
 * Set the background of the receiver's selection
 * to the color defined by the RGB int values <code>red, green,
 * and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */
public void setSelectionBackgroundColor(int red, int green, int blue, boolean refresh) {
	Color color = null;
	JTextComponent tc = (JTextComponent) getBasicComponent();
	if (tc != null) {
		if (red == -1 || green == -1 || blue == -1) {
			color = UIManager.getColor("TextField.selectionBackground");
		} else {
			color = new Color(red, green, blue);
		}
		tc.setSelectionColor(color);
		if (refresh) {
			refreshComponentColor(tc);
		}
		fTextComponent.setSelectionStart(fTextComponent.getSelectionStart()); //force the selection color to be repainted.
	}
}
/**
 * Set the background of the receiver's selection
 * to the color defined by the RGB int values <code>red, green,
 * and blue</code>. 
 * <br>Update the UI if <code>refresh</code> is true
 * <br>If any of the color values are -1, set the code to
 * the LookAndFeel default.
 * 
 * @param red int the red value of the RGB value
 * @param green int the red value of the RGB value
 * @param blue int the blue value of the RGB value
 * @param refresh boolean indicate whether to refresh the UI after
 * the change
 */

public void setSelectionForegroundColor(int red, int green, int blue, boolean refresh) {
	Color color = null;
	JTextComponent tc = (JTextComponent) getBasicComponent();
	if (tc != null) {
		if (red == -1 || green == -1 || blue == -1) {
			color = UIManager.getColor("TextField.selectionForeground");
		}
		else {
			color = new Color(red, green, blue);
		}
		tc.setSelectedTextColor(color);
		if (refresh) {
			refreshComponentColor(tc);
		}
	}
}
/**
 * Set the text of the TextComponent if its different from the current one.
 * Disable notification while doing so.
 */
private void setText(String text) {
	if (fTextComponent == null)
		return;
	String oldText = fTextComponent.getText();
	if (oldText == null)
		oldText = "";
	if (!oldText.equals(text)) {
		Document d = fTextComponent.getDocument();
		d.removeDocumentListener(this);
		try {
			fTextComponent.setText(text);
		} catch (IllegalStateException e) {
			// do nothing: document is already being changed
		}
		//fValue=text;
		d.addDocumentListener(this);
		enableListeners(shouldEnableListener());
	}
}
/**
 * Set the receiver to contain the appropriate widget type,
 * based on the available information that it needs to display.
 */
protected void setType(boolean password) {
	Document model = null;
	if (fTextComponent != null)
		model = fTextComponent.getDocument();
	if (password) {
		fTextComponent = new JPasswordField("", fCols) {
			public JToolTip createToolTip() {
				return new UiJMultiLineToolTip();
			}
			public void updateUI() {
				super.updateUI();
				if (UIField.this != null)
					UIField.this.handleLAFChanged();
			}
		};
		((JPasswordField) fTextComponent).addActionListener(this);
		// Swing 1.1 when it doesnt have enough space to display the component
		// displays the component with its minimum width. This is a workaround that
		// will at least display 10 chars of the component at the very minimum
		FontMetrics metrics = fTextComponent.getFontMetrics(fTextComponent.getFont());
		((JTextField) fTextComponent).setMinimumSize(new Dimension((fCols >= 10 ? 10 : fCols) * metrics.charWidth('m'), metrics.getHeight() + 4));
		// end hack

	} else {
		if (fRows <= 1) {
			fTextComponent = new JTextField("", fCols) {
				public JToolTip createToolTip() {
					return new UiJMultiLineToolTip();
				}
				public void updateUI() {
					super.updateUI();
					if (UIField.this != null)
						UIField.this.handleLAFChanged();
				}
			};
			// Swing 1.1 when it doesnt have enough space to display the component
			// displays the component with its minimum width. This is a workaround that
			// will at least display 10 chars of the component at the very minimum
			FontMetrics metrics = fTextComponent.getFontMetrics(fTextComponent.getFont());
			((JTextField) fTextComponent).setMinimumSize(new Dimension((fCols >= 10 ? 10 : fCols) * metrics.charWidth('m'), metrics.getHeight() + 4));
			// end hack
		 	((JTextField) fTextComponent).setColumns(fCols);
			((JTextField) fTextComponent).addActionListener(this);
			fTextComponent.setDocument(new UiPlainDocument(this));
		} else {
			fTextComponent = new JTextArea(fRows, fCols) {
				public JToolTip createToolTip() {
					return new UiJMultiLineToolTip();
				}
				public void updateUI() {
					super.updateUI();
					if (UIField.this != null)
						UIField.this.handleLAFChanged();
				}
			};
			fTextComponent.setDocument(new UiPlainDocument(this));
			fScrollPane = new UiJScrollPane(fTextComponent);
			fScrollPane.setBorder(new BevelBorder(BevelBorder.LOWERED));
			fScrollPane.setPreferredSize(new Dimension(10 * fCols, 16 * fRows));
		}
	}
	if (model != null)
		fTextComponent.setDocument(model);
}
/**
 * Change the current value of this field but only if there is no FormModel.
 * Otherwise ignore call and emit a diagnostic message.
 *
 * If no validator (fDataType) is present, then the value provided in the 
 * parameter can be set right away. If a validator is resent, however, we
 * need to inform validate the input before settin it.
 *
 */
protected void setValue(Object object) {
	if (fFormModel == null) {
		if (fDataType == null) {
			fValue = object; // can be set since nothing to validate
			updateFromModel(); // notify myself
		} else {
			Object newValue = null;
			try {
				newValue = fDataType.convertToObject(FORM_NOTIFICATION_ON_REQUEST, convertToString(object), fValue);
			} catch (DataTypeConversionException e) {
				UI.beep();
				//newValue= e.getValue();
			} finally {
				fValue = newValue;
				updateFromModel();
				sendEventULC("value", "value", convert(newValue));
			}
		}
	} else {
		//trouble("setValue", "can't do this because UIField already uses a FormModel");
		fFormModel.setValueAt(object, fKey, null, FORM_NOTIFICATION_ON_FOCUS_CHANGE);
	}
}
/**
 * Sets the word-wrapping policy of the text area.  If set
 * to true the lines will be wrapped at word breaks if they 
 * are too long to fit within the allocated width.
 * If set to false, the lines will be wrapped at character boundaries
 * 
 * @see setLineWrap
 * @param wrap indicates if lines should be wrapped at word boundaries.
 */
public void setWordWrap(boolean wrap) {
	if (fTextComponent instanceof JTextArea) {
		JTextArea ta = (JTextArea) fTextComponent;
		ta.setWrapStyleWord(wrap);
		ta.revalidate();
		ta.repaint();
	}
}
/**
 * Return true if the component for which I am an enabler should be enabled.
 *
 */
public boolean shouldEnableListener() {
	boolean state = false;
	if (fTextComponent != null)
		state = fTextComponent.getText().length() > 0;
	return !isEnabled() || state;
}
protected void switchBackgroundColor() {
	if (!UIManager.getLookAndFeel().getName().equals("Windows"))
		return;
	if (isEditable() && (fReadOnlyBackgroundColor == null || fReadOnlyBackgroundColor.equals(getBackgroundComponent().getBackground()))) {
		getBackgroundComponent().setBackground(fEditableBackgroundColor);
	} else if (!isEditable()) {
		Container parent = getBackgroundComponent().getParent();
		if (parent != null)
			fReadOnlyBackgroundColor = parent.getBackground();
		else {
			return; // do nothing - deferred request will fix it
		}
		if (fEditableBackgroundColor == null)
			fEditableBackgroundColor = getBackgroundComponent().getBackground();
		if (fEditableBackgroundColor.equals(getBackgroundComponent().getBackground()))
			getBackgroundComponent().setBackground(fReadOnlyBackgroundColor);
	}
	getBackgroundComponent().repaint();
}
/**
 * Update widget from model.
 */
public void updateFromModel() {
	fLastInput= null;
	Object value= getValue();
	if (value instanceof PendingCell)
		value= null;
	if (value == null && fValue != null)
		setText(convertToString(fValue));
	else if (value == null) {
		setText(convertToString(null));
		if (fDataType == null)
			fValue= convertToString(null);
		}
	else
		setText(convertToString(value, true));
}
/**
 * Update widget from model when the widget gets the focus.
 */
public void updateFromModel(boolean inFocus) {
	if (!inFocus) {
		updateFromModel();
		return;
	}
//	setEditable(fEditable);
	if ((fLastInput != null) /*&& (fLastInput.length() > 0)*/) {
		setText(fLastInput);
		return;
	}
	Object value = getValue();
	if (value == null) {
	} else
		setText(convertToString(value));
}
/**
 * Update model from widget.
 * If a dataType has been specified for this widget then the method
 * updateToModelForDataType(String,Policy) is called.
 */
protected void updateToModel(int policy) {
	/*
	 * DM:New valueChanged handling:
	 *		A value has changed if the user made some input.
	 *		No value comparison is made
	 */
	if (fTextComponent == null)
		return;
	String newString = fTextComponent.getText();

	// keep the currently entered character
	if (policy == FORM_NOTIFICATION_IMMEDIATE)
		fLastInput= newString;
	
	if (fDataType != null) {
		updateToModelForDataType(newString, policy);
		return;
	}
	if (fValue == null)
		fValue= "";

	setText(newString); //causes fValue to be changed

	if ((policy == FORM_NOTIFICATION_IMMEDIATE) && fFormModel != null)
		fFormModel.setValueAt(newString, fKey, this, policy);
	if (fNotificationPolicy == policy && fFormModel == null && !fValue.equals(newString)) {
		sendEventULC("value", "value", convert(newString));
		fValue= newString;
	}
}
/**
 * Update model from widget.
 */
protected void updateToModelForDataType(String newString, int policy) {
	Object newValue = null;
	boolean valueChanged= (policy == FORM_NOTIFICATION_IMMEDIATE);

	// convert the text in the field to a value
	try {
		newValue = fDataType.convertToObject(policy, newString, fValue);
	} catch (DataTypeConversionException e) {
		UI.beep();
		valueChanged= false;
		newValue= fDataType.getDefaultValue(newString);
	}

	String newValueAsString= convertToString(newValue);
	// if not in edit mode display the value as defined by the converter
	if (policy != FORM_NOTIFICATION_IMMEDIATE)
		setText(newValueAsString);

	if (fLastInput != null && policy == FORM_NOTIFICATION_ON_REQUEST) {
		fLastInput= newValueAsString;
		valueChanged= true;
	}

	// verify if input is invalid
	try {
		newValue = fDataType.convertToObject(FORM_NOTIFICATION_ON_REQUEST, newString, fValue);
	} catch (DataTypeConversionException e) {
		if (fFormModel == null && fNotificationPolicy != FORM_NOTIFICATION_IMMEDIATE) {
			valueChanged= false;			
			newValue= fDataType.getDefaultValue(newString);
		}
	}
	if (newValue == null && fFormModel == null && fNotificationPolicy != FORM_NOTIFICATION_IMMEDIATE)
		valueChanged= false;

	if (valueChanged && fFormModel != null) // PR 1FOCJS0
		fFormModel.setValueAt(newValue, fKey, this, policy);
	if (fNotificationPolicy == policy && fFormModel == null && ((fValue == null && newValue != null) || (fValue != null && !fValue.equals(newValue)))) {
		fValue= newValue;
		sendEventULC("value", "value", convert(newValue));
	}
}
}
