package com.ibm.ulc.comm;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.io.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;

/**
 * Encapsulates a request for a registered object.
 * An ORBRequests consists of a target
 * object id, a request name, and arguments.
 */
public class ORBRequest extends Request {
	protected Anything fData = null;
	protected ORBConnection fConnection = null;
	protected IAnythingWriter fWriter = null;
	/**
	 * Creates a request and associates with the given connection.
	 * This constructor is used for received requests.
	 * The request data is filled in by the read method.
	 * @see #read
	 */
	public ORBRequest(ORBConnection conn) {
		fConnection= conn;
	}
/**
 * Creates a request for the given target.
 * This constructor is used for requests to be sent.
 */
public ORBRequest(IAnythingWriter writer, int oid, String request, Anything args) {
	this(writer, oid, request, args, -1);
}
	/**
	 * Creates a request for the given target.
	 * This constructor is used for requests to be sent.
	 */
	public ORBRequest(IAnythingWriter writer, int oid, String request, Anything args, long contextId) {
		fWriter= writer;
		fData= new Anything();
		fData.put("id", oid);
		fData.put("req", request);
		if (contextId >= 0) 
			fData.put("contextId", contextId);
	   	if (args != null)
			fData.put("a", args);
	}
/**
 * Creates a request for the given target.
 * This constructor is used for requests to be sent.
 */
public ORBRequest(IAnythingWriter writer, int oid, String request, Anything args, long contextId, ORBConnection connection) {
	fWriter = writer;
	fData = new Anything();
	fData.put("id", oid);
	fData.put("req", request);
	if (contextId >= 0)
		fData.put("contextId", contextId);
	if (args != null)
		fData.put("a", args);
	fConnection = connection;
}
/**
 * Executes the request by looking up the target object in the
 * associated object space.
 */
public void dispatch() {
	try {
		String req = fData.get("req", null);
		int id = fData.get("id", 0);
		ICallable c = fConnection.find(id);
		if (c != null && req != null) {
			Anything args = fData.get("a");
			fConnection.setCurrentRequest(this);
			//System.out.println("Recv Req id: " + getRequestId() + " reply id: " + getReplyId());
			fConnection.receivedRequest(id, req, args);
			c.handleRequest(fConnection, req, args);
			fConnection.setCurrentRequest(null);
		} else if (fConnection.fDebug && (req == null || !req.equals("terminate")))
			System.out.print("ORBRequest.dispatch: object " + id + " not found; message: ");
	} catch (Throwable e) {
		System.err.print("ORBRequest.dispatch: got exception while processing:");
		fData.dump(System.err);
		e.printStackTrace();
	}
}
	/**
	 * Pretty print the request to the given output stream.
	 * For debugging purposes only.
	 */
	public void dump(OutputStream os) {
		fData.dump(os);
	}
/**
 * Gets the number of bytes a request will take up when streamed.
 * This is used for debugging and monitoring purposes only.
 */
public int getByteCount() {
	ByteArrayOutputStream os= new ByteArrayOutputStream();
	if (fWriter != null)
		write(os);
	else
		fData.write(os,new AnythingWriter2());   // packed format
	return os.size();
}
public ORBConnection getConnection() {
	return fConnection;
}
/**
 * This method was created in VisualAge.
 * @return com.ibm.ulc.util.Anything
 */
public Anything getData() {
	return fData;
}
	/**
	 * Get the name of this request.
	 * For debugging only.
	 */
	public String getName() {
		return fData.get("req", "??");
	}
/**
 * Returns the id of the request that triggered this request or -1 if undefined.
 * This is used for debugging and monitoring purposes only.
 */
public int getReplyId() {
	return fData.get("rpid", -1);
}
/**
 * Returns the unique id of this request within the active connection's life cycle.
 * This is used for debugging and monitoring purposes only.
 */
public int getRequestId() {
	return fData.get("rid", -1);
}
/**
 * Tests if a next request is available on the input stream.
 * If the sender allowed the read() to time out then this method
 * will never block - otherwise it could block.
 *
 * Note: This might put back the first character to the reader
 *
 * @return			<code>true</code> if available; <code>false</code> otherwise
 * @IOException		except the InterruptedIOException which cause <code>false</code> to be returned
 */
public boolean isAvailable(InputStream is) throws IOException {
	if (is.available() > 0)
		return true;
	try {
		IAnythingReader ar= fConnection.getAnythingReaderEx(is);
		if (ar.hasPutBack())
			return true;
	  	int c= is.read();
	  	ar.putBack(c);
	} catch (InterruptedIOException e) {
		return false;
	}
	return true;
}
	/**
	 * Reads a request from the given stream.
	 * Returns true on success; false on failure.
	 */
	public boolean read(InputStream is) {
	
		IAnythingReader ar= fConnection.getAnythingReader(is);
	  	fData= ar.read(is);

		if (fData != null && fData.get("req", null) != null)
			return true;
		return false;
	}
/**
 * Reads a request from the given stream.
 * @returns true on success; returns false or throws an exception on failure.
 */
public boolean readEx(InputStream is) throws IOException {
	IAnythingReader ar= fConnection.getAnythingReader(is);
  	fData= ar.readEx(is);
	if (fData != null && fData.get("req", null) != null)
		return true;
	return false;
}
/**
 * Sets the id of the request that triggered this request.
 * This is used for debugging and monitoring purposes only.
 */
public void setReplyId(int replyId) {
	fData.put("rpid", replyId);
}
/**
 * Sets the id of this request unique within the connection that it is running in.
 * This is used for debugging and monitoring purposes only.
 */
public void setRequestId(int reqId) {
	fData.put("rid", reqId);
}
	/**
	 * Write the request to the given output stream.
	 * The streaming format is determined by fWriter.
	 */
	public void write(OutputStream os) {
		if (fWriter != null)
			fWriter.print(os, fData);
		else
			trouble("write", "no AnythingWriter set!");
	}
/**
 * Write the request to the given output stream.
 * The streaming format is determined by fWriter.
 */
public void writeEx(OutputStream os)
throws InterruptedIOException, IOException {
	if (fWriter != null)
		fWriter.printEx(os,fData);
	else
		trouble("write", "no AnythingWriter set!");
}
}
