package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.comm.ORBConnection;

/**
 * This widget is used for representing an hierarchical relationship
 * between objects. The tree has a 'root  node' which has 'children';
 * each child in turn may have children of it's own. 
 * The tree is also associated with an adapter (ULCTreeModel) for mapping
 * onto the underlying hierarchical business objects. 
 * Note that the business objects do not need to be modelled as 'nodes'
 * in order to be used in the tree - for example, a Directory structure 
 * where a directory contains other directories and files, can also be
 * displayed easily as a tree.
 *
 * Also, a tree can have columns associated with it, in the same way
 * as a table can. Thus, each business object can display it's attributes 
 * (in addition to it's icon and label) in the tree.
 *
 * @see ULCTreeModel
 * @see ULCColumn 
 */

public class ULCTreeTable extends ULCAbstractTree {
	/**
	 * The current column resize mode.
	 	 * <pre>
	 * Valid values are :
	 *	TABLE_AUTO_RESIZE_OFF
	 * 	TABLE_AUTO_RESIZE_LAST_COLUMN
	 *	TABLE_AUTO_RESIZE_ALL_COLUMNS
	 * </pre>
	 * @serial
	 */
	private int fAutoResize = TABLE_AUTO_RESIZE_ALL_COLUMNS;

	/**
	 * The mandatory treeColumn which shows the tree in the receiver
	 */
	private ULCTreeColumn fTreeColumn;
/**
 * Constructs a new ULCTreeTable.
 */
public ULCTreeTable() {
	this(null, TREE_SINGLE_SELECTION, 10);
}
/**
 * Constructs a new ULCTreeTable which displays values from the given ULCAbstractTableModel.
 *
 * @param model 	The ULCAbstractTableModel that serves as the data source.
 */
public ULCTreeTable(ULCAbstractTableModel model) {
	this(model, TREE_SINGLE_SELECTION, 10);
}
/**
 * Constructs a new ULCTreeTable, which displays values from the 
 * given ULCAbstractTableModel.
 *
 * @param model 	The ULCAbstractTableModel that serves as the data source.
 * @param selectionMode 	Specifies the selection policy for the treeTable.
 * 							<pre>
 * 							Valid values are :
 *								TREE_SINGLE_SELECTION
 * 								TREE_CONTIGUOUS_SELECTION
 *								TREE_DISCONTIGUOUS_SELECTION
 * 							</pre>
 */
public ULCTreeTable(ULCAbstractTableModel model, int selectionMode) {
	this(model, selectionMode, 10);
}
/**
 * Constructs a new instance of ULCTreeTable, which displays values from the 
 * given ULCAbstractTableModel.
 *
 * @param model 	The ULCAbstractTableModel that serves as the data source.
 * @param selectionMode 	Specifies the selection policy for the treeTable.
 * 							<pre>
 * 							Valid values are :
 *								TREE_SINGLE_SELECTION
 * 								TREE_CONTIGUOUS_SELECTION
 *								TREE_DISCONTIGUOUS_SELECTION
 * @param rowHeight		The height specified as number of visible rows for this treeTable.
 * 							</pre>
 */
public ULCTreeTable(ULCAbstractTableModel model, int selectionMode, int rowHeight) {
	super(model, selectionMode, rowHeight);
	setDefaultColumn();
}
/**
 * Appends a ULCColumn at the right end of the existing columns of the receiver
 *
 * @param column	The <code>ULCColumn</code> to be added as the last column. 
 */
public void addColumn(ULCColumn column) {
	internalAdd(column);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right 
 * end of the existing columns of the receiver
 *
 * @param key	The <code>String</code> columnId for the new column
 */
public void addColumn(String key) {
	addColumn(key, key);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right 
 * end of the existing columns of the receiver
 *
 * @param colId	The <code>String</code> key to be used to retrieve values from the ULCTreeModel.
 * @param label	The <code>String</code> label for this column.
 */
public void addColumn(String colId, String label) {
	ULCColumn col = new ULCColumn(colId, label);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of 
 * the existing columns of the receiver
 *
 * @param colId	The <code>String</code> key to be used to retrieve values from the ULCTreeModel.
 * @param label	The <code>String</code> label for this column.
 * @param width The width of the column.
 */
public void addColumn(String colId, String label, int width) {
	ULCColumn col = new ULCColumn(colId, label, width);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right 
 * end of the existing columns of the receiver
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTreeModel.
 * @param 	label	
 *			The <code>String</code> label for this column.
 * @param width
 * 			The <code>int</code> width of this column.
 * @param 	editable 
 *			If true allow the column being added to allow direct editing.
 * @param 	renderer
 *			The <code>ULCComponent</code> that will be used to render values in this column.
 */
public void addColumn(String colId, String label, int width, boolean editable, ULCComponent renderer) {
	ULCColumn col = new ULCColumn(colId, label, renderer);
	col.setWidth(width);
	col.setEditable(editable);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right 
 * end of the existing columns of the receiver
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTreeModel.
 * @param 	label	
 *			The <code>String</code> label for this column.
 * @param 	renderer 
 *			The <code>ULCComponent</code> that will be used to render the 
 *			values in this column.
 */
public void addColumn(String colId, String label, ULCComponent renderer) {
	ULCColumn col = new ULCColumn(colId, label, renderer);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the 
 * right end of the existing columns of the receiver
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTreeModel.
 * @param 	label	
 *			The <code>String</code> label for this column.
 * @param 	editable 
 *			If true allow the column being added to allow direct editing.
 */
public void addColumn(String colId, String label, boolean editable) {
	ULCColumn col = new ULCColumn(colId, label);
	col.setEditable(editable);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right 
 * end of the existing columns of the receiver
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTreeModel.
 * @param 	label	
 *			The <code>String</code> label for this column.
 * @param 	editable 
 *			If true allow the column being added to allow direct editing.
 * @param 	renderer
 *			The <code>ULCComponent</code> that will be used to render values in this column.
 */
public void addColumn(String colId, String label, boolean editable, ULCComponent renderer) {
	ULCColumn col = new ULCColumn(colId, label, renderer);
	col.setEditable(editable);
	internalAdd(col);
}
/**
 * Appends ULCColumns at the right end of the existing columns of the receiver
 *
 * @param 	columns
 *			The <code>Vector</code> containing a list of <code>ULCColumn</code> objects. 
 */
public void addColumns(Vector columns) {
	internalAdd(columns);
}
/**
 * Add the receiver's attribute that should be preloaded to the UI to the specified Vector.
 *
 * @param vectorOfPreloadAttributes : The attributes to be prelaoded
 */
public void addPreloadTableAttributesInto(Vector vectorOfPreloadAttributes) {
	Vector columns = getColumns();
	for (int i = 0; i < columns.size(); i++) {
		ULCColumn column = (ULCColumn) columns.elementAt(i);
		if (!vectorOfPreloadAttributes.contains(column.getAttributeName())) {
			vectorOfPreloadAttributes.addElement(column.getAttributeName());
		}
	}
}
/**
 * Returns the column resize mode.
 *
 * @return mode The column resize behavior mode.
 * <pre>
 *	mode can be one of:
 *		TABLE_AUTO_RESIZE_OFF;
 *		TABLE_AUTO_RESIZE_LAST_COLUMN;
 *		TABLE_AUTO_RESIZE_ALL_COLUMNS;
 * </pre>
 */
public int getAutoResize() {
	return fAutoResize;
}
/**
 * Returns the current list of all ULCColumns in this tree.
 *
 * @return	The <code>Vector</code> containing a list of <code>ULCColumn</code> objects.
 */
public Vector getColumns() {
	Vector components = new Vector();
	Vector v = internalGetComponents();
	for (int i = 0; i < v.size(); i++) {
		Object next = v.elementAt(i);
		Assert.isTrue(next instanceof ULCColumn);
		components.addElement((ULCColumn) next);
	}
	return components;
}
/**
 * Answer the object identified by the given parameter.
 *
 * @param args	Anything	Contains the identifier to locate the
 * 							required object.
 * @return the Object identified by this id.
 *
 */
protected Object getNodeFrom(Anything args) {
	int nodeId = args.asInt(-1);
	return internalGetModel().getRow(nodeId);
}
/**
 * Answer the treeColumn, i.e. the column that shows the tree,
 * in the receiver.
 */
public ULCTreeColumn getTreeColumn() {
	if (fTreeColumn == null) {
		setTreeColumn(new ULCTreeColumn());
	}
	return fTreeColumn;
}
/**
 * Answer the attribute to ge displayed in the tree-column of
 * the receiver.
 */
public String getTreeColumnAttributeName() {
	return getTreeColumn().getAttributeName();
}
/**
 * Answer the label to ge displayed in the tree-column of
 * the receiver.
 */
public String getTreeColumnLabel() {
	return getTreeColumn().getLabel();
}
/**
 * Gets the tree column's width
 *
 * @return	the tree columns width as <code>int</code>
 * @since	R3.1
 */
public int getTreeColumnWidth() {
	return getTreeColumn().getWidth();
}
/**
 * Removes the specified <code>ULCColumn</code> from this tree.
 *
 * @param column The <code>ULCColumn</code> to be removed.
 */
public void removeColumn(ULCColumn column) {
	if (column != getTreeColumn())
		internalRemove(column);
}
/**
 * Removes all the specified ULCColumns from this tree.
 *
 * @param columns The <code>Vector</code> containing a list of <code>ULCColumn</code> objects.
 */
public void removeColumns(Vector columns) {
	if (columns.contains(getTreeColumn())) {
		Vector newColumns = new Vector();
		for (int i = 0; i < columns.size(); i++) {
			if (columns.elementAt(i) != getTreeColumn())
				newColumns.addElement(columns.elementAt(i));
		}
		removeColumns(newColumns);
	}
	else
		internalRemove(columns);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fAutoResize != TABLE_AUTO_RESIZE_ALL_COLUMNS)
		a.put("autoResize", fAutoResize);
}
/**
 * Sets the column resize mode.
 *
 * @param mode The column resize behavior mode.
 * <pre>
 *	mode can be one of:
 *		TABLE_AUTO_RESIZE_OFF;
 *		TABLE_AUTO_RESIZE_LAST_COLUMN;
 *		TABLE_AUTO_RESIZE_ALL_COLUMNS;
 * </pre>
 */
public void setAutoResize(int mode) {
	fAutoResize = mode;
	sendUI("setAutoResize", new Anything(fAutoResize));
}
/**
 * Sets the current list of all ULCColumns in this tree.
 *
 * @param	The <code>Vector</code> containing a list of <code>ULCColumn</code> objects.
 */
public void setColumns(Vector columns) {
	removeColumns(getColumns());
	addColumns(columns);
}
/**
 * The treeColumn should always be present. Set this treeColumn
 * as the default column on the receiver.
 * The identifier for this column is pre-defined 
 * @see TREETABLE_FIRST_COLUMN_ATTRIBUTE_NAME
 * This column has no label by default. 
 * Use @setTreeColumnLabel api to set this label if required.
 */
void setDefaultColumn() {
	ULCTreeColumn column = new ULCTreeColumn(TREETABLE_FIRST_COLUMN_ATTRIBUTE_NAME, "");
	setTreeColumn(column);
}
/**
 * Set the given treeColumn on the receiver.
 *
 * @param newTreeColumn :  The ULCTreeColumn to be set.
 */
public void setTreeColumn(ULCTreeColumn newTreeColumn) {
	if (fTreeColumn != null)
		fList.removeElement(getTreeColumn());
	fTreeColumn = newTreeColumn;
	if (fList == null)
		fList = new Vector();
	fList.insertElementAt(newTreeColumn, 0);
}
/**
 * Change the attribute to be displayed in the tree-column of
 * the receiver.
 *
 * @param treeColumnAttribute String : The new identifier for the tree column.
 */
public void setTreeColumnAttributeName(String treeColumnAttribute) {
	getTreeColumn().setAttributeName(treeColumnAttribute);
}
/**
 * Set the label to ge displayed in the tree-column of
 * the receiver.
 *
 * @param label String : The new label for the tree column. 
 */
public void setTreeColumnLabel(String label) {
	getTreeColumn().setLabel(label);
}
/**
 * Sets the tree column's width to <I>newWidth</I>.  If <I>newWidth</I>
 * exceeds the minimum or maximum width, it's adjusted to the
 * appropriate limiting value in the UI Engine.
 *
 * @param	newWidth The width to be set
 * @since	R3.1
 */
public void setTreeColumnWidth(int newWidth) {
	getTreeColumn().setWidth(newWidth);
}
/**
 * Return the fully qualified class path to my proxy class in the UI.
 *
 */
public String typeString() {
	return "com.ibm.ulc.ui.lists.UITreeTable";
}
}
