package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;


/**
 * Internal implementation detail of ULCCheckBox and ULCRadioButton.
 * This class represents a widget that can be used
 * as a toggle, i.e. it toggles it's state on each
 * button press.
 *
 * This class has been made public in order that it's documentation be
 * published in javadoc.
 */
public abstract class ULCToggleButton extends ULCFormComponent implements IEnabler {
	/**
	 * If true the toggle has been selected
	 * @serial	 	 
	 */
	protected boolean fSelected = false;
	/**
	 * The optional <code>int</code> character value of the mnemonic key.
	 * @serial	 	 
	 */
	protected int fMnemonic = -1;
	/**
	 * The optional <code>ULCButtonGroup</code> which this button is part off.
	 * @serial	 
	 */
	protected ULCButtonGroup fGroup = null;
	/**
	 * Any valid ULCTrigger object which will be triggered in the UI when 
	 * this widgets actionPerformed method is called
	 * @serial	 
	 */
	protected ITrigger fTrigger = null;
/**
 * Construct a ToggleButton with the given label.
 * The ToggleButton tracks the state of the specified (boolean) attribute of the given FormModel.
 *
 * @param formModel			The model that will acts as the data source
 * @param formAttributeName	The attribute of the model that will be interpreted as a boolean.
 * @param label				The string to be displayed
 */
public ULCToggleButton(IForm formModel, String formAttributeName, String label) {
	super(label, formModel, formAttributeName);
}
/**
 * Construct a ToggleButton with the given label.
 * The ToggleButton tracks the state of the specified (boolean) attribute of the given FormModel.
 *
 * @param formModel			The model that will acts as the data source
 * @param formAttributeName The attribute of the model that will be interpreted as a boolean.
 * @param label				The string to be displayed
 * @param selected			The initial selection state
 *
 * @deprecated in R1.32
 */
public ULCToggleButton(IForm formModel, String formAttributeName, String label, boolean selected) {
	super(label, formModel, formAttributeName);
	fSelected = selected;
}
/**
 * Construct a ToggleButton with the given label and the specified initial state.
 * On state changes the given listener is notified.
 *
 * @param label		The string to be displayed
 * @param listener	The widget interested in my value changes
 * @param selected	The initial selection state
 */
public ULCToggleButton(String label, IValueChangedListener listener, boolean selected) {
	this(label, selected);
	addValueChangedListener(listener);
}
/**
 * Construct a ToggleButton with the given label and the specified initial state.
 *
 * @param label		The string to be displayed
 * @param selected	The initial selection state
 */
public ULCToggleButton(String label, boolean state) {
	super(label);
	fSelected = state;
}
/**
 * Registers the given listener to begin receiving notifications
 * when changes are made to the toggle button.
 *
 * @param listener	The object interested in my changes.
 */
public void addValueChangedListener(IValueChangedListener listener) {
	internalAddListener("value", listener);
}
/**
 * Return the Button Group for this Toggle Button.
 *
 * @return ULCButtonGroup 
 */
public ULCButtonGroup getButtonGroup() {
	return fGroup;
}
/**
 * Gets the ToggleButtons's label.
 *
 * @return The current label string 
 */
public String getLabel() {
	return fLabel;
}
/**
 * Returns this widgets mnemonic code.
 *
 * @return int The keyCode that represents the character to be used as a mnemonic
 */
public int getMnemonic() {
	return fMnemonic;
}
/**
 * Returns the configured ULCTrigger object which will be triggered in the UI when 
 * this widget's actionPerformed method is called.
 *
 * @return ITrigger	 
 */
public ITrigger getTrigger() {
	return fTrigger;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("itemStateChange")) {
			fSelected = args.get("selected", fSelected);
			distributeToListeners("value", new ULCValueChangedEvent(this, new Boolean(fSelected)));
		}
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Returns true if I am currently selected in the UI.
 *
 * @return true if selected, false otherwise.
 */
public boolean isSelected() {
	return fSelected;
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive value changed events. 
 *
 * @param listener	The object that was interested in my changes.
 */
public void removeValueChangedListener(IValueChangedListener listener) {
	internalRemoveListener("value", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fGroup != null)
		a.put("group", fGroup.getRef(fContext));
	if (fSelected)
		a.put("selected", fSelected);
	if (fMnemonic != -1)
		a.put("mnemonic", fMnemonic);
	if (fTrigger != null)
		a.put("trigger", ((ULCProxy) fTrigger).getRef(fContext));
}
/**
 * Set the Button Group for this Toggle Button.
 *
 * @param ULCButtonGroup 
 */
public void setButtonGroup(ULCButtonGroup buttonGroup) {
	fGroup = buttonGroup;
}
/**
 * Sets the widgets's <code>String</code> label.
 * Note: If the label contains the <code>IDefaults.MNEMONIC_INDICATOR</code> then
 * the character after this indicator is treated as the mnemonic character for this widget.
 *
 * @param label the label <code>String</code> of the widget.
 */
public void setLabel(String label) {
	internalSetLabel(label);
}
/**
 * Sets this widgets mnemonic code.
 *
 * @param keyCode The keyCode that represents the character to be used as a mnemonic
 */
public void setMnemonic(int keyCode) {
	if (keyCode != fMnemonic) {
		fMnemonic = keyCode;
		sendUI("setMnemonic", new Anything(fMnemonic));
	}
}
/**
 * Sets the selected-state of the button.
 *
 * @param selected if true the button will be selected in the UI
 */
public void setSelected(boolean selected) {
	if (!fSelected == selected) {
		fSelected = selected;
		sendUI("setSelected", new Anything(fSelected));
	}
}
/**
 * Sets the configured ULCTrigger which will be triggered in the UI when 
 * this widgets actionPerformed method is called
 *
 * @param ULCTrigger	 
 */
public void setTrigger(ITrigger trigger) {
	if (trigger != fTrigger) {
		fTrigger = trigger;
		sendUI("setTrigger", (ULCTrigger)fTrigger);
	}
}
}
