package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.comm.*;
import java.awt.Color;

/**
 * The ULCTable displays a collection of objects in the form of a scrollable
 * table containing columns and allows the user to select from this list.
 * The list of items is specified via a ULCTableModel.
 *
 * @see ULCTableModel
 * @see ULCColumn
 */
public class ULCTable extends ULCTableList implements IEnabler {
	/**
	 * The current column resize mode.
	 	 * <pre>
	 * Valid values are :
	 *	TABLE_AUTO_RESIZE_OFF
	 * 	TABLE_AUTO_RESIZE_LAST_COLUMN
	 *	TABLE_AUTO_RESIZE_ALL_COLUMNS
	 * </pre>
	 * @serial
	 */
	private int fAutoResize = TABLE_AUTO_RESIZE_ALL_COLUMNS;
	/**
	 * The foreground color for the header of this widget.
	 * @serial	 
	 */
	protected Color fHeaderForeColor;
	/**
	 * The background color of the header of this widget.
	 * @serial	 
	 */
	protected Color fHeaderBackColor;
/**
 * Constructs a new ULCTable which displays values from the given ULCTableModel.
 */
public ULCTable() {
	super();
}
/**
 * Constructs a new ULCTable which displays values from the given ULCAbstractTableModel.
 *
 * @param model 		The ULCAbstractTableModel that serves as the data source.
 * @param width 		Width of the ScrollPanes content area in pixels.
 * @param heightInRows 	Height of the ScrollPanes content area in full rows.
 */
public ULCTable(ULCAbstractTableModel model, int width, int heightInRows) {
	super(model, width, heightInRows);
}
/**
 * Appends a ULCColumn at the right end of the ULCTable.
 *
 * @param column	The <code>ULCColumn</code> to be added as the last column.
 */
public void addColumn(ULCColumn column) {
	internalAdd(column);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right 
 * end of the existing columns of the receiver
 *
 * @param key	The <code>String</code> columnId for the new column
 */
public void addColumn(String key) {
	internalAdd(new ULCColumn(key, key));
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param key	The <code>String</code> label and columnId for the new column
 * @param width The width of the column.
 */
public void addColumn(String key, int width) {
	ULCColumn col = new ULCColumn(key, width);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param key		The <code>String</code> label and columnId for the new column
 * @param renderer 	ULCComponent used to render the values of this column.
 */
public void addColumn(String key, ULCComponent renderer) {
	ULCColumn col = new ULCColumn(key, key, renderer);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param colId	The <code>String</code> key to be used to retrieve values from the ULCTableModel.
 * @param label	The <code>String</code> label for this column.
 */
public void addColumn(String colId, String label) {
	ULCColumn col = new ULCColumn(colId, label);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param colId	The <code>String</code> key to be used to retrieve values from the ULCTableModel.
 * @param label	The <code>String</code> label for this column.
 * @param width The width of the column.
 */
public void addColumn(String colId, String label, int width) {
	ULCColumn col = new ULCColumn(colId, label, width);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTableModel.
 * @param 	label	
 *			The <code>String</code> label for this column.
 * @param 	renderer 
 *			The <code>ULCComponent</code> that will be used to render the 
 *			values in this column.
 */
public void addColumn(String colId, String label, ULCComponent renderer) {
	ULCColumn col = new ULCColumn(colId, label, renderer);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTableModel.
 * @param 	label	
 *			The <code>String</code> label for this column.
 * @param 	editable 
 *			If true allow the column being added to allow direct editing.
 */
public void addColumn(String colId, String label, boolean editable) {
	ULCColumn col = new ULCColumn(colId, label);
	col.setEditable(editable);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTableModel.
 * @param 	label	
 *			The <code>String</code> label for this column.
 * @param 	editable 
 *			If true allow the column being added to allow direct editing.
 * @param 	width
 *			The width in pixels for this column.
 */
public void addColumn(String colId, String label, boolean editable, int width) {
	ULCColumn col = new ULCColumn(colId, label, width);
	col.setEditable(editable);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTableModel.
 * @param 	label	
 *			The <code>String</code> label for this column.
 * @param 	editable 
 *			If true allow the column being added to allow direct editing.
 * @param 	renderer
 *			The <code>ULCComponent</code> that will be used to render values in this column.
 */
public void addColumn(String colId, String label, boolean editable, ULCComponent renderer) {
	ULCColumn col = new ULCColumn(colId, label, renderer);
	col.setEditable(editable);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTableModel.
 * @param 	editable 
 *			If true allow the column being added to allow direct editing.
 */
public void addColumn(String colId, boolean editable) {
	ULCColumn col = new ULCColumn(colId, colId);
	col.setEditable(editable);
	internalAdd(col);
}
/**
 * Constructs a new ULCColumn for the specified key and appends it at the right end of the ULCTable.
 *
 * @param 	colId	
 *			The <code>String</code> key to be used to retrieve values
 *			from the ULCTableModel.
 * @param 	editable 
 *			If true allow the column being added to allow direct editing.
 * @param 	width
 *			The width in pixels for this column.
 */
public void addColumn(String colId, boolean editable, int width) {
	ULCColumn col = new ULCColumn(colId, width);
	col.setEditable(editable);
	internalAdd(col);
}
/**
 * Append all the specified <code>ULCColumn's</code> to my list of columns.
 *
 * @param 	columns
 *			The <code>Vector</code> containing a list of <code>ULCColumn</code> objects.
 */
public void addColumns(Vector columns) {
	internalAdd(columns);
}
/**
 * Add all table attributes that should be preloaded to the UI to the specified Vector.
 * This receiver has no table attributes to preload.
 *
 * @param vectorOfPreloadAttributes	Vector	into which the receiver adds the
 *											table attributes it needs to preload.
 */
public void addPreloadTableAttributesInto(Vector vectorOfPreloadAttributes) {
	Vector columns = getColumns();
	for (int i = 0; i < columns.size(); i++) {
		ULCColumn column = (ULCColumn) columns.elementAt(i);
		if (!vectorOfPreloadAttributes.contains(column.getAttributeName())) {
			vectorOfPreloadAttributes.addElement(column.getAttributeName());
		}
	}
}
/**
 * Registers the given listener to begin receiving notifications
 * when one of the columns has been clicked upon triggering a sort request
 *
 * @param listener	The object interested in my sortEvents.
 */
public void addSortListener(ISortListener listener) {
	internalAddListener("sort", listener);
	enableOptionalEvent("sort");
}
/**
 * Returns the column resize mode.
 *
 * @return mode The column resize behavior mode.
 * <pre>
 *	mode can be one of:
 *		TABLE_AUTO_RESIZE_OFF;
 *		TABLE_AUTO_RESIZE_LAST_COLUMN;
 *		TABLE_AUTO_RESIZE_ALL_COLUMNS;
 * </pre>
 */
public int getAutoResize() {
	return fAutoResize;
}
/**
 * Returns the current list of all ULCColumns in this table.
 *
 * @return	The <code>Vector</code> containing a list of <code>ULCColumn</code> objects.
 */
public Vector getColumns() {
	Vector components = new Vector();
	Vector v = internalGetComponents();
	for (int i = 0; i < v.size(); i++) {
		Object next = v.elementAt(i);
		Assert.isTrue(next instanceof ULCColumn);
		components.addElement((ULCColumn) next);
	}
	return components;
}
/**
 * Returns the background color of the header for this widget.
 *
 * @return color The <code>Color</code> that should be used within this widget
 */
public Color getHeaderBackground() {
	return fHeaderBackColor;
}
/**
 * Returns the foreground color of the header for this widget.
 *
 * @return color The <code>Color</code> that should be used within this widget
 */
public Color getHeaderForeground() {
	return fHeaderForeColor;
}
/*
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", null);
		if (type.equals("sort")) {
			ULCColumn sortColumn = null, lastSortedColumn = null;
			int columnOid = args.get("col", -1);
			if (columnOid != -1) {
				ICallable c = fContext.getRegistry().find(columnOid);
				if (c != null)
					sortColumn = (ULCColumn) c;
			}
			columnOid = args.get("lastcol", -1);
			if (columnOid != -1) {
				ICallable c = fContext.getRegistry().find(columnOid);
				if (c != null)
					lastSortedColumn = (ULCColumn) c;
			}
			distributeToListeners("sort", new ULCSortEvent(this, lastSortedColumn, sortColumn));
			return;
		}
	}
	super.handleRequest(conn, request, args);
}
/**
 * Removes the specified <code>ULCColumn</code> from this table.
 *
 * @param column The <code>ULCColumn</code> to be removed.
 */
public void removeColumn(ULCColumn column) {
	internalRemove(column);
}
/**
 * Removes all the specified ULCColumns from this table.
 *
 * @param columns The <code>Vector</code> containing a list of <code>ULCColumn</code> objects.
 */
public void removeColumns(Vector columns) {
	internalRemove(columns);
}
/**
 * UnRegisters the given listener from receiving notifications
 * when one of the columns has been clicked upon triggering a sort request
 *
 * @param listener	The object interested in my sortEvents.
 */
public void removeSortListener(ISortListener listener) {
	internalRemoveListener("sort", listener);
	disableOptionalEvent("sort");
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fAutoResize != TABLE_AUTO_RESIZE_ALL_COLUMNS)
		a.put("autoResize", fAutoResize);
	if (fHeaderForeColor != null)
		a.put("hfc", internalConvertColor(fHeaderForeColor));
	if (fHeaderBackColor != null)
		a.put("hbc", internalConvertColor(fHeaderBackColor));
}
/**
 * Sets the column resize mode.
 *
 * @param mode The column resize behavior mode.
 * <pre>
 *	mode can be one of:
 *		TABLE_AUTO_RESIZE_OFF;
 *		TABLE_AUTO_RESIZE_LAST_COLUMN;
 *		TABLE_AUTO_RESIZE_ALL_COLUMNS;
 * </pre>
 */
public void setAutoResize(int mode) {
	fAutoResize = mode;
	sendUI("setAutoResize", new Anything(fAutoResize));
}
/**
 * Set the current list of all ULCColumns in this table.
 * All existing columns are removed before the new columns are set.
 *
 * @param	The <code>Vector</code> containing a list of <code>ULCColumn</code> objects.
 */
public void setColumns(Vector columns) {
	removeColumns(getColumns());
	addColumns(columns);
}
/**
 * Sets the background color of the header for this widget.
 *
 * @param color The <code>Color</code> that should be used within this widget
 */
public void setHeaderBackground(Color color) {
	fHeaderBackColor = color;
	if (color == null)
		sendUI("setHeaderBackground", new Anything());
	else
		sendUI("setHeaderBackground", internalConvertColor(fHeaderBackColor));
}
/**
 * Sets the foreground color of the header for this widget.
 *
 * @param color The <code>Color</code> that should be used within this widget
 */
public void setHeaderForeground(Color color) {
	fHeaderForeColor = color;
	if (color == null)
		sendUI("setHeaderForeground", new Anything());
	else
		sendUI("setHeaderForeground", internalConvertColor(fHeaderForeColor));
}
/**
 * returns the fully qualified class name of my UI class
 */
protected String typeString() {
	return "Table";
}
}
