package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.*;
/**
 * ULCSplitPane is used to divide two (and only two) <code>ULCComponents</code>. 
 * The two Components are graphically divided based on the look and feel
 * implementation, and the two components can then be interactively 
 * resized by the user.
 * <p>
 * The preferred way to change the size of the Components is to invoke
 * <code>setDividerLocation</code>.
 * <p>
 */
public class ULCSplitPane extends ULCComponent {
	/**
	 * If true the two components are split in vertically.
	 * @serial
	 */
	public boolean fVertical = false;
	/**
	 * The <code>ULCComponent</code> to be displayed on the left or on the top 
	 * depending on the orientation.
	 * @serial	 
	 */
	protected ULCComponent fLeftComponent = null;
	/**
	 * The <code>ULCComponent</code> to be displayed on the right or on the bottom 
	 * depending on the orientation.
	 * @serial	 
	 */
	protected ULCComponent fRightComponent = null;
	/**
	 * The current <code>double</code> dividerLocation. valid values are 0.0 to 1.0.
	 * @serial	 
	 */
	protected double fDividerLocation = -1.0;
/**
 * Construct a ULCSplitPane with vertical orientation.
 *
 */
public ULCSplitPane() {
	super("Splitpane");
	fVertical = true;
}
/**
 * Construct a ULCSplitPane with vertical orientation if <code>vertical</code> is true.
 *
 * @param 	vertical
 *			If true the components are arranged above/below each other.
 * @param 	leftComponent 
 *			the Component that will appear on the left of a horizontally-split pane, 
 *			or at the top of a vertically-split pane.
 * @param 	rightComponent 
 *			the Component that will appear on the right of a horizontally-split pane, 
 *			or at the bottom of a vertically-split pane.
 */
public ULCSplitPane(boolean vertical, ULCComponent leftComponent, ULCComponent rightComponent) {
	super("Splitpane");
	fVertical = vertical;
	fLeftComponent = leftComponent;
	fRightComponent = rightComponent;
}
/**
 * Gets the divider location as a percentage of the JSplitPane's size.
 *
 * @return proportionalLocation  a double-precision floating point value that
 *        specifies a percentage, from zero (top/left) to 1.0 (bottom/right)
 */
public double getDividerLocation() {
	return fDividerLocation;
}
/**
 * Gets the left component of this widget
 *
 * @return ULCComponent leftComponent 
 */
public ULCComponent getLeftComponent() {
	return fLeftComponent;
}
/**
 * Gets the right component of this widget
 *
 * @return ULCComponent rightComponent 
 */
public ULCComponent getRightComponent() {
	return fRightComponent;
}
/**
 * Returns true if the orientation is vertical. 
 * Components will be arranged on top of each other if vertical is true.
 *
 * @return  if false, Components will be arranged alongside each other.
 */
public boolean isVertical() {
	return fVertical;
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fVertical)
		a.put("v", fVertical);
	a.put("l", fLeftComponent.getRef(fContext));
	a.put("r", fRightComponent.getRef(fContext));
	if (fDividerLocation != -1.0)
		a.put("dl", fDividerLocation);
}
/**
 * Sets the divider location as a percentage of the JSplitPane's size.
 *
 * @param proportionalLocation  a double-precision floating point value that
 *        specifies a percentage, from zero (top/left) to 1.0 (bottom/right)
 */
public void setDividerLocation(double proportionalLocation) {
	if (fDividerLocation != proportionalLocation) {
		fDividerLocation= proportionalLocation;
		sendUI("setDividerLocation", new Anything(fDividerLocation));
	}
}
/**
 * Set the left component of this widget
 *
 * @param 	leftComponent 
 */
public void setLeftComponent(ULCComponent leftComponent) {
	fLeftComponent = leftComponent;
}
/**
 * Set the right component of this widget
 *
 * @param 	rightComponent 
 */
public void setRightComponent(ULCComponent rightComponent) {
	fRightComponent = rightComponent;
}
/**
 * Sets the orientation to be vertical if a boolean is true
 * Components will be arranged on top of each other if vertical is true.
 *
 * @param vertical if false, Components will be arranged alongside each other.
 */
public void setVertical(boolean vertical) {
	if (fVertical != vertical) {
		fVertical = vertical;
		sendUI("setVertical", new Anything(fVertical));
	}
}
}
