package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;

/**
 * The slider widget which can show the values (between a pre-defined
 * minimum and maximum) as different positions on the 'slider'.
 * The current value is either determined by the state of this widget or
 * by an attribute of a ULCFormModel.
 */
public class ULCSlider extends ULCFormComponent {
	/**
	 * @serial
	 */
	private int fValue = 0;
	/**
	 * @serial
	 */
	private int fMinValue = 0;
	/**
	 * @serial
	 */
	private int fMaxValue = 100;
	/**
	 * @serial
	 */
	private boolean fHorizontal = true;
	/**
	 * @serial
	 */
	private boolean fPaintLabels = false;
	/**
	 * @serial
	 */
	private boolean fPaintTicks = false;
	/**
	 * @serial
	 */
	private int fMajorTickSpacing = 10;
	/**
	 * @serial
	 */
	private int fMinorTickSpacing = 5;
/**
 * Creates a Slider for the range from 0 to 100 and the initial value of 0.
 */
public ULCSlider() {
}
/**
 * Creates a ULCSlider for the specified range and the specified value.
 *
 * @param value 	The <code>int</code> initial value to be displayed.
 * @param minimumValue 	The <code>int</code> minimum allowed value.
 * @param maximumValue 	The <code>int</code> maximum allowed value.
 * @param horizontal 	If true this Slider has a horizontal orientation, otherwise vertical.
 */
public ULCSlider(int value, int minimumValue, int maximumValue, boolean horizontal) {
	fValue = value;
	fMinValue = minimumValue;
	fMaxValue = maximumValue;
	fHorizontal = horizontal;
}
/**
 * Constructs a ULCSlider which tracks the state of the specified attribute of the given FormModel.
 *
 * @param model 		The <code>ULCFormModel</code> that acts as the data source.
 * @param key	 		The <code>String</code> key used to retrieve the value from the ULCFormModel.
 * @param minimumValue 	The <code>int</code> minimum allowed value.
 * @param maximumValue 	The <code>int</code> maximum allowed value.
 * @param horizontal 	If true this Slider has a horizontal orientation, otherwise vertical.
 */
public ULCSlider(IForm model, String key, int minimumValue, int maximumValue, boolean horizontal) {
	super(model, key);
	fMinValue = minimumValue;
	fMaxValue = maximumValue;
	fHorizontal = horizontal;
}
/**
 * Registers the given listener to begin receiving notifications
 * when changes are made to the slider.
 *
 * @param listener	The object interested in my changes.
 */
public void addValueChangedListener(IValueChangedListener listener) {
	internalAddListener("value", listener);
}
/**
 * Gets the major tick spacing for this slider.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @return majorTickSpacing The <code>int</code> majorTickSpacing value for this Slider.
 */
public int getMajorTickSpacing() {
	return fMajorTickSpacing;
}
/**
 * Returns the maximum value for this slider.
 *
 * @return The maximum <code>int</code> value 
 */
public int getMaximumValue() {
	return fMaxValue;
}
/**
 * Returns the minimum value for this slider.
 *
 * @return The minimum <code>int</code> value 
 */
public int getMinimumValue() {
	return fMinValue;
}
/**
 * Gets the minor tick spacing for this slider.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @return minorTickSpacing The <code>int</code> minorTickSpacing value for this Slider.
 */
public int getMinorTickSpacing() {
	return fMinorTickSpacing;
}
/**
 * Returns true if the labels are painted in the UI.
 *
 */
public boolean getPaintLabels() {
	return fPaintLabels;
}
/**
 * Returns true if the ticks are painted in the UI.
 *
 */
public boolean getPaintTicks() {
	return fPaintTicks;
}
/**
 * Returns the current value.
 *
 * @return The current <code>int</code> value 
 */
public int getValue() {
	return fValue;
}
/**
 * Returns the current value as an Integer.
 *
 * @return The current <code>Integer</code> value 
 */
public Integer getValueAsInteger() {
	return new Integer(fValue);
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("change")) {
			Object o = convert(args.get("value"));
			if (o instanceof Integer) {
				Integer i = (Integer) o;
				fValue = i.intValue();
				distributeToListeners("value", new ULCValueChangedEvent(this, i));
			}
		}
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Returns true if the slider orientation is horizontal
 *
 * @return true If slider is horizontal
 */
public boolean isHorizontal() {
	return fHorizontal;
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive value changed events. 
 *
 * @param listener	The object that was interested in my changes.
 */
public void removeValueChangedListener(IValueChangedListener listener) {
	internalRemoveListener("value", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fValue != 0)
		a.put("value", fValue);
	if (fMinValue != 0)
		a.put("minValue", fMinValue);
	if (fMaxValue != 100)
		a.put("maxValue", fMaxValue);
	if (!fHorizontal)
		a.put("horizontal", fHorizontal);
	if (fPaintTicks)
		a.put("paintTicks", fPaintTicks);
	if (fPaintLabels)
		a.put("paintLabels", fPaintLabels);
	if (fMajorTickSpacing != 10)
		a.put("majorts", fMajorTickSpacing);
	if (fMinorTickSpacing != 5)
		a.put("minorts", fMinorTickSpacing);
}
/**
 * set to false if the slider orientation is vertical
 * calling this method after the widget has been uploaded has no effect.
 */
public void setHorizontal(boolean horizontal) {
	fHorizontal = horizontal;
}
/**
 * Sets the major tick spacing for this slider.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @param majorTickSpacing The <code>int</code> majorTickSpacing value for this Slider.
 */
public void setMajorTickSpacing(int majorTickSpacing) {
	fMajorTickSpacing = majorTickSpacing;
}
/**
 * Sets the maximum <code>int</code> value. 
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @param value The <code>int</code> maximum value for this Slider.
 */
public void setMaximumValue(int value) {
	fMaxValue = value;
}
/**
 * Sets the minimum <code>int</code> value.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @param value The <code>int</code> minimum value for this Slider.
 */
public void setMinimumValue(int value) {
	fMinValue = value;
}
/**
 * Sets the minor tick spacing for this slider.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @param minorTickSpacing The <code>int</code> minorTickSpacing value for this Slider.
 */
public void setMinorTickSpacing(int minorTickSpacing) {
	fMinorTickSpacing = minorTickSpacing;
}
/**
 * Set to true if the labels should be painted in the UI.
 * Setting this value after the widget has been uploaded has no effect.
 */
public void setPaintLabels(boolean paintLabels) {
	fPaintLabels = paintLabels;
}
/**
 * Set to true if the labels should be painted in the UI.
 * Setting this value after the widget has been uploaded has no effect.
 */
public void setPaintTicks(boolean paintTicks) {
	fPaintTicks = paintTicks;
}
/**
 * Sets the current <code>int</code> value. It is adjusted to lie within the current range of the <code>ULCSlider</code>.
 *
 * @param value The <code>int</code> current value for this Slider.
 */
public void setValue(int value) {
	if (value != fValue) {
		fValue = value;
		sendUI("setValue", new Anything(fValue));
	}
}
}
