package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.awt.Dimension;
import java.awt.Point;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;

/**
 * An ULCShell represents the top level widget of a widget hierarchy, that is typically a modal or non-modal window.
 * (An exception is a ULC application running as an applet. In this case the ULCShell is a
 * AppletFrame).
 * A Shell has a title and/or an icon typically displayed in its window's title bar and an optional
 * menubar.
 */
public class ULCShell extends ULCComponent {
	/**
	 * The ULCApplication of this <code>ULCShell</code>.
	 * @serial	 
	 */
	protected ULCApplication fApplication = null;
	/**
	 * The optional parent shell of this <code>ULCShell</code>.
	 * @serial	 
	 */
	protected ULCShell fParent = null;
	/**
	 * The initial position this shell should open on. Default is -1, -1 which allows
	 * the UI to use a default position for the shell.
	 * @serial	 
	 */
	protected int fXPos = -1, fYPos = -1;
	/**
	 * The initial size of this shell. Default is -1, -1 which allows
	 * the UI to use the default size based on the layout of the shell.
	 * @serial	 
	 */
	protected int fWidth = -1, fHeight = -1;
	/**
	 * The <code>int</code> margin between the shell and its contents.
	 * @serial	 
	 */
	protected int fMargin = 5;
	/**
	 * The optional alignment string for this Shell. Default is null.
	 * @serial
	 * @see ULCBox
	 */
	protected String fAlign = null;
	/**
	 * The optional <code>ULCMenuBar</code> for this Shell.
	 * @serial	 
	 */
	protected ULCMenuBar fMenuBar = null;
	/**
	 * Internal
	 * @serial	 
	 */
	protected ULCComponent fModalComponent = null;
	/**
	 * If true the application will be asked before the shell is closed.
	 * @serial	 
	 */
	protected boolean fCloseBoxVeto = false;
	/**
	 * Internal
	 * @serial	 
	 */
	protected boolean fBlock = false;
	/**
	 * If true this shell is resizeable in the UI.
	 * @serial	 
	 */
	protected boolean fResizable = true;
	/**
	 * If true this shell should be opened modal. <code>fParent</code> must then hold the parent shell.
	 * @serial	 
	 */
	protected boolean fModal = false;
	/**
	 * The <code>ULCIcon</code> to be displayed in the title bar of this shell.
	 * @serial	 
	 */
	protected ULCIcon fIcon = null;
	/**
	 * The default <code>ULCButton</code>.
	 * @serial	 
	 */
	protected ULCButton fDefaultButton = null;
	/**
	 * If true this shell will be destroyed in the UI when closed.
	 * @serial	 
	 */
	protected boolean fDestroyOnClose = false;
	/**
	 * If true this shell has already been destroyed in the UI.
	 * @serial	 
	 */
	protected boolean fDestroyed = false;
public ULCShell() {
	super();
	fVisible = false;
}
/**
 * Constructs a new, initially invisible ULCShell with the specified parent shell and modal flag.
 *
 * @param parent If set and <code>modal</code> is true this shell is opened modal.
 * @param modal  If true this shell is opened modal to the <code> parent</code>
 * @param visible  If true this shell is opened visible
 */
public ULCShell(ULCShell parent, boolean modal, boolean visible) {
	super();
	fModal = modal;
	fVisible = visible;
	setParentShell(parent);
}
/**
 * Constructs a new, initially invisible ULCShell with the specified title.
 *
 * @param label The label <code>String</code> to be displayed in the title bar.
 */
public ULCShell(String title) {
	this(title, null, false);
}
/**
 * Constructs a new, initially invisible ULCShell with the specified title and parent shell.
 *
 * @param label	 The label <code>String</code> to be displayed in the title bar.
 * @param parent If set and <code>modal</code> is true this shell is opened modal.
 * @param modal  If true this shell is opened modal to the <code> parent</code>
 */
public ULCShell(String title, ULCShell parent, boolean modal) {
	super(title);
	fModal = modal;
	fVisible = false;
	setParentShell(parent);
}
/**
 * Constructs a new, initially invisible ULCShell with the specified title and parent shell.
 *
 * @param title	 	The label <code>String</code> to be displayed in the title bar.
 * @param parent 	If set and <code>modal</code> is true this shell is opened modal.
 * @param modal  	If true this shell is opened modal to the <code> parent</code>
 * @param visible	If true this shell is shown when created
 */
public ULCShell(String title, ULCShell parent, boolean modal, boolean visible) {
	super(title);
	fModal = modal;
	fVisible = visible;
	setParentShell(parent);
}
/**
 * Constructs a new, initially invisible ULCShell with the specified title.
 *
 * @param 	label	 The label <code>String</code> to be displayed in the title bar.
 * @param 	closeBoxVeto 
 *			If true application has to handle closebox clicks. 
 *			If false the UIEngine closes the window without application intervention.
 */
public ULCShell(String title, boolean closeBoxVeto) {
	this(title, null, false);
	fCloseBoxVeto = closeBoxVeto;
}
/**
 * Sets the ULCShell's content.
 *
 * @param c The <code>ULCComponent</code> that makes up this shells contents.
 */
public void add(ULCComponent c) {
	internalAdd(c);
}
/**
 * Append the given menu to the menubar. If a menubar doesn't exist this method creates one.
 *
 * @param menu The <code>ULCMenu</code> that is added to my Menubar.
 */
public void addMenu(ULCMenu menu) {
	if (fMenuBar == null)
		setMenuBar(new ULCMenuBar());
	fMenuBar.add(menu);
}
/**
 * Add a listener to be notified when this shell has the focus.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void addWindowActivatedListener(IWindowListener listener) {
	enableOptionalEvent("windowActivated");
	internalAddListener("windowActivated", listener);
}
/**
 * Add a listener to be notified when this shell is closed in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void addWindowClosedListener(IWindowListener listener) {
	enableOptionalEvent("windowClosed");
	internalAddListener("windowClosed", listener);
}
/**
 * Add a listener to be notified when this shell is closing in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void addWindowClosingListener(IWindowListener listener) {
	internalAddListener("windowClosing", listener);
}
/**
 * Add a listener to be notified when this shell looses the focus.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void addWindowDeactivatedListener(IWindowListener listener) {
	enableOptionalEvent("windowDeactivated");
	internalAddListener("windowDeactivated", listener);
}
/**
 * Add a listener to be notified when this shell is hidden in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void addWindowHiddenListener(IWindowListener listener) {
	enableOptionalEvent("windowHidden");
	internalAddListener("windowHidden", listener);
}
/**
 * Add a listener to be notified when this shell is moved in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void addWindowMovedListener(IWindowListener listener) {
	enableOptionalEvent("moved");
	internalAddListener("moved", listener);
}
/**
 * Add a listener to be notified when this shell is resized in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void addWindowResizedListener(IWindowListener listener) {
	enableOptionalEvent("resized");
	internalAddListener("resized", listener);
}
/**
 * Add a listener to be notified when this shell is shown in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void addWindowShownListener(IWindowListener listener) {
	enableOptionalEvent("windowShown");
	internalAddListener("windowShown", listener);
}
/**
 * Internal method. 
 */
void clearModalComponent(ULCComponent alert) {
	fModalComponent = null;
}
protected void destroy() {
	if (fDestroyed)
		return;
	fDestroyed = true;
	String type= "windowClosed";
	distributeToListeners(type, new ULCWindowEvent(this, type));
	free();
}
/**
 * This widget is being destroyed perform any cleanUp necessary.
 */
public void free() {
	super.free();
	fDestroyed = true;
	fApplication = null;
	fParent = null;
	fMenuBar = null;
	fModalComponent = null;
	fIcon = null;
	fDefaultButton = null;
}
/**
 * Internal method.
 */
public ULCApplication getApplication() {
	return fApplication;
}
/**
 * Return the current icon.
 */
public ULCIcon getIcon() {
	return fIcon;
}
/**
 * Gets the Shell's label.
 *
 * @return The current label string 
 */
public String getLabel() {
	return fLabel;
}
/**
 * Gets the location of this component in the form of a point specifying the shell's top-left corner.
 * The location will be relative to the parent's coordinate space.
 * Note: This method returns the current location in the ULCShell and does not
 * send a request to the UI to obtain the actual value in the UI. 
 * If <code>addWindowMovedListener</code> has not been called then the moved events
 * can be optimized in the UI and these values will not be updated.
 * 
 * @return An instance of Point representing the top-left corner of the shell's bounds in the coordinate space of the shell's parent.
 */
public Point getLocation() {
	return new Point(fXPos, fYPos);
}
/**
 * Gets the Shell's MenuBar.
 *
 * @return The current <code>ULCmenuBar</code> or null.
 */
public ULCMenuBar getMenuBar() {
	return fMenuBar;
}
/**
 * Gets this  Shell's parent.
 *
 * @return The parent <code>ULCShell</code> or null.
 */
public ULCShell getParentShell() {
	return fParent;
}
/**
 * Returns the size of this Shell in the form of a Dimension object.
 * The height field of the Dimension object contains this
 * Shell's height, and the width field of the Dimension object contains this Shell's width.
 * * Note: This method returns the current size in the ULCShell and does not
 * send a request to the UI to obtain the actual value in the UI. 
 * If <code>addWindowResizedListener</code> has not been called then the resized events
 * can be optimized in the UI and these values will not be updated.
 * 
 * @return A <code>Dimension</code> object that indicates the size of this shell. 
 */
public Dimension getSize() {
	return new Dimension(fWidth, fHeight);
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * Received events are converted to <code>ULCWindowEvents</code> and distributed to all
 * registered listeners.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	boolean distribute = true;
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("moved")) {
			fXPos = args.get("x", fXPos);
			fYPos = args.get("y", fYPos);
		} else
			if (type.equals("resized")) {
				fWidth = args.get("w", fWidth);
				fHeight = args.get("h", fHeight);
			} else
				if (type.equals("windowClosing")) {
					distributeToListeners(type, new ULCWindowEvent(this, type));
					distribute = false;
					handleWindowClosingRequest(conn, request, args);
				} else
					if (type.equals("windowHidden")) {
						// Do nothing except distributing the even (see below)
					} else
						if (type.equals("windowShown")) {
							fVisible = true;
						}
		if (distribute)
			distributeToListeners(type, new ULCWindowEvent(this, type));
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * The UI has sent a windowClosing event. Do all processing necessary.
 */
protected void handleWindowClosingRequest(ORBConnection conn, String request, Anything args) {
	if (!isVeto()) {
		fVisible= false;
		fApplication.shellClosed(this);
		if (fModal)
			fBlock= false;
		if (fDestroyOnClose)
			destroy();
	}
}
/**
 * Makes this shell hidden in the UI.
 */
public void hide() {
	setVisible(false);
}
/**
 * Return true if this shell will be destroyed when closed.
 * If destroyed the shell cannot be reopened.(Default is false)
 *
 * @return boolean
 */
public boolean isDestroyOnClose() {
	return fDestroyOnClose;
}
/**
 * Return the current modal setting.
 */
public boolean isModal() {
	return fModal;
}
/**
 * Indicates whether this Shell is resizable. By default, all Shells are initially resizable.
 *
 * @return true if the user can resize this frame; false otherwise.
 */
public boolean isResizable() {
	return fResizable;
}
/**
 * Returns if confirmation of windowClosing
 *
 */
public boolean isVeto() {
	return fCloseBoxVeto;
}
/**
 * Print this shell on a printer using
 * default page and printer settings.
 *
 * @since	R3.1 
 */
public void printScreen() {
	printScreen(false, false);
}
/**
 * Print this shell on a printer
 *
 * @param	showPrintDialog		Show dialog (<code>boolean</code>) to select printer and other print properties
 * @param	showPageDialog		Show dialog (<code>boolean</code>) to select page properties
 * @since	R3.1
 */
public void printScreen(boolean showPrintDialog, boolean showPageDialog) {
	Anything args= new Anything();
	args.put("printDialog", showPrintDialog);
	args.put("pageDialog", showPageDialog);
	sendUI("printScreen", args);
}
/**
 * Remove a listener from being notified when this shell has the focus.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void removeWindowActivatedListener(IWindowListener listener) {
	disableOptionalEvent("windowActivated");
	internalRemoveListener("windowActivated", listener);
}
/**
 * Remove a listener from being notified when this shell is closed in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void removeWindowClosedListener(IWindowListener listener) {
	disableOptionalEvent("windowClosed");
	internalRemoveListener("windowClosed", listener);
}
/**
 * Remove a listener from being notified when this shell is closing in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void removeWindowClosingListener(IWindowListener listener) {
	internalRemoveListener("windowClosing", listener);
}
/**
 * Remove a listener from being notified when this shell looses the focus.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void removeWindowDeactivatedListener(IWindowListener listener) {
	disableOptionalEvent("windowDeactivated");
	internalRemoveListener("windowDeactivated", listener);
}
/**
 * Remove a listener from being notified when this shell is hidden in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void removeWindowHiddenListener(IWindowListener listener) {
	disableOptionalEvent("windowHidden");
	internalRemoveListener("windowHidden", listener);
}
/**
 * Remove a listener from being notified when this shell is moved in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void removeWindowMovedListener(IWindowListener listener) {
	disableOptionalEvent("moved");
	internalRemoveListener("moved", listener);
}
/**
 * Remove a listener from being notified when this shell is resized in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void removeWindowResizedListener(IWindowListener listener) {
	disableOptionalEvent("resized");
	internalRemoveListener("resized", listener);
}
/**
 * Remove a listener from being notified when this shell is shown in the UI.
 *
 * @param listener The <code>IWindowListener</code> object that will be notified.
 */
public void removeWindowShownListener(IWindowListener listener) {
	disableOptionalEvent("windowShown");
	internalRemoveListener("windowShown", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fDefaultButton != null)
		a.put("defaultButton", fDefaultButton.getRef(fContext));
	Anything size = new Anything();
	if (fXPos != -1)
		size.put("x", fXPos);
	if (fYPos != -1)
		size.put("y", fYPos);
	if (fWidth != -1)
		size.put("w", fWidth);
	if (fHeight != -1)
		size.put("h", fHeight);
	if (size.size() > 0)
		a.put("size", size);
	if (fParent != null)
		a.put("parent", fParent.getRef(fContext));
	if (fModal)
		a.put("modal", fModal);
	if (fMargin != 5)
		a.put("m", fMargin);
	if (fAlign != null)
		a.put("a", fAlign);
	if (fMenuBar != null)
		a.put("menubar", fMenuBar.getRef(fContext));
	if (fModalComponent != null)
		a.put("alert", fModalComponent.getRef(fContext));
	if (fCloseBoxVeto)
		a.put("closeBoxVeto", fCloseBoxVeto);
	if (!fResizable)
		a.put("resizable", fResizable);
	if (fIcon != null) {
		boolean temp = fIcon.isCachedOnUI();
		fIcon.setCachedOnUI(false);
		a.put("icon", fIcon.getRef(fContext));
		fIcon.setCachedOnUI(temp);
	}
	if (fDestroyOnClose)
		a.put("destroyOnClose", fDestroyOnClose);
}
/**
 * Internal method.
 */
public void setApplication(ULCApplication application) {
	fApplication = application;
}
/**
 * Internal method.
 */
void setContext(ULCContext context) {
	fContext = context;
}
/**
 * Set the default button for this Shell.
 *
 * @param button Tne <code>ULCButton</code> that is to be the default button.
 */
public void setDefaultButton(ULCButton button) {
	if (fDefaultButton != button) {
		fDefaultButton = button;
		sendUI("setDefaultButton", fDefaultButton);
	}
}
/**
 * Set to true if this shell should be destroyed when closed.
 * If destroyed the shell cannot be reopened.(Default is false)
 * This method is only valid if set before the shell is uploaded.
 *
 * @return boolean
 */
public void setDestroyOnClose(boolean destroyOnClose) {
	fDestroyOnClose = destroyOnClose;
}
/**
 * Set the icon of the button.
 *
 * @param icon The ULCIcon to be displayed.
 */
public void setIcon(ULCIcon icon) {
	if (icon != fIcon) {
		icon.setCachedOnUI(false);
		fIcon = icon;
		sendUI("setIcon", fIcon);
	}
}
/**
 * Sets the widgets's <code>String</code> label.
 *
 * @param label the label <code>String</code> of the widget.
 */
public void setLabel(String label) {
	internalSetLabel(label);
}
/**
 * Moves this Shell to a new location.
 * The top-left corner of the new location is specified by the x and y parameters in the coordinate
 * space of this shell's parent.
 *
 * @param x The x-coordinate of the new shell's top-left corner in the parent's coordinate space.
 * @param y The y-coordinate of the new shell's top-left corner in the parent's coordinate space.
 */
public void setLocation(int x, int y) {
	Anything a = new Anything();
	a.put("x", fXPos = x);
	a.put("y", fYPos = y);
	sendUI("move", a);
}
/**
 * Moves this Shell to a new location.
 * The top-left corner of the new location is specified by the x and y parameters in the coordinate
 * space of this shell's parent.
 *
 * @param p The point defining the top-left corner of the new location,
 * 			given in the coordinate space of this component's parent. 
 */
public void setLocation(Point p) {
	setLocation(p.x, p.y);
}
/**
 * Sets the menubar for this shell if not already set.
 *
 * @param menubar 	The <code>ULCMenuBar</code> to be assigned to this shell.
 */
public void setMenuBar(ULCMenuBar menubar) {
	if (menubar != null) {
		fMenuBar = menubar;
		sendUI("setMenuBar", fMenuBar);
	}
}
/**
 * Sets the modal flag, which determines whether this Shell is modal.
 * By default, all shells are not modal.
 *
 * @param resizable if true, this shell is modal; false otherwise.
 */
public void setModal(boolean modal) {
	fModal = modal;
}
/**
 * Internal method.
 */
void setModalComponent(ULCComponent alert) {
	fModalComponent = alert;
}
/**
 * Sets this Shell's parent.
 *
 * @param The parent <code>ULCShell</code> or null.
 */
public void setParentShell(ULCShell parentShell) {
	fParent = parentShell;
	if (fParent != null)
		fParent.getApplication().add(this);
}
/**
 * Sets the resizable flag, which determines whether this Shell is resizable.
 * By default, all shells are initially resizable.
 *
 * @param resizable if true, this shell is resizable; false otherwise.
 */
public void setResizable(boolean resizable) {
	if (fResizable != resizable) {
		fResizable = resizable;
		sendUI("setResizable", new Anything(fResizable));
	}
}
/**
 * Resizes this component so that it has width width and height.
 *
 * @param width 	The new width of this component in pixels.
 * @param height 	The new height of this component in pixels.
 */
public void setSize(int width, int height) {
	Anything a = new Anything();
	a.put("w", fWidth = width);
	a.put("h", fHeight = height);
	sendUI("resize", a);
}
/**
 * Resizes this Shell so that it has width d.width and height d.height.
 *
 * @param d The dimension specifying the new size of this Shell. 
 */
public void setSize(Dimension d) {
	Anything a = new Anything();
	a.put("w", fWidth = d.width);
	a.put("h", fHeight = d.height);
	sendUI("resize", a);
}
/**
 * Requests confirmation of windowClosing
 *
 * @param b If true, enable confirmation
 */
public void setVeto(boolean b) {
	fCloseBoxVeto = b;
	sendUI("setVeto", new Anything(b));
}
/**
 * Shows or hides this shell depending on the value of <code>visible</code>.
 *
 * @param visible If true, shows this shell; otherwise, hides this shell.
 */
public void setVisible(boolean visible) {
	setVisible(visible, fModal);
}
/**
 * Shows or hides this shell depending on the value of <code>visible</code>.
 * If <code>block</code> is true then block the process and wait for the shell 
 * to be closed.
 *
 * @param visible 	If true, shows this shell; otherwise, hides this shell.
 * @param block		If true, blocks the process till the shell is closed.
 */
public void setVisible(boolean visible, boolean block) {
	if (fDestroyed) {
		trouble("setVisible", "ULCShell[" + getLabel() + "] is already destroyed cannot be reopened.");
		return;
	}
	if (visible) {
		fVisible = true;
		if (fModal)
			upload(fContext);
		else {
			Assert.isNotNull(fContext);
			fContext.upload(); // upload not only Shell but Application too
		}
		sendUI("setVisible", new Anything(fVisible));
		fApplication.shellShown(this);
		if (block) {
			fBlock = true;
			while ((fBlock) && !fContext.processNextRequest(0)) {
				try {
					Thread.sleep(100);
				} catch (Exception e) {
				}
			};
		}
	} else {
		fVisible = false;
		sendUI("setVisible", new Anything(fVisible));
		if (isUploaded())
			fApplication.shellClosed(this);
		if (fModal)
			fBlock = false;
		if (fDestroyOnClose)
			destroy();
	}
}
/**
 * Makes this shell visible in the UI.
 */
public void show() {
	setVisible(true);
}
/**
 * This returns the UI component name for this class.
 * Subclasses need to override this string only if they are not using the UIShell as their proxy.
 *
 * @return java.lang.String
 */
protected String typeString() {
	return "Shell";
}
}
