package com.ibm.ulc.application;

/*
 * Copyright (c) 1997-2000 Object Technology International Inc.
 */
import com.ibm.ulc.base.*;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.util.*;

/**
 * This class is used to launch programs or work on files on the UI side.
 * The program starter can either be used as a simple interface to exec on
 * the UI side or as a smarter program starter.
 * Note: If the UI is being run as an Applet it might cause a <code>SecurityException</code>
 * if the corresponding priveleges are not granted.
 *
 * @see		java.lang.Runtime#exec
 * @since	R3.1
 */
public class ULCProgramStarter extends ULCProxy {
/**
 * Constructor to create a ULCProgramStarter with a <code>null</code> ULCContext.
 * Note: The context must be set in order to work correctly!
 *
 * @since	R3.1 
 */
public ULCProgramStarter() {
}
/**
 * Constructor to create a ULCBrowserContext on a ULCContext.
 *
 * @param	context	The ULCContext on which this instance runs
 * @since	R3.1
 */
public ULCProgramStarter(ULCContext context) {
	super(context);
}
/**
 * Registers the given listener to begin receiving notifications
 * when an error occurs.
 *
 * @param	listener	The object interested in my link error events.
 * @since	R3.1
 */
public void addProgramStarterErrorListener(IProgramStarterErrorListener listener) {
	enableOptionalEvent("error");
	internalAddListener("error", listener);
}
/**
 * Execute the specified <code>commandLine</code> in a separate process with the 
 * specified environment variables. It is up to the caller to pass a commandLine
 * which will be accepted by exec.
 *
 * Note: Using this call is most likely NOT PLATFORM INDEPENDENT!
 *
 * @param		commandLine	a commandLine (can also be only a file).
 * @param		environment	array of strings, each element of which 
 *							has environment variable settings in format
 *							<i>name</i>=<i>value</i>.
 * @see			java.lang.Runtime#exec(String, String[])
 * @since		R3.1
 */
public void exec(String commandLine, String environment[]) {
	Anything args= new Anything();
	args.put("cmdLine", commandLine);
	if (environment != null)
		args.put("env", new Anything(environment));

	if (!isUploaded())
		upload(fContext);
	
	sendUI("exec", args);
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If the call chain has reached this object it means that the request is an unknown request.
 * Write the received request to the console.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	String type;
	if (request.equals("event")
		&& ((type= args.get("type", "???")).equalsIgnoreCase("error"))) {
		distributeToListeners("error",
			new ULCProgramStarterErrorEvent(
				this,
				args.get("method", "???"),
				args.get("cmdLine", "???"),
				args.get("error", "???")));
	}
}
/**
 * Deregister the given listener from receiving notifications
 * when an error occurs.
 *
 * @param	listener	The object interested in my link error events.
 * @since	R3.1
 */
public void removeProgramStarterErrorListener(IProgramStarterErrorListener listener) {
	disableOptionalEvent("error");
	internalRemoveListener("error", listener);
}
/**
 * Start the specified <code>commandLine</code> in a separate process with the 
 * specified environment variables. Start is more sophisticated than exec because,
 * if supported by the OS, it can also launch files. In this case it is up to the
 * OS to find the program (eg Windows uses file extension).
 *
 * Underneath it is the same as <code>exec</code> with some enhancements:
 * The Java system property <i>ulc.programStarter.environment</i> is used
 * to get the environment for start. If this property is not defined
 * or can not be accessed then:
 * - under Windows 9x and OS2 <i>start</i> is used to execute the command
 * - under Windows NT and Windows 2000 a command shell created and the
 *   <code>commandLine</code> is executed with <i>start</i>
 * - for all other platforms the call is equal to exec.
 *
 * Note: Using this call is most likely NOT PLATFORM INDEPENDENT!
 *
 * @param      commandLine	a commandLine (can also be only a file).
 * @param      environment	array of strings, each element of which 
 *							has environment variable settings in format
 *							<i>name</i>=<i>value</i>.
 */
public void start(String command, String environment[]) {
	Anything args= new Anything();
	args.put("cmdLine", command);
	if (environment != null)
		args.put("env", new Anything(environment));

	if (!isUploaded())
		upload(fContext);

	sendUI("start", args);
}
}
