package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.comm.ORBConnection;

/**
 * A Pagebook provides the facility to display one of several
 * components on the UI, depending on the value of the fPage field of this widget or
 * by an attribute of a ULCFormModel.
 * Each possible component is then a 'page' in the pagebook;
 * and is associated with an identifier (a string). Then, based
 * on the request from the application, the appropriate 'page'
 * (component) will be made visible. All other pages of the
 * pagebook remain hidden.
 */
public class ULCPagebook extends ULCFormComponent {
	/**
	 * @serial	 	 
	 */
	private String fPage = null;
	/**
	 * @serial	 	 
	 */
	private int fWidth = 0;
	/**
	 * @serial	 	 
	 */
	private int fHeight = 0;
/**
 * Constructs an empty ULCPageBook.
 */
public ULCPagebook() {
}
/**
 * Constructs an empty ULCPageBook with the given width and height.
 * The dimensions have to be given in advance because pages are loaded on demand and their size
 * is not know initially.
 */
public ULCPagebook(int width, int height) {
	fWidth = width;
	fHeight = height;
}
/**
 * Constructs an empty ULCPageBook which tracks the state of the specified attribute of the given FormModel.
 * The value of this attribute is used to select the shown page.
 */
public ULCPagebook(IForm formModel, String formAttributeName) {
	super(formModel, formAttributeName);
}
/**
 * Constructs an empty ULCPageBook which tracks the state of the specified attribute of the given FormModel.
 * The value of this attribute is used to select the shown page.
 * The dimensions have to be given in advance because pages are loaded on demand and their size
 * is not know initially.
 */
public ULCPagebook(IForm formModel, String formAttributeName, int width, int height) {
	super(formModel, formAttributeName);
	fWidth = width;
	fHeight = height;
}
/**
 * Constructs an empty ULCPageBook which tracks the state of the specified attribute of the given FormModel.
 * The value of this attribute is used to select the shown page.
 * The dimensions have to be given in advance because pages are loaded on demand and their size
 * is not know initially.
 * If enablePageChangeEvent is true then each time the page changes in the UI the application is notified
 */
public ULCPagebook(IForm formModel, String formAttributeName, int width, int height, boolean enablePageChangeEvent) {
	super(formModel, formAttributeName);
	fWidth = width;
	fHeight = height;
	if (enablePageChangeEvent)
		enableOptionalEvent("tabChanged");
}
/**
 * Add the specified page to the receiver.
 *
 * @param page	A ULCPage
 */
public void addPage(ULCPage page) {
	internalAdd(page);
}
/**
 * Add a page to the receiver, giving the title of the
 * page and the component associated with this title.
 * Then, when the receiver is asked to display the page
 * with the given title, this component will be displayed.
 */
public void addPage(String title, ULCComponent c) {
	internalAdd(new ULCPage(title, c, 0, false));
}
/**
 * Add a page to the ULCPageBook, giving the title of the
 * page and the component associated with this title.
 * When the receiver is asked to display the page
 * with the given title, this component will be displayed.
 * @param preload Determines whether the contents of this pages should be loaded on demand or preloaded.
 */
public void addPage(String title, ULCComponent c, boolean preload) {
	internalAdd(new ULCPage(title, c, 0, preload));
}
/**
 * Registers the given listener to begin receiving notifications
 * when switching pagebook pages.
 *
 * @param listener	The object interested in my UlcPageChangedEvents.
 */
public void addPageChangedListener(IPageChangedListener listener) {
	enableOptionalEvent("tabChanged");
	internalAddListener("tabChanged", listener);
}
/**
 * Adds the collection of <code>ULCPage's</code> to the notebook.
 *
 * @param pages A <code>Vector</code> containing <code>ULCPages's</code>
 */
public void addPages(Vector pages) {
	internalAdd(pages);
}
public String getActivePage() {
	return fPage;
}
/**
 * Gets the current height of the Pagebook.
 */
public int getHeight() {
	return fHeight;
}
/**
 *
 * @deprecated As of R1.4 use <code>getActivePage</code>
 */
public String getPage() {
	return getActivePage();
}
/**
 * Return the <code>Vector</code> containing all the <code>ULCPage's</code> in this pagebook.
 *
 * @return The <code>Vector</code> of <code>ULCPage's</code>
 */
public Vector getPages() {
	Vector components = new Vector();
	Vector v = internalGetComponents();
	for (int i = 0; i < v.size(); i++) {
		Object next = v.elementAt(i);
		Assert.isTrue(next instanceof ULCPage);
		components.addElement((ULCPage) next);
	}
	return components;
}
/**
 * returns the current width of the Pagebook
 */
public int getWidth() {
	return fWidth;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("tabChanged")) {
			fPage = args.get("tab", fPage);
			distributeToListeners("tabChanged", new ULCPageChangedEvent(this, fPage));
			return;
		}
	}
	super.handleRequest(conn, request, args);
}
public void removePage(ULCComponent c) {
	internalRemove(c);
}
/**
 * Remove the specified ULCpage from my list of pages.
 *
 * @param page	The <code>ULCpage</code> to be removed.
 */
public void removePage(ULCPage page) {
	internalRemove(page);
}
/**
 * Remove the specified ULCPage from my list of pages.
 * Removes the first page named <code>label</code>.
 * Does nothing if there is no page named <code>label</code>.
 *
 * @param label	The label of the <code>ULCPage</code> to be removed.
 */
public void removePage(String label) {
	for (Enumeration pages = getPages().elements(); pages.hasMoreElements();) {
		ULCPage page = (ULCPage) pages.nextElement();
		if (page.getLabel().equals(label))
			removePage(page);
	}
}
/**
 * deRegisters the given listener to begin receiving notifications
 * when switching pagebook pages.
 *
 * @param listener	The object interested in my UlcPageChangedEvents.
 */
public void removePageChangedListener(IPageChangedListener listener) {
	disableOptionalEvent("tabChanged");
	internalRemoveListener("tabChanged", listener);
}
public void removePages(Vector v) {
	internalRemove(v);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fWidth != 0)
		a.put("w", fWidth);
	if (fHeight != 0)
		a.put("h", fHeight);
	if (fPage != null)
		a.put("page", fPage);
}
/**
 * Make the component associated with the given
 * identifier (string) visible on the UI.
 */
public void setActivePage(String page) {
	if (fPage == null || !fPage.equals(page)) {
		fPage = page;
		sendUI("setPage", new Anything(fPage));
	}
}
/**
 * Sets the current height of the pagebook.
 * This method must be used before the pagebook is uploaded to the UI.
 * If used after the pagebook is uploaded it has no effect.
 */
public void setHeight(int height) {
	fHeight = height;
}
/**
 * Make the component associated with the given
 * identifier (string) visible on the UI.
 *
 * @deprecated As of R1.4 use <code>setActivePage</code>
 */
public void setPage(String page) {
	setActivePage(page);
}
/**
 * Set my collection of pages
 */
public void setPages(Vector pages) {
	removePages(getPages());
	addPages(pages);
}
/**
 * Sets the current width of the Pagebook.
 * This method must be used before the pagebook is uploaded to the UI.
 * If used after the pagebook is uploaded it has no effect.
 */
public void setWidth(int width) {
	fWidth = width;
}
}
