package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;

/**
 * A ULCItemListAbstract defines the API for maintaining a collection of elements 
 * within a specified order. The objects themselves are maintained in the ULCTableModel.
 * List Widgets have to access rows by index through an item list.
 *
 * @see ULCTableModel
 */
abstract public class ULCItemListAbstract extends ULCProxy implements IEnabler, IItemList {
	/**
	 * the invalid row identifier
	 */
	protected static int INVALID_OID = -1;
	/**
	 * @serial	 
	 */
	private static final boolean DEBUG = false;
	/**
	 * the TableModel holding the rows of the receiver
	 */
	protected ULCAbstractTableModel fModel = null;
	/**
	 * define the size of batches uploaded to the UI with every index request
	 */
	private int fPrefetch = 20;
	/**
	 * the list of widgets currently using the receiver
	 */
	protected IItemListOwner[] fOwners = new IItemListOwner[0]; 
	/**
	 * an item list that has no widget using it is considered inactive. In that case
	 * it does not keep its state in synch with the TableModel notifications. When re-activated,
	 * an item list that is not in synch with its TableModel must update its state and inform
	 * the UI.
	 */
	private boolean fActive = false;
	/**
	 * When the receiver is inactive and it receives a notification, the needsRefresh flag is set to true.
	 * When the receiver is (re)activated, and  this flag is set, the #contentsChanged message is sent to
	 * force a resynch  with the receiver's model
	 */
	private boolean fNeedsRefresh = false;

/**
 * Construct a nwe instance of the receiver
 */
public ULCItemListAbstract() {
}
/**
 * Add an owner to the receiver.
 *
 * @param owner 	IItemListOwner	The owner to be added
 */
public void addOwner(IItemListOwner owner) {
	for (int i = 0; i < fOwners.length; i++) {
		if (fOwners[i] == owner)
			return;
	}
	IItemListOwner[] newOwners = new IItemListOwner[fOwners.length + 1];
	if (fOwners.length > 0)
		System.arraycopy(fOwners, 0, newOwners, 1, fOwners.length);
	else
		setActive(true);
	newOwners[0] = owner;
	fOwners = newOwners;
}
/**
 * The contents of the receiver's tableModel has changed. The receiver must reinitialize
 * itself. It is called when the receiver has been uploaded upon changes in the receiver's
 * TableModel, or before the receiver is saving its state.
 *
 * @see #saveState
 */
abstract protected void contentsChanged();
/**
 * Prepare the given object identifiers, so that they may be uploaded
 * to the UI. Depending on the boolean flag, prepare the preload attributes
 * of the receiver for upload as well.
 *
 * @param oids	int[]	The oid's to be uploaded
 * @param withPreloadAttributes 	Determines whether to upload the preloadAttributes as well.
 */
protected Vector convertOidsForUpload(int[] oids, boolean withPreloadAttributes) {
	Vector answer = new Vector();
	int[] indices = new int[oids.length];
	for (int i = 0; i < indices.length; i++)
		indices[i] = getIndexForOid(oids[i]);
	Vector ranges = UlcRange.createFromIntArray(indices);
	for (int i = 0; i < ranges.size(); i++) {
		UlcRange range = (UlcRange) ranges.elementAt(i);
		Anything table = new Anything();
		answer.addElement(table);
		table.put("if", range.fStartIndex);
		table.put("it", range.fEndIndex);
		int[] rowids = getOidRange(range.fStartIndex, range.fEndIndex);
		table.put("rowids", rowids);
		if (withPreloadAttributes) {
			Anything parm = new Anything();
			parm.append(table);
			getModel().getData(parm);
		}
		boolean indicesEqualsOids = true;
		for (int ri = 0; ri < rowids.length; ri++) {
			if (rowids[ri] != (range.fStartIndex + ri)) {
				indicesEqualsOids = false;
				break;
			}
		}
		if (indicesEqualsOids) {
			table.remove("rowids");
		} else {
			if (rowids.length > 3) {
				boolean sorted = true;
				for (int ri = 0; ri < (rowids.length - 1); ri++) {
					if (Math.abs(rowids[ri] - rowids[ri + 1]) != 1) {
						sorted = false;
						break;
					}
				}
				if (sorted) {
					table.remove("rowids");
					table.put("of", rowids[0]);
					table.put("ot", rowids[rowids.length - 1]);
				}
			}
		}
	}
	return answer;
}
/**
 * Get the data from the receiver based on the incoming
 * args, and send the reply to the UI.
 * The actual data is accessed by the receiver from it's model.
 *
 * @param args		The <code>Anything</code> specifying the
 *					data required by the UI.
 */
protected void getData(Anything args) {
	int start = args.get("s").asInt(-1);
	int end = args.get("e").asInt(-1);
	if (end < 0 || start < 0)
		return;
	start = Math.min(start, getRowCount());
	end = Math.min(end, getRowCount());
	int[] oids = getOidRange(start, end);
	Anything parm = new Anything();
	Anything data = new Anything();
	data.put("if", start);
	data.put("it", end);
	data.put("rowids", oids);
	data.put("rows", getRowCount());
	parm.append(data);
	getModel().getData(parm);
	boolean removeRowids = true;
	for (int i = start; i <= end; i++) {
		if (oids[i - start] != i) {
			removeRowids = false;
			break;
		}
	}
	if (removeRowids)
		data.remove("rowids");
	sendUI("setData", data);
}
/**
 * Return the index for the rowId specified. Answer -1 if the oid was not found.
 *
 * @param oid	int	The rowId whose index is required.
 */
abstract public int getIndexForOid(int oid);
/**
 * Answer the receiver's tableModel
 */
public ULCAbstractTableModel getModel() {
	return fModel;
}
/**
 * Return the row id for the index or <code>INVALID_OID</code> if not found
 *
 * @param index	int	The index for which the oid is required
 */
abstract public int getOidForIndex(int index);
/**
 * Answer the row identifiers of the rows specified by the range.
 *
 * Both <code>from</code> and <code>to</code> must be indices of the receiver, and not of some
 * other itemList
 *
 * @param	from	the range start index
 * @param	to		the range end index
 */
private int[] getOidRange(int from, int to) {
	int[] answer = new int[to - from + 1];
	for (int i = from; i <= to; i++) {
		answer[i - from] = getOidForIndex(i);
	}
	return answer;
}
/**
 * Answer the number of rows to upload initially.
 */
protected int getPrefetchCount() {
	if (fModel == null)
		return 0;
	int answer = Math.max(fModel.getPrefetchCount(), fPrefetch);
	if (hasPreloadedColumns()) {
		return getRowCount();
	}
	return answer;
}
/**
 * Answer the array of attributeNames that the receiver needs to upload with every row.
 *
 * This array may cause unrequested attributes to be uploaded to the UI.
 */
public String[] getPreloadedAttributes() {
	Vector vector = new Vector();
	for (int i = 0; i < fOwners.length; i++) {
		fOwners[i].addPreloadTableAttributesInto(vector);
	}
	String[] answer = new String[vector.size()];
	vector.copyInto(answer);
	return answer;
}
/**
 * Answer the row at the specified index in the receiver
 *
 * @param index	The index of the row being accessed.
 */
public Object getRowAt(int index) {
	return getModel().getRow(getOidForIndex(index));
}
/**
 * Override this method to return the number of rows of this table.
 *
 * The number answered does not have to be the same as that of the receiver's tableModel.
 * ItemLists may display only excerpts from the tableModel
 */
abstract public int getRowCount();
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("requestIndices")) {
		setOidIndices(args);
		return;
	}
	if (request.equals("getData")) {
		getData(args);
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Answer true, if the receiver or any of its owners defines any preload columns
 */
protected boolean hasPreloadedColumns() {
	for (int i = 0; i < fOwners.length; i++) {
		if ((fOwners[i].computePreloadedColumns()).length > 0)
			return true;
	}
	return false;
}
/**
 * Answer the array of row identifiers that should be uploaded when the receiver 
 * is uploaded for the first time.
 */
public int[] initialOids() {
	int prefetchCount = Math.min(getPrefetchCount(), getRowCount());
	if (prefetchCount == 0)
		return new int[0];
	int[] oids = getOidRange(0, prefetchCount - 1);
	return oids;
}
/**
 * Set the <code>ULCAbstractTableModel</code> that will serve as my data source.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * @param tableModel	The <code>ULCAbstractTableModel</code>
 */
protected void internalSetModel(ULCAbstractTableModel model) {
	if (fModel != model) {
		removeFromModel(fModel);
		fModel = model;
		modelChanged(model);
		if (shouldProcessNotification())
			notifyContentsChanged();
	}
}
/**
 * Answer true, if the receiver is set to active.
 */
protected boolean isActive() {
	return fActive;
}
/**
 * The TableModel of the receiver has been (re)set. Add the receiver to its itemLists.
 *
 * @param model The new <code>ULCAbstractTableModel</code>
 */
protected void modelChanged(ULCAbstractTableModel model) {
	model.addItemList(this);
}
/**
 * Set to true, if the receiver is set to active.
 *
 * @param bool 	Boolean		Whether the receievr should be refreshed
 */
protected void needsRefresh(boolean bool) {
	fNeedsRefresh = bool;
}
/**
 * The entire contents of the receiver's model has changed. 
 * Update the receiver and its UI proxy accordingly.
 */
public void notifyContentsChanged() {
	if (shouldProcessNotification()) {
		contentsChanged();
		sendContentsChanged();
	}
}
/**
 * The rows specified by rowIds have been added to the receiver's model. 
 * Update the receiver and its UI proxy accordingly.
 *
 * @param rowIds	int[]	The set of rowIds that have been added to my model
 */
public void notifyRowsAdded(int[] rowIds) {
	if (shouldProcessNotification()) {
		Anything args = new Anything();
		args.put("rows", getRowCount());
		args.put("ranges", convertOidsForUpload(rowIds, false));
		sendUI("insertRows", args);
	}
}
/**
 * The rows specified by rowIds have changed. Default is do nothing, 
 * as a row change does not affect the order of items in the list. 
 * Subclasses may need to reimplement this method.
 *
 * @param rowIds 			int[] 		The rowids of the changed nodes.
 * @param attributeNames 	String[] 	The changed attributes
 */
public void notifyRowsChanged(int[] rowIds, String[] attributeNames) {
}
/**
 * The rows specified by rowIds have been removed from the receiver's model. 
 * Update the receiver, but do NOT update the UI, as this is done internally 
 * via the update of the UI ItemCache.
 *
 * @param rowIds		The row identifiers removed from my model
 */
abstract public void notifyRowsRemoved(int[] rowIds);
/**
 * Sort the nodes in the given array.
 */
private void quickSort(int[] list, int left, int right) {
	int original_left = left;
	int original_right = right;
	int mid = list[ (left + right) / 2];
	do {
		while (list[left] < mid) {
			left++;
		}
		while (mid < list[right]) {
			right--;
		}
		if (left <= right) {
			int tmp = list[left];
			list[left] = list[right];
			list[right] = tmp;
			left++;
			right--;
		}
	} while (left <= right);
	if (original_left < right) {
		quickSort(list, original_left, right);
	}
	if (left < original_right) {
		quickSort(list, left, original_right);
	}
}
/**
 * Remove the receiver from the given model.
 *
 * @param model		ULCAbstractTableModel	The model from which I have to remove myself
 */
public void removeFromModel(ULCAbstractTableModel model) {
	if (model != null)
		model.removeItemList(this);
}
/**
 * Remove the given owner from the receiver.
 *
 * @param owner 	IItemListOwner	The owner to remove
 */
public void removeOwner(IItemListOwner owner) {
	boolean removed = false;
	for (int i = 0; i < fOwners.length; i++) {
		if (fOwners[i] == owner) {
			fOwners[i] = null;
			removed = true;
			break;
		}
	}
	if (!removed)
		return;
	IItemListOwner[] newOwners = new IItemListOwner[fOwners.length - 1];
	int counter = 0;
	for (int i = 0; i < fOwners.length; i++) {
		if (fOwners[i] != null)
			newOwners[counter++] = fOwners[i];
	}
	fOwners = newOwners;
	if (fOwners.length == 0)
		setActive(false);
}
/**
 * Store the data available with the initial upload to anything.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveInitialData(Anything anything) {
	Vector oids = convertOidsForUpload(initialOids(), false);
	if (oids.size() > 0)
		anything.put("rowids", (Anything) oids.firstElement());
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	setActive(true);
	super.saveState(a);
	int prefetch = getPrefetchCount();
	if (prefetch != 20)
		a.put("prefetchCount", prefetch);
	if (fModel != null)
		a.put("model", getModel().getRef(fContext));
	a.put("rows", getRowCount());
	saveInitialData(a);
}
/**
 * Send the changed contents of the receiver to the UI.
 */
protected void sendContentsChanged() {
	Anything args = new Anything();
	Vector oids = convertOidsForUpload(initialOids(), false);
	if (oids.size() > 0)
		args.put("r", new Anything(oids));
	args.put("rows", getRowCount());
	sendUI("contentsChanged", args);
}
/**
 * Set to true, if the receiver is set to active.
 *
 * @param active	boolean		New active state of the receiver
 */
protected void setActive(boolean active) {
	if ((isActive() != active) && isUploaded()) {
		fActive = active;
		if (isActive() && fNeedsRefresh) {
			notifyContentsChanged();
			needsRefresh(false);
		}
	}
}
/**
 * Set the model which provides the data for the receiver.
 *
 * @param model 	ITableModel	The model for the receiver
 */
public void setModel(ITableModel model) {
	internalSetModel((ULCAbstractTableModel) model);
}
/**
 * send the indices of the oids passed to the UI.
 */
private void setOidIndices(Anything args) {
	Anything answer = new Anything();
	Anything rowIdsAny = args.get("rowids");
	int[] rowIds= new int[rowIdsAny.size()];
	int i= 0;
	for (Enumeration enum= rowIdsAny.toCollection().elements(); enum.hasMoreElements();) {
		rowIds[i]= ((Integer) enum.nextElement()).intValue();
		i++;
	}
	answer.put("r", convertOidsForUpload(rowIds, true));
	answer.put("rows", new Anything(getRowCount()));
	sendUI("setIndices", answer);
}
/**
 * Answer true, if the receiver is set to active.
 */
protected boolean shouldProcessNotification() {
	if (!isActive())
		needsRefresh(true);
	return (isActive());
}
private int[] sort(int[] toBeSorted) {
	int[] answer = new int[toBeSorted.length];
	System.arraycopy(toBeSorted, 0, answer, 0, toBeSorted.length);
	if (answer.length > 1)
		quickSort(answer, 0, answer.length - 1);
	return answer;
}
private int[] sort(Vector toBeSorted) {
	int[] ints = new int[toBeSorted.size()];
	for (int i = 0; i < ints.length; i++) {
		ints[i] = ((Integer) toBeSorted.elementAt(i)).intValue();
	}
	return sort(ints);
}
/**
 * returns a logical name for this component.
 * This name is used to locate the other half of this object in the UI Engine.
 * The default implementation extracts the name from the class name by stripping
 * off the package name and an optional prefix "ULC".
 * widgets that are not found in the com.ibm.ulc.ui.swing package should
 * override this method to return the fully qualified class name of the UI class
 * eg: com.ibm.ulc.ui.UIApplication
 *
 * @return The Logical name for this component.
 */
protected String typeString() {
	
	return "ItemList";
}
}
