package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;

/**
 * A ULCItemListAbstract is an abstract implementation for an object that returns
 * the elements in a specific order. 
 * List Widgets have to access rows by index through an item list.
 *
 * @see ULCAbstractTableModel
 */
abstract public class ULCIndexedItemList extends ULCItemListAbstract implements IIndexedItemList {
	/**
	 * The last oid assigned by the receiver, while mapping a row indices to oids.
	 */
	protected int fLastOid = INVALID_OID;

	/**
	 * The mapping of oid toindex and vice-versa is handled by this class hierarchy.
	 */
	protected ULCIndexToOidMapAbstract fOidMap;
/**
 * Construct a new ULCIndexedItemList
 */
public ULCIndexedItemList() {
	//setOidMap(new ULCLookupIndexToOidMap());
	setOidMap(new ULCArrayedIndexToOidMap());
}
/**
 * The contents of the receiver's tableModel has changed. The receiver must reinitialize
 * itself. It is called when the receiver has been uploaded upon changes in the receiver's
 * TableModel, or before the receiver is saving its state.
 *
 * The default ItemList installs all mapping information.
 *
 * @see #saveState
 */
protected void contentsChanged() {
	getOidMap().clear();
	getObjectIdentifiers(0, Math.min(getPrefetchCount(), getRowCount()));
}
/**
 * Return the index for the rowId specified. Answer -1 if the oid was not found.
 *
 * @param oid	int	The rowId whose index is required.
 */
public int getIndexForOid(int oid) {
	return getOidMap().getIndexForOid(oid);
}
/**
 * Internal method for collecting Object identifiers for the selected rows.
 * If the row/oid identifiers do not exist they are allocated and inserted
 * into the mapping tables.
 *
 * @param	startRow 		int		The index of the first row whose oid is required
 * @param	numberOfRows 	int		The number or rows (starting at the startRow), 
 *									whose oids are required
 */
public int[] getObjectIdentifiers(int startRow, int numberOfRows) {
	int[] answer = new int[numberOfRows];
	for (int i = 0; i < numberOfRows; i++) {
		answer[i] = getOidForIndex(i + startRow);
	}
	return answer;
}
/**
 * Return the row id for the index or <code>INVALID_OID</code> if not found
 *
 * @param index	int	The index for which the oid is required
 */
public int getOidForIndex(int index) {
	return getOidMap().getOidForIndex(index, INVALID_OID);
}
/**
 * Return the object managing the index to oid mappings
 */
protected ULCIndexToOidMapAbstract getOidMap() {
	return fOidMap;
}
/**
 * Answer the total number of rows in the receiver
 */
public int getRowCount() {
	return fModel.getRowCount();
}
/**
 * An oid was not defined at index.
 * Subclasses must reimplement this method. To do that, they must have a way to find the
 * row at the specified index from application code.
 *
 * @param index		int		The index at which the oid is missing
 */
abstract protected int handleMissingOid(int index);
/**
 * Insert the given oids to the receiver. Start this insertion at the specified index.
 *
 * @param insertionIndex 	the index at which the oids are to be inserted
 * @param oids				the array of oids to be inserted
 */
public void insertRowsAt(int insertionIndex, int[] oids) {
	getOidMap().insertRowsAt(insertionIndex, oids);
}
/**
 * The rows specified by rowIds have been removed from the receiver's model. 
 * Update the receiver's oid map, but do NOT update the UI, as this is done 
 * internally via the update of the UI ItemCache.
 *
 * @param rowIds		The row identifiers removed from my model
 */
public void notifyRowsRemoved(int[] rowIds) {
	if (shouldProcessNotification()) {
		getOidMap().removeRows(rowIds);
	}
}
/**
 * Set the object managing the index to oid mappings.
 *
 * @param map 	ULCIndexToOidMapAbstract The new index-oid mapping.
 */
public void setOidMap(ULCIndexToOidMapAbstract map) {
	map.setItemList(this);
	if (fOidMap != null)
		fOidMap.initialize(map);
	fOidMap = map;
}
}
