package com.ibm.ulc.application;

import java.util.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.ORBConnection;
/**
 * A ULCHierarchicalItemList defines the API for maintaining a collection of elements 
 * in a hierarchical order. The objects themselves are maintained in the ULCTreeTableModel.
 *
 * @see ULCTreeTableModel
 */
abstract public class ULCHierarchicalItemList extends ULCItemListAbstract implements IHierarchicalItemList {
	/**
	 * A hashtable to hold the mapping from an objectIdentifier (oid) to the actual
	 * node
	 */
	protected UlcHashtable fOidToNode = new UlcHashtable();

	/**
	 * The root of the hierarchical structure.
	 */
	protected Object fRoot;

	/**
	 * The initial nodes to upload to the UI. By default the root and it's immediate
	 * children are uploaded. Any additional nodes to be uploaded should be added here.
	 */
	protected Vector fInitialNodes = new Vector();
/**
 * Construct a new ULCHierarchicalItemList
 */
public ULCHierarchicalItemList() {
	super();
}
/**
 * Convert the given nodes, along with their complete paths, into an
 * Anything which can be uploaded to the UI.
 *
 * @param nodes		Vector	The set of nodes whose paths are to be built.
 *
 * @see #getPath(int)
 */
protected Anything buildPathsFor(Vector nodes) {
	int[][] paths = new int[nodes.size()][];
	for (int i = 0; i < nodes.size(); i++) {
		int rowId = internalGetModel().getRowIdFor(nodes.elementAt(i));
		if (!isUploaded(rowId)) {
			paths[i] = getPath(rowId);
		} else
			paths[i] = new int[0];
	}
	Anything args = new Anything();
	for (int i = 0; i < paths.length; i++) {
		Anything pathAny = new Anything();
		if (paths[i].length > 0) {
			for (int pi = 0; pi < paths[i].length; pi++) {
				Anything nodeAny = new Anything();
				convertNodeForUpload(nodeAny, paths[i][pi]);
				if (nodeAny.isNull())
					return null;
				pathAny.append(nodeAny);
			}
		}
		args.append(pathAny);
	}
	return args;
}
/**
 * This method is called when the specified rowIds should be reloaded with their complete subtrees.
 * In order to force a reload from the data source, the receiver's model needs to remove all traces of
 * the currently defined children of the specified nodes. The oids of all the known children are added
 * by all nodes concerned to the @oidsToRemove hashtable as keys, with their respective values null.
 *
 * @param oidsToRemove	The rowIds of all subtrees of the specified nodes.
 * @param rowIds The rowids of the changed nodes.
 * @param attributes The changed attributes
 */
public void collectChildOids(UlcHashtable oidsToRemove, int[] rowIds, String[] attributes) {
	Vector parents = new Vector();
	if (haveChildrenChanged(rowIds, attributes)) {
		for (int i = 0; i < rowIds.length; i++) {
			getCachedNode(rowIds[i]).collectChildren(oidsToRemove, fOidToNode, false);
		}
	}
}
/**
 * The receiver's root has changed. The receiver must reinitialize
 * itself. It is called when the receiver has been uploaded upon 
 * changes in the receiver's TreeModel, or before the receiver is 
 * saving its state.
 *
 * @see #saveState
 */
protected void contentsChanged() {
	fOidToNode = new UlcHashtable();
}
/**
 * Convert the given node to anything, for uploading it to the UI.
 *
 * @param a		Anything	The object into which my state should be saved.
 * @parem oid	int			The objectId of the node to be uploaded 
 */
protected void convertNodeForUpload(Anything anything, int oid) {
	if (oid != INVALID_OID) {
		int parentOid = getParentOid(oid);
		int ci= getChildIndexForOid(oid);
		if (ci == INVALID_OID || parentOid == INVALID_OID)
			return;
		anything.put("id", parentOid);
		anything.put("cc", getChildCount(parentOid));
		Anything childrenAny = new Anything();
		Anything childAny = new Anything();
		convertNodeForUpload(childAny, oid, false);
		childrenAny.append(childAny);
		anything.put("c", childrenAny);
		anything.put("si", ci);
	}
}
/**
 * Convert the given node to anything, for uploading it to the UI. Upload
 * it's children based on the given additional parameter.
 *
 * @param a		Anything	The object into which my state should be saved.
 * @param oid	int			The objectId of the node to be uploaded
 * @param withChildren		Should the children be uploaded as well
 */
protected void convertNodeForUpload(Anything anything, int oid, boolean withChildren) {
	if (oid != INVALID_OID) {
		markUploaded(oid);
		anything.put("id", oid);
		anything.put("cc", getChildCount(oid));
		if (withChildren) {
			Anything childrenAny = new Anything();
			int[] childOids = getChildOids(oid);
			for (int i = 0; i < (Math.min(getPrefetchCount(), childOids.length)); i++) {
				Anything childAny = new Anything();
				convertNodeForUpload(childAny, childOids[i], false);
				childrenAny.append(childAny);
			}
			anything.put("c", childrenAny);
			anything.put("si", 0);
		}
	}
}
/**
 * Answer the node object corresponding to the given objectId.
 *
 * @param oid		The oid whose node is queried
 */
protected UlcCachedNode getCachedNode(int oid) {
	if (oid == -1)
		return newNode(oid);
	Integer key = new Integer(oid);
	if (!fOidToNode.containsKey(key))
		fOidToNode.put(key, newNode(oid));
	return (UlcCachedNode) fOidToNode.get(key);
}
/**
 * Answer the number of children for the given userObject identified by oid.
 *
 * @param userObject	The domain object whose childCount is required.
 */
protected int getChildCount(int oid) {
	return getHierarchyAdapter().getChildCount(getNode(oid));
}
/**
 * Answer the index of the given child object in the children collection of its parent.
 * Answer -1, if the specified child is not contained in the children of its parent.
 *
 * @param oid	The oid of the domain object whose index in its parent's children is required
 */
protected int getChildIndexForOid(int oid) {
	Object node = getNode(oid);
	Object parent = getHierarchyAdapter().getParent(node);
	Vector children = getHierarchyAdapter().getChildren(parent);
	if (children.contains(node))
		return children.indexOf(node);
	else
		return -1;
}
/**
 * Answer the array of oids for the children of the node identified by oid.
 * The array is expected to contain the child oids in the order used by the receiver.
 *
 * @param oid		The oid of the node queried
 */
protected int[] getChildOids(int oid) {
	if (oid == INVALID_OID)
		return new int[0];
	UlcCachedNode node = getCachedNode(oid);
	int childCount = getChildCount(oid);
	if (!node.hasAllChildOids(childCount)) {
		Vector children = getHierarchyAdapter().getChildren(getNode(oid));
		int[] answer = new int[children.size()];
		for (int i = 0; i < children.size(); i++) {
			answer[i] = internalGetModel().getRowIdFor(children.elementAt(i));
		}
		node.setChildOids(answer);
	}
	return node.getChildOids();
}
/**
 * Get the data from the receiver based on the incoming
 * args, and send the reply to the UI.
 * The actual data is accessed by the receiver from it's hierarchical
 * structure, as well as from it's model.
 *
 * @see #convertNodeForUpload
 *
 * @param args		The <code>Anything</code> specifying the
 *					data required by the UI.
 */
protected void getData(Anything args) {
	Anything attrAny = new Anything();
	Vector requestedOids = new Vector();
	for (int i = 0; i < args.size(); i++) {
		Anything rangeAny = args.get(i);
		if (rangeAny != null && !rangeAny.isNull()) {
			int parentId = rangeAny.get("p", -1);
			int[] childIds = getChildOids(parentId);
			int from = rangeAny.get("if", -1);
			int to = rangeAny.get("it", -1);
			Anything children = new Anything();
			if (from > -1 && to > -1) {
				for (int ci = from; ci < (Math.min(to + 1, childIds.length)); ci++) {
					Anything childAny = new Anything();
					convertNodeForUpload(childAny, childIds[ci], false);
					children.append(childAny);
					requestedOids.addElement(new Integer(childIds[ci]));
				}
			}
			rangeAny.put("c", children);
		}
	}
	attrAny.put("rowids", requestedOids);
	Anything attrVectorAny = new Anything();
	attrVectorAny.append(attrAny);
	getModel().getData(attrVectorAny);
	args.put("data", attrVectorAny);
	args.put("an", attrAny.get("an"));
	sendUI("setData", args);
}
/**
 * Answer the object that adapts the receiver's hierarchy
 *
 */
abstract protected IHierarchyAdapter getHierarchyAdapter();
/**
 * Return the index for the rowId specified. Answer -1 if the oid was not found.
 *
 * @param oid	int	The rowId whose index is required.
 */
public int getIndexForOid(int oid) {
	return internalGetModel().getIndexForOid(oid);
}
/**
 * Answer the node (the user object) identified by oid.
 *
 * @param oid		the oid whose node is required
 */
protected Object getNode(int oid) {
	return getModel().getRow(oid);
}
/**
 * Return the object id for the index or <code>INVALID_OID</code> if not found
 *
 * @param index	int	The index for which the oid is required
 */
public int getOidForIndex(int index) {
	return internalGetModel().getOidForIndex(index);
}
/**
 * Answer the row identifier of the parent of the node identified by oid.
 * Answer <code>INVALID_OID</code> if the specified oid is that of the receiver's root.
 *
 * @param oid		the oid of the node queried
 */
protected int getParentOid(int oid) {
	UlcCachedNode node = getCachedNode(oid);
	if (node.getParentOid() == INVALID_OID) {
		Object parent = getHierarchyAdapter().getParent(getNode(oid));
		int parentOid = internalGetModel().getRowIdFor(parent);
		node.setParentOid(parentOid);
	}
	return node.getParentOid();
}
/**
 * Answer the path from the given oid upto it's last uploaded ancestor.
 *
 * @param	oid		int		The oid corresponding to the node whose path is required
 */
protected int[] getPath(int oid) {
	Vector path = new Vector();
	boolean isUploaded = false;
	int lastOid = oid;
	while (!isUploaded) {	//stop as soon as an uploaded ancestor is found. 
		path.addElement(new Integer(lastOid));
		int parentOid = getParentOid(lastOid);
		isUploaded = isUploaded(parentOid);
		lastOid = parentOid;
	}
	int[] answer = new int[path.size()];
	int count = -1;
	for (int i = path.size() - 1; i >= 0; i--) {
		answer[++count] = ((Integer) path.elementAt(i)).intValue();
	}
	return answer;
}
/**
 * Answer the root object for the receiver's hierarchy
 */
public java.lang.Object getRoot() {
	if (fRoot == null)
		fRoot = getHierarchyAdapter().getDefaultRoot();
	return fRoot;
}
/**
 * Answer the oid of the receiver's root node.
 */
protected int getRootOid() {
	try { // the model may not have been set
		return internalGetModel().getRowIdFor(getRoot());
	}
	catch (Throwable t) {
		return INVALID_OID;
	}
}
/**
 * Answer the total number of rows in the receiver
 */
public int getRowCount() {
	if (getRootOid() == INVALID_OID)
		return 0;
	else
		return 1 + getChildCount(getRootOid());
}
/**
 * Answer the vector of objects identified by the objectIdentifiers.
 *
 * @param args	oids	Contains the objectIdentifiers to locate the
 * 							required objects.
 *
 */
protected Vector getSelectedNodesFrom(Vector oids) {
	Vector nodes = new Vector(oids.size());
	for (int i = 0; i < oids.size(); i++) {
		int nextOid = ((Integer) oids.elementAt(i)).intValue();
		Object nextNode = getModel().getRow(nextOid);
		nodes.addElement(nextNode);
	}
	return nodes;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("getPaths")) {
		sendPaths(args.get("nodeIds"));
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Answer true if the attributes provided include the one used to determine the parent
 * of a given node. Default is to answer false.
 *
 * @param rowIds The rowids of the changed nodes.
 * @param attributes The changed attributes
 */
protected boolean hasParentChanged(int[] rowIds, String[] attributes) {
	for (int i = 0; i < attributes.length; i++) {
		if (getHierarchyAdapter().isParentAttribute(attributes[i]))
			return true;
	}
	return false;
}
/**
 * Answer true if the attributes provided include the one used to determine the children
 * of a given node. Default is to answer false.
 *
 * @param rowIds The rowids of the changed nodes.
 * @param attributes The changed attributes
 */
protected boolean haveChildrenChanged(int[] rowIds, String[] attributes) {
	for (int i = 0; i < attributes.length; i++) {
		if (getHierarchyAdapter().isChildrenAttribute(attributes[i]))
			return true;
	}
	return false;
}
/**
 * Answer the array of row identifiers that should be uploaded when the receiver 
 * is uploaded for the first time.
 */
public int[] initialOids() {
	int[] childOids = getChildOids(getRootOid());
	int prefetchCount = Math.min(getPrefetchCount(), childOids.length);
	int[] oids = new int[prefetchCount + 1];
	oids[0] = getRootOid();
	for (int i = 0; i < prefetchCount; i++)
		oids[i + 1] = childOids[i];
	return oids;
}
/**
 * Send all the nodes to be inserted, right away to the UI. This means that
 * all the oids of the inserted nodes are being sent.
 * => time taken to do this operation increases with number of children being
 *		inserted.
 *
 * @param oids	Vector	The new nodes which have been inserted
 */
public void insertRowsImmediately(Vector oids) {
	Anything args = new Anything();
	for (int i = 0; i < oids.size(); i++) {
		Anything nodeAny = new Anything();
		convertNodeForUpload(nodeAny, ((Integer) oids.elementAt(i)).intValue());
		args.append(nodeAny);
	}
	sendUI("insertRows", args);
}
/**
 * For all the nodes to be inserted to the UI, send just the new childCount. 
 * => time taken to do this operation is independent number of children being
 *		inserted.
 *
 * @param parents	Hashtable	The parents into which new node(s) have been inserted
 */
public void insertRowsLazily(Hashtable parents) {
	Anything args = new Anything();
	Enumeration keys = parents.keys();
	while (keys.hasMoreElements()) {
		Anything anything = new Anything();
		int parentOid = ((Integer) keys.nextElement()).intValue();
		anything.put("id", parentOid);
		anything.put("cc", getChildCount(parentOid));
		Vector indices = (Vector) parents.get(new Integer(parentOid));
		Vector sortedIndices = new UlcSorter().sortIntegerVector(indices);
		anything.put("indices", sortedIndices);
		args.append(anything);
	}
	sendUI("insertRows", args);
}
/**
 * Answer the receiver's tableModel
 */
protected ULCAbstractTableModel internalGetModel() {
	return fModel;
}
/**
 * Answer boolean whether the given node is already uploaded to UI
 *
 * @param	int	oid		The objectId of the node which is checked for upload.
 */
protected boolean isUploaded(int oid) {
	return getCachedNode(oid).isUploaded();
}
/**
 * Set the given node as already uploaded to UI
 *
 * @param	int	oid		The objectId of the node which is set as upload.
 */
protected void markUploaded(int oid) {
	if (oid != INVALID_OID) {
		getCachedNode(oid).setUploaded(true);
	}
}
/**
 * Create and answer a new cachedNode.
 * This is an internal method.
 *
 * @param	int	oid		The objectId of the node which to be created
 */
protected UlcCachedNode newNode(int oid) {
	return new UlcCachedNode(oid);
}
/**
 * The entire contents of the receiver's model has changed. 
 * Update the receiver and its UI proxy accordingly.
 */
public void notifyContentsChanged() {
	if (shouldProcessNotification()) {
		setRootAfterContentsChanged();
		super.notifyContentsChanged();
	}
}
/**
 * The rows specified by rowIds have been added to the receiver's model. 
 * Update the receiver and its UI proxy accordingly.
 *
 * @param rowIds	int[]	The set of rowIds that have been added to my model
 */
public void notifyRowsAdded(int[] rowIds) {
	if (shouldProcessNotification()) {
		Vector uploadable = new Vector(rowIds.length);
		Hashtable parents = new Hashtable(); //Hold parent -> indices where children are inserted
		for (int i = 0; i < rowIds.length; i++) {
			int parentOid = getParentOid(rowIds[i]);
			if (isUploaded(parentOid))
				uploadable.addElement(new Integer(rowIds[i]));
			getCachedNode(rowIds[i]).setParentOid(parentOid);
			int insertionIndex = getChildIndexForOid(rowIds[i]);
			getCachedNode(parentOid).insertOid(rowIds[i], insertionIndex);
			Integer parentOidInt = new Integer(parentOid);
			Object indices = parents.get(parentOidInt);
			if (indices == null)
				indices = new Vector();
			((Vector) indices).addElement(new Integer(insertionIndex));
			parents.put(parentOidInt, indices);
		}
		insertRowsLazily(parents);
		//insertRowsImmediately(uploadable);
	}
}
/**
 * The specified rows have changed. If the attributes changed includes the
 * parent attribute of the receiver's nodes, reload the parent nodes. If the
 * include the children attribute of a node, reload the nodes.
 *
 * @param rowIds The rowids of the changed nodes.
 * @param attributes The changed attributes
 */
public void notifyRowsChanged(int[] rowIds, String[] attributes) {
	if (shouldProcessNotification()) {
		Vector parents = new Vector();
		if (hasParentChanged(rowIds, attributes)) {
			for (int i = 0; i < rowIds.length; i++) {
				UlcCachedNode node = getCachedNode(rowIds[i]);
				if (node.getParentOid() != INVALID_OID) {
					Integer oldParent = new Integer(node.getParentOid());
					if (!parents.contains(oldParent))
						parents.addElement(oldParent);
					node.removeFromParent(fOidToNode);
				}
				Integer newParent = new Integer(getParentOid(rowIds[i]));
				if (newParent.intValue() == INVALID_OID) {
					node.removeWithChildren(fOidToNode);
				}
				else {
					getCachedNode(newParent.intValue()).insertOid(rowIds[i], getChildIndexForOid(rowIds[i]));
					if (!parents.contains(newParent))
						parents.addElement(newParent);
				}
			}
		}
		if (haveChildrenChanged(rowIds, attributes)) {
			for (int i = 0; i < rowIds.length; i++) {
				UlcCachedNode node = getCachedNode(rowIds[i]);
				node.removeChildren(fOidToNode);
				if (!parents.contains(new Integer(rowIds[i])))
					parents.addElement(new Integer(rowIds[i]));
			}
		}
		if (parents.size() > 0) {
			int[] parentOids = new int[parents.size()];
			for (int i = 0; i < parents.size(); i++)
				parentOids[i] = ((Integer) parents.elementAt(i)).intValue();
			refreshChildren(parentOids);
		}
	}
}
/**
 * The rows specified by rowIds have been removed from the receiver's model. 
 * Update the receiver's oid map, but do NOT update the UI, as this is done 
 * internally via the update of the UI ItemCache.
 *
 * @param rowIds		The row identifiers removed from my model
 */
public void notifyRowsRemoved(int[] rowIds) {
	if (shouldProcessNotification()) {
		for (int i = 0; i < rowIds.length; i++) {
			getCachedNode(rowIds[i]).removeFrom(fOidToNode);
		}
	}
	if (fOidToNode.size() == 0)
		setRoot(null);
}
/**
 * Refresh the children of the nodes of identified by rowIds. Update the UI
 * with the new data.
 *
 * @param rowids	int[]	The set of node identifiers whose children need refresh
 */
public void refreshChildren(int[] rowIds) {
	Anything args = new Anything();
	for (int i = 0; i < rowIds.length; i++) {
		if (isUploaded(rowIds[i])) {
			Anything rowAny = new Anything();
			convertNodeForUpload(rowAny, rowIds[i], false);
			args.append(rowAny);
		}
	}
	sendUI("reload", args);
}
/**
 * Store the data available with the initial upload to anything.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveInitialData(Anything anything) {
	Enumeration nodes = fOidToNode.elements();
	while (nodes.hasMoreElements()) {
		UlcCachedNode node = (UlcCachedNode) nodes.nextElement();
		node.setUploaded(false);
	}
	convertNodeForUpload(anything, getRootOid(), true);
}
/**
 * Send the changed contents of the receiver to the UI.
 */
protected void sendContentsChanged() {
	Anything a = new Anything();
	a.put("rows", getRowCount());
	saveInitialData(a);
	sendUI("contentsChanged", a);
}
/**
 * Upload the given oids inside the Anything, to the UI.
 *
 * @param args	Anything	The object in which my paths will be uploaded to UI.
 */
public void sendPaths(Anything args) {
	Vector nodes = getSelectedNodesFrom(args.toCollection());
	sendPaths(nodes);
}
/**
 * Upload the given nodes, along with their complete paths,
 * to the UI. The path is calculated only upto the last
 * already-uploaded ancestor of the receiver.
 *
 * @param nodes		Vector		The nodes whose paths need to be
 *								sent up to the UI.
 */
public void sendPaths(Vector nodes) {
	Anything args= buildPathsFor(nodes);
	if (args != null)
		sendUI("setPaths", args);
}
/**
 * Set the given object as the new root for the receiver
 *
 * @param 	newRoot 	Object	The new root for the receiver's hierarchy
 */
public void setRoot(Object newRoot) {
	fRoot = newRoot;
	sendContentsChanged();
}
/**
 * The receiver's model has changed its contents. We assume the receiver's root to be still valid 
 * and ignore this message.
 * If the root is no longer valid, the application must update it accordingly.
 */
protected void setRootAfterContentsChanged() {
}
/**
 * Return the fully qualified class path to my proxy class in the UI.
 */
public String typeString() {
	return "HierarchicalItemList";
}
}
