package com.ibm.ulc.application;

/*
 * Copyright (c) 1999 Object Technology International Inc.
 */
import com.ibm.ulc.application.*;
import com.ibm.ulc.util.*;
import java.io.*;
import java.awt.*;
import com.ibm.ulc.comm.*;
/**
 * This widget allows in conjunction with a <code>ULCGraphics</code> 
 * the ability to do simple graphic operations.
 * @see ULCGraphics
 */
public class ULCGraphicsPane extends ULCComponent {
	private ByteArrayOutputStream fGraphicsBytes = null;
	private DataOutputStream fGraphicsBuffer = null;
	private Dimension fSize = null;
	private IGraphicsProducer fGraphicsProducer = null;
/**
 * Creates a GraphicsPane with a default size of 100,100
 */
public ULCGraphicsPane() {
	this(null, 100, 100);
}
/**
 * Creates a GraphicsPane with the specified size and backgroundColor
 *
 * @param width 			The <code>int</code> initial width of this pane.
 * @param height 			The <code>int</code> initial height of this pane.
 * @param backgroundColor 	The <code>Color</code> of the background of this pane.
 */
public ULCGraphicsPane(int width, int height, Color backgroundColor) {
	this(null, width, height, backgroundColor);
}
/**
 * Creates a GraphicsPane with the specified size and <code>IGraphicsproducer</code>
 * Every paint request will be forwarded to the IGraphicsProducer with the active ULCGraphics.
 *
 * @param graphicsProducer 	The <code>IGraphicsProducer</code> which will perform the actual painting.
 * @param width 			The <code>int</code> initial width of this pane.
 * @param height 			The <code>int</code> initial height of this pane.
 *
 * @see IGraphicsProducer#paint
 */
public ULCGraphicsPane(IGraphicsProducer graphicsProducer, int width, int height) {
	this(graphicsProducer, width, height, null);
}
/**
 * Creates a GraphicsPane with the specified size and <code>IGraphicsproducer</code>
 * Every paint request will be forwarded to the IGraphicsProducer with the active ULCGraphics.
 *
 * @param graphicsProducer 	The <code>IGraphicsProducer</code> which will perform the actual painting.
 * @param width 			The <code>int</code> initial width of this pane.
 * @param height 			The <code>int</code> initial height of this pane.
 * @param backgroundColor 	The <code>Color</code> of the background of this pane.
 *
 * @see IGraphicsProducer#paint
 */
public ULCGraphicsPane(IGraphicsProducer graphicsProducer, int width, int height, Color backgroundColor) {
	fSize = new Dimension(width, height);
	fGraphicsProducer = graphicsProducer;
	fBackColor = backgroundColor;
}
/**
 * Return the <code>IGraphicsProducer</code> which will perform the actual painting within this pane.
 *
 * @see IGraphicsProducer#paint
 */
public IGraphicsProducer getGraphicsProducer() {
	return fGraphicsProducer;
}
/**
 * Return the current size of this pane in the UI.
 *
 */
public Dimension getSize() {
	return fSize;
}
/**
 * The UI has sent an error message caused by a print request. Handle the error.
 *
 * @param args		Anything		The arguments associated with this request.
 */
protected void handlePrintError(Anything args) {
	//System.out.println("ULCGraphicsPane.handlePrintError" + args.asString("printError"));
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public synchronized void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("resized")) {
		fSize = new Dimension(args.get("w", 100), args.get("h", 100));
		paint();
		return;
	}
	if (request.equals("zoom")) {
		if (fGraphicsProducer instanceof IScaleableGraphicsProducer) {
			((IScaleableGraphicsProducer) fGraphicsProducer).zoom(args.get("x", 0), args.get("y", 0), args.get("w", 100), args.get("h", 100));
		}
		return;
	}
	if (request.equals("print")) {
		internalPrint(args);
		return;
	}
	if (request.equals("printError")) {
		handlePrintError(args);
		return;
	}
}
/**
 * Send the contents of this graphics pane to the printer.
 * This will cause the user to be prompted for the printer and the contents
 * of this graphics pane will be output to the printer.
 */
private void internalPrint(Anything printerInfo) {
	int w = printerInfo.get("w", 100);
	int h = printerInfo.get("h", 100);
	int resolution = printerInfo.get("r", 0);
	String printJobName = printerInfo.get("job", "ULC Print Job");
	try {
		fGraphicsBytes = new ByteArrayOutputStream();
		fGraphicsBuffer = new DataOutputStream(fGraphicsBytes);
		fGraphicsProducer.paint(new ULCGraphics(fGraphicsBuffer), w, h);
		fGraphicsBuffer.flush();
	} catch (Exception e) {
		System.out.println("ULCGraphicsPane internalPrint: " + e);
		e.printStackTrace();
	}
	Anything a = new Anything();
	a.put("data", fGraphicsBytes.toByteArray());
	a.put("job", printJobName);
	sendUI("print", a);
}
/**
 * This method is called whenever a resize occurs in the UI.
 * A new <code>ULCGraphics</code> is created and passed to the <code>IGraphicsProducer</code>.
 * The resulting bytes in the byte stream of the ULCGraphics is sent to the UI to be displayed.
 *
 * @see IGraphicsProducer#paint
 */
public void paint() {
	if (fSize == null)
		return;
	try {
		fGraphicsBytes = new ByteArrayOutputStream();
		fGraphicsBuffer = new DataOutputStream(fGraphicsBytes);
		fGraphicsProducer.paint(new ULCGraphics(fGraphicsBuffer), fSize.width, fSize.height);
		fGraphicsBuffer.flush();
	} catch (Exception e) {
		System.out.println("ULCGraphicsPane.paint: " + e);
		e.printStackTrace();
	}
	sendBytes(fGraphicsBytes.toByteArray());
}
/**
 * Call this method to force a resize/paint of this widget in the UI.
 */
public void repaint() {
	sendUI("repaint", new Anything());
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
public void saveState(Anything a) {
	super.saveState(a);
	a.put("w", fSize.width);
	a.put("h", fSize.height);
	a.put("zoomable", fGraphicsProducer instanceof IScaleableGraphicsProducer);
}
/**
 * The contents of the byteArray are sent to the UI to be displayed.
 *
 * @param byteArray	The byteArray that has been created using a ULCGraphics
 */
public void sendBytes(byte[] byteArray) {
	Anything a = new Anything();
	a.put("w", fSize.width);
	a.put("h", fSize.height);
	a.put("data", byteArray);
	sendUI("paint", a);
}
/**
 * Set the <code>IGraphicsProducer</code> which will perform the actual painting within this pane.
 *
 * @param graphicsProducer 	The <code>IGraphicsProducer</code> which will perform the actual painting.
 * @see IGraphicsProducer#paint
 */
public void setGraphicsProducer(IGraphicsProducer graphicsProducer) {
	fGraphicsProducer = graphicsProducer;
}
/**
 * Set the initial size of this widget.
 *
 * @param size 	The <code>Dimension</code> initial size.
 */
public void setSize(Dimension size) {
	fSize = size;
}
/**
 * Request a print job to be created the UI will turn around and send the job information back at
 * which time the contents of this graphics pane can be rendered to the size of the printer.
 *
 * @param printJobName	A <code>String</code> to identify this job
 */
public void startPrintJob(String printJobName) {
	Anything a = new Anything();
	a.put("job", printJobName);
	sendUI("startPrintJob", a);
}
/**
 * returns the fully qualified class name of my UI class
 */
public String typeString() {
	return "com.ibm.ulc.ui.UIGraphicsPane";
}
}
