package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import com.ibm.ulc.util.Anything;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.base.IDefaults;
import java.awt.Color;
/**
 * A ULCTextField allows the editing of a single line or multiple lines of text (Strings).
 * The current value is either determined by the fValue field of this widget or
 * by an attribute of a ULCFormModel.
 * A DataType object can be specified to allow for editing of non String value objects.
 * In this case the ULCFields fValue object is converted into a String and back into an object
 * by means of the DataType object.
 */
public class ULCField extends ULCFormComponent implements IEnabler {
	/**
	 * The last valid input in this field.
	 * @serial
	 */
	protected Object fValue = "";
	/**
	 * The default number of columns and rows for this field.
	 * @serial	 
	 */
	protected int fCols = 10, fRows = 1;
	/**
	 * If fPassword is true all input is masked out with the password character.
	 * @serial	 
	 */
	protected boolean fPassword = false;
	/**
	 * If fEditable is false then input into this field is disabled.
	 * @serial	 
	 */
	protected boolean fEditable = true;
	/**
	 * If fLineWrap is false then line wrapping is disabled
	 * @serial	 
	 */
	protected boolean fLineWrap = true;
	/**
	 * If fWordWrap is false then word wrapping takes place at character boundaries
	 * @serial	 
	 */
	protected boolean fWordWrap = true;
	/**
	 * The optional dataType object.
	 * If this is set then all input into this field is filtered/converted by the dataType object.
	 * @serial	 
	 */
	protected IDataType fDataType = null;
	/**
	 * The current selection within this field.
	 * @serial	 
	 */
	private Anything fSelection = null;
	/**
	 * The current horizontal text alignment
	 * @serial	 
	 */
	protected int fHorizontalAlignment = BOX_LEFT;
	/**
	 * The selection foreground color for this widget.
	 * @serial	 
	 */
	protected Color fSelectionForeColor;
	/**
	 * The selection background color for this widget.
	 * @serial	 
	 */
	protected Color fSelectionBackColor;
	/**
	 * Any valid ULCTrigger object which will be triggered in the UI when 
	 * this widgets actionPerformed method is called
	 * @serial	 
	 */
	protected ITrigger fTrigger = null;
/**
 * Constructs an empty ULCField with the specified DataType. 
 *
 */
public ULCField() {
	fValue = null;
	fCols = 10;
}
/**
 * Constructs a new empty ULCField with the specified number of columns.
 *
 * @param columns The minimum number of columns for this field.
 */
public ULCField(int columns) {
	fCols = columns;
}
/**
 * Constructs a new empty ULCField with the specified number of rows and columns. 
 *
 * @param rows The minimum number of rows for this field.
 * @param columns The minimum number of columns for this field.
 */
public ULCField(int rows, int columns) {
	fRows = rows;
	fCols = columns;
}
/**
 * Constructs an empty ULCField with the specified DataType. 
 *
 * @param 	dataType
 *			The object that will perform validation/conversion on all input within this field.
 */
public ULCField(IDataType datatype) {
	fValue = null;
	fCols = 10;
	fDataType = datatype;
}
/**
 * Constructs a new ULCField with the specified DataType and initial value. 
 *
 * @param 	dataType
 *			The object that will perform validation/conversion on all input within this field.
 * @param 	value	
 *			The initial value for this field.
 */
public ULCField(IDataType datatype, Object value) {
	fValue = value;
	fCols = 10;
	fDataType = datatype;
}
/**
 * Constructs a new ULCField with the specified DataType, initial value, and number of columns. 
 *
 * @param 	dataType
 *			The object that will perform validation/conversion on all input within this field.
 * @param 	value	
 *			The initial value for this field.
 * @param 	cols	
 *			The minimum number of columns for this field.
 */
public ULCField(IDataType datatype, Object value, int cols) {
	fValue = value;
	fCols = cols;
	fDataType = datatype;
}
/**
 * Constructs a ULCField which tracks the state of the specified attribute of the given FormModel.
 *
 * @param model The <code>ULCFormModel</code> whose state the field should track.
 * @param key 	The attribute key of the <code>ULCFormModel</code> whose value the field should track.
 * @param cols 	Mimimum number of columns.
 */
public ULCField(IForm model, String key, int cols) {
	super(model, key);
	fCols = cols;
}
/**
 * Constructs a ULCField which tracks the state of the specified attribute of the given FormModel.
 *
 * @param model The <code>ULCFormModel</code> whose state the field should track.
 * @param key 	The attribute key of the <code>ULCFormModel</code> whose value the field should track.
 * @param rows 	Mimimum number of rows.
 * @param cols 	Mimimum number of columns.
 */
public ULCField(IForm model, String key, int rows, int cols) {
	super(model, key);
	fRows = rows;
	fCols = cols;
}
/**
 * Constructs a ULCField which tracks the state of the specified attribute of the given FormModel.
 * The value of the attribute is formatted by using the given DataType.
 *
 * @param 	model	The <code>ULCFormModel</code> whose state the field should track.
 * @param 	key 	The attribute key of the <code>ULCFormModel</code> whose value the field should track.
 * @param 	dataType
 *			The object that will perform validation/conversion on all input within this field.
 */
public ULCField(IForm model, String key, IDataType dataType) {
	super(model, key);
	fDataType = dataType;
}
/**
 * Constructs a ULCField which tracks the state of the specified attribute of the given FormModel.
 * The value of the attribute is formatted by using the given DataType.
 *
 * @param 	model 	The <code>ULCFormModel</code> whose state the field should track.
 * @param 	key 	The attribute key of the <code>ULCFormModel</code> whose value the field should track.
 * @param 	dataType
 *			The object that will perform validation/conversion on all input within this field.
 * @param 	cols	The minimum number of cols for this field.
 */
public ULCField(IForm model, String key, IDataType dataType, int cols) {
	super(model, key);
	fDataType = dataType;
	fCols = cols;
}
/**
 * Constructs a new ULCField with the initial text, and number of columns. 
 *
 * @param 	text	The initial contents of this field.
 * @param 	columns	The minimum number of columns.
 */
public ULCField(String text, int columns) {
	fValue = text;
	fCols = columns;
}
/**
 * Constructs a new ULCField with the specified number of rows and columns, and the given initial text. 
 *
 * @param 	text	The initial contents of this field.
 * @param 	rows	The minimum number of rows.
 * @param 	columns	The minimum number of columns.
 */
public ULCField(String text, int rows, int columns) {
	fValue = text;
	fRows = rows;
	fCols = columns;
}
/**
 * Constructs a new ULCField with the specified number of columns, the given initial text.
 * The text is formatted by using the given DataType. 
 *
 * @param 	text	The initial contents of this field.
 * @param 	columns	The minimum number of columns.
 * @param 	dataType
 *			The object that will perform validation/conversion on all input within this field.
 */
public ULCField(String text, int columns, IDataType datatype) {
	fValue = text;
	fCols = columns;
	fDataType = datatype;
}
/**
 * Registers the given listener to begin receiving notifications
 * when the Field signals an action (Usually the Enter key). 
 *
 * @param listener	The object interested in my actionEvents.
 */
public void addActionListener(IActionListener listener) {
	enableOptionalEvent("action");
	internalAddListener("action", listener);
}
/**
 * Registers the given listener to begin receiving notifications
 * when changes are made to the field.
 *
 * @param listener	The object interested in my changes.
 */
public void addValueChangedListener(IValueChangedListener listener) {
	internalAddListener("value", listener);
}
/**
 * Copy the current selection into the Clipboard
 */
public void copy() {
	sendUI("copy");
}
/**
 * Cut the current selection into the Clipboard
 */
public void cut() {
	sendUI("cut");
}
/**
 * Returns the formatter and validator object for this field.
 *
 * @return 	dataType
 *			The object that will perform validation/conversion on all input within this field.
 */
public IDataType getDataType() {
	return fDataType;
}
/**
 * Returns the alignment of the label's contents along the X axis.
 * <p>
 *
 * @param alignment  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>BOX_LEFT</code> (the default for text-only labels),
 *           <code>BOX_CENTER</code> (the default for image-only labels), or
 *           <code>BOX_RIGHT</code>.
 *
 * @see IDefaults 
 */
public int getHorizontalAlignment() {
	return fHorizontalAlignment;
}
/**
 * Returns the line-wrapping policy of the text area.  If set
 * to true the lines will be wrapped if they are too long
 * to fit within the allocated width.  If set to false,
 * the lines will always be unwrapped. 
 *
 * @return wrap if true indicates if lines should be wrapped.
 */
public boolean getLineWrap() {
	return fLineWrap;
}
/**
 * Returns the selection background color for this widget.
 *
 * @return The <code>Color</code> that should be used within this widget
 */
public Color getSelectionBackground() {
	return fSelectionBackColor;
}
/**
 * Returns the selection foreground color for this widget.
 *
 * @return The <code>Color</code> that should be used within this widget
 */
public Color getSelectionForeground() {
	return fSelectionForeColor;
}
/**
 * Convenience method to return the value of the field as a String.
 */
public String getText() {
	return getValue().toString();
}
/**
 * Returns the configured ULCTrigger object which will be triggered in the UI when 
 * this widget's actionPerformed method is called.
 *
 * @return ITrigger	 
 */
public ITrigger getTrigger() {
	return fTrigger;
}
/**
 * Returns the value of this field.
 */
public Object getValue() {
	return fValue;
}
/**
 * Returns the word-wrapping policy of the text area.  If set
 * to true the lines will be wrapped at word breaks if they 
 * are too long to fit within the allocated width.
 * If set to false, the lines will be wrapped at character boundaries
 * 
 * @see #setLineWrap
 * @param wrap indicates if lines should be wrapped at word boundaries.
 */
public boolean getWordWrap() {
	return fWordWrap;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("value")) {
			fValue = convert(args.get("value"));
			distributeToListeners("value", new ULCValueChangedEvent(this, fValue));
		} else
			if (type.equals("action")) {
				distributeToListeners("action", new ULCActionEvent(this, "action"));
			}
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Determines whether the ULCField is editable.
 *
 * @return  if true allows editing within this field
 */
public boolean isEditable() {
	return fEditable;
}
/**
 * Returns the mode of the Field.
 * If <code>password</code> is true then all input within this field will be masked
 * out with the password mask character.
 *
 * @return passsword If true enables masking of all input using the password mask.
 */
public boolean isPassword() {
	return fPassword;
}
/**
 * Pastes the current selection into the Clipboard
 */
public void paste() {
	sendUI("paste");
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive action events.
 *
 * @param listener	The widget that was registered to receive my actionEvents.
 */
public void removeActionListener(IActionListener listener) {
	disableOptionalEvent("action");
	internalRemoveListener("action", listener);
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive value changed events.
 *
 * @param listener	The object that was interested in my changes.
 */
public void removeValueChangedListener(IValueChangedListener listener) {
	internalRemoveListener("value", listener);
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fFormModel == null && fValue != null)
		a.put("value", convert(fContext, fValue));
	if (fRows > 1)
		a.put("rows", fRows);
	if (fCols != 10)
		a.put("cols", fCols);
	if (fPassword)
		a.put("password", fPassword);
	if (!fEditable)
		a.put("editable", fEditable);
	if (!fLineWrap)
		a.put("linewrap", fLineWrap);
	if (!fWordWrap)
		a.put("wordwrap", fWordWrap);
	if (fDataType != null)
		a.put("dataType", fDataType.getRef(fContext));
	if (fSelection != null)
		a.put("selection", fSelection);
	if (fHorizontalAlignment != BOX_LEFT)
		a.put("ha", new Anything(fHorizontalAlignment));
	if (fSelectionForeColor != null)
		a.put("ofc", internalConvertColor(fSelectionForeColor));
	if (fSelectionBackColor != null)
		a.put("obc", internalConvertColor(fSelectionBackColor));
	if (fTrigger != null)
		a.put("trigger", ((ULCProxy) fTrigger).getRef(fContext));
}
/**
 * Convenience method for selecting all characters of field.
 */
public void selectAll() {
	setSelection(0, 0, -1);
}
/**
 * Sets the formatter and validator object for this field.
 *
 * @param 	dataType
 *			The object that will perform validation/conversion on all input within this field.
 */
public void setDataType(IDataType datatype) {
	fDataType = datatype;
	sendUI("setDataType", fDataType);
}
/**
 * Determines whether the ULCField is editable.
 *
 * @param editable if true allows editing within this field
 */
public void setEditable(boolean editable) {
	if (fEditable != editable) {
		fEditable = editable;
		sendUI("setEditable", new Anything(fEditable));
	}
}
/**
 * Sets the alignment of the label's contents along the X axis.
 * <p>
 *
 * @param alignment  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>BOX_LEFT</code> (the default for text-only labels),
 *           <code>BOX_CENTER</code> (the default for image-only labels), or
 *           <code>BOX_RIGHT</code>.
 *
 * @see IDefaults 
 */
public void setHorizontalAlignment(int horizontalAlignment) {
	if (horizontalAlignment != fHorizontalAlignment) {
		fHorizontalAlignment = horizontalAlignment;
		sendUI("setHorizontalAlignment", new Anything(fHorizontalAlignment));
	}
}
/**
 * Sets the line-wrapping policy of the text area.  If set
 * to true the lines will be wrapped if they are too long
 * to fit within the allocated width.  If set to false,
 * the lines will always be unwrapped. 
 *
 * @see #setWordWrap
 * @param wrap indicates if lines should be wrapped.
 */
public void setLineWrap(boolean wrap) {
	if ((fLineWrap != wrap) && (fRows > 1)) {
		fLineWrap = wrap;
		sendUI("setLineWrap", new Anything(fLineWrap));
	}
}
/**
 * Sets the mode of the Field.
 * If <code>password</code> is true then all input within this field will be masked
 * out with the password mask character.
 * Note calling this method after the widget has been uploaded has no effect.
 *
 * @param passsword If true enables masking of all input using the password mask.
 */
public void setPassword(boolean password) {
	fPassword = password;
}
/**
 * Selects <code>length</code> characters of the text starting at the given offset.
 * The <code>type</code> determines the origin of the selection and <code> offset</code>
 * determines the relative position from the origin.
 * <pre>
 * 	type= 0:	from start
 * 	type= 1:	from current position
 * 	type= 2:	from end
 * </pre>
 */
public void setSelection(int type, int offset, int length) {
	fSelection = new Anything();
	fSelection.put("t", type);
	fSelection.put("o", offset);
	fSelection.put("l", length);
	sendUI("setSelection", fSelection);
}
/**
 * Sets the selection background color for this widget.
 *
 * @param color The <code>Color</code> that should be used within this widget
 */
public void setSelectionBackground(Color color) {
	fSelectionBackColor = color;
	if (color == null)
		sendUI("setSelectionBackground", new Anything());
	else
		sendUI("setSelectionBackground", internalConvertColor(fSelectionBackColor));
}
/**
 * Sets the selection foreground color for this widget.
 *
 * @param color The <code>Color</code> that should be used within this widget
 */
public void setSelectionForeground(Color color) {
	fSelectionForeColor = color;
	if (color == null)
		sendUI("setSelectionForeground", new Anything());
	else
		sendUI("setSelectionForeground", internalConvertColor(fSelectionForeColor));
}
/**
 * Convenience method to set the value of the field to a String.
 */
public void setText(String text) {
	setValue(text);
}
/**
 * Sets the configured ULCTrigger which will be triggered in the UI when 
 * this widgets actionPerformed method is called
 *
 * @param ITrigger	 
 */
public void setTrigger(ITrigger trigger) {
	if (trigger != fTrigger) {
		fTrigger = trigger;
		sendUI("setTrigger", (ULCTrigger)fTrigger);
	}
}
/**
 * Sets the value shown in this field. If the field is FormModel based this call is ignored.
 *
 * @param value The object to be displayyed in the field.
 */
public void setValue(Object value) {
	if (fValue == null || !fValue.equals(value)) {
		fValue = value;
		sendUI("setValue", convert(fContext, fValue));
		distributeToListeners("value", new ULCValueChangedEvent(this, fValue));
	}
}
/**
 * Sets the word-wrapping policy of the text area.  If set
 * to true the lines will be wrapped at word breaks if they 
 * are too long to fit within the allocated width.
 * If set to false, the lines will be wrapped at character boundaries
 * 
 * @see #setLineWrap
 * @param wrap indicates if lines should be wrapped at word boundaries.
 */
public void setWordWrap(boolean wrap) {
	if ((fWordWrap != wrap) && (fRows > 1)) {
		fWordWrap = wrap;
		sendUI("setWordWrap", new Anything(fWordWrap));
	}
}
}
