package com.ibm.ulc.application;

import java.util.*;
import com.ibm.ulc.application.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.UlcHashtable;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.base.*;

/**
 * Displays an item list in an icon container.
 * The items must conform to the IContainerItem interface and
 * it is assumed that there are not hundreds of
 * such items because there is no model underneath.
 * If you want to modify items later (e.g. color, enabled) then
 * your item has to be either ULCLabel or a subclass of it.
 */
public class ULCContainer extends com.ibm.ulc.application.ULCComponent implements IDefaults {
	/**
	 * There are 3 ways in which a scroll bar can be displayed.
	 * This integer will represent one of these 3 constants -
	 * (SCROLLBARS_ALWAYS, SCROLLBARS_AS_NEEDED, SCROLLBARS_NEVER)
 	 * @see #getScrollbarDisplayPolicy()
 	 * @serial
	 */
	 private int fScrollbarDisplayPolicy= SCROLLBARS_AS_NEEDED;

	 /**
	 * The items of the container which are passed by the user
	 * must implement the IContainerItem interface.
	 * Internally (i.e. in the fList field) we store a ULCLabel
	 * for each item.
	 * If the type of an element does not conform a <code>ClassCastException</code>
	 * will be thrown
 	 * @serial	 
	 */
	private Vector fItems= new Vector();
	/**
	 * List with the oids of the selected items
	 * @serial	 
	 */
	private Vector fSelectedOids= new Vector();
	/**
	 * <code>UlcHashtable</code> that maps the id
	 *  of the internally used <code>ULCLabel</code> to
	 * actual item
	 * @serial	 
	 */
	private UlcHashtable fIdToItemMapping= new UlcHashtable();
	/**
	 * <code>Vector</code> that stores the selected
	 * items before uploading takes place. Afterwards
	 * this field is never used
	 * @serial	 
	 */
	 private Vector fInitiallySelectedItems= new Vector();
	/**
	 * The initial width of the container
	 * @serial	 
	 */
	private int fInitialWidth= 0;
	/**
	 * The initial height of the container
	 * @serial	 
	 */
	private int fInitialHeight= 0;
	/**
	 * Single or multiple select
	 * @serial	 
	 */
	private boolean fMultiSelectionAllowed= true;
	/**
	 * Auto-arrange icons
	 * @serial	 
	 */
	private boolean fAutoArranging= true;
	/**
	 * Label position relative to icon
	 * @serial	 
	 */
	private int fLabelPosition= BOX_BOTTOM;
	/**
	 * Drag icons allowed or not
	 * @serial	 
	 */
	private boolean fDragIconsAllowed= false;
	/**
	 * Horizontal gap between icons
	 * @serial	 
	 */
	private int fHorizontalGap= 5;
	/**
	 * Vertical gap between icons
	 * @serial	 
	 */
	private int fVerticalGap= 5;
/**
 * Constructs a new ULCContainer which displays
 * a list of <code>IContainerItem</code>s.
 */
public ULCContainer() {
}
/**
 * Constructs a new ULCContainer which displays
 * a list of <code>IContainerItem</code>s.
 * If <code>items</code> is <code>null</code> we do nothing.
 *
 * Note: The vector is cloned to prevent
 *       outside manipulation.
 *
 * @throws	IllegalArgumentException if <code>items</code> don't conform to <code>IContainerItem</code>
 * @param	items		Vector of <code>IContainerItem</code>s
 */
public ULCContainer(Vector items) {
	if (items != null)
		internalSetItems(items);
}
/**
 * Constructs a new ULCContainer which displays
 * a list of <code>IContainerItem</code>s.
 * If <code>items</code> is <code>null</code> we do nothing.
 *
 * Note: The vector is cloned to prevent
 *       outside manipulation.
 *
 * @throws	IllegalArgumentException if <code>items</code> don't conform to <code>IContainerItem</code>
 * @param items			Vector of <code>IContainerItem</code>s
 * @param width			The width in pixels for this container.
 * @param height		The height in pixels for this container.
 */
public ULCContainer(Vector items, int width, int height) {
	this(items);
	fInitialWidth= width;
	fInitialHeight= height;
}
/**
 * Constructs a new ULCContainer which displays
 * a list of <code>IContainerItem</code>s.
 * If <code>items</code> is <code>null</code> we do nothing.
 *
 * Note: The vector is cloned to prevent
 *       outside manipulation.
 *
 * @throws	IllegalArgumentException if <code>items</code> don't conform to <code>IContainerItem</code>
 * @param items						Vector of <code>IContainerItem</code>s
 * @param width						The width in pixels for this container.
 * @param height					The height in pixels for this container.
 * @param multiSelectionAllowed		The selection mode (<code>boolean</code>)
 */
public ULCContainer(Vector items, int width, int height, boolean multiselectionAllowed) {
	this(items, width, height);
	fMultiSelectionAllowed= multiselectionAllowed;
}
/**
 * Registers the given listener to begin receiving notifications
 * when the icon is double-clicked or the enter has been pressed.
 * The cmd parameter of ULCActionEvent will contain a <code>String</code>
 * denoting if the double-click ("double-clicked") or the enter
 * key ("enter") triggered the event.
 *
 * @param listener	The object interested in my actionEvents.
 */
public void addActionListener(IActionListener listener) {
	internalAddListener("action", listener);
}
/**
 * Appends another item to the container.
 * If the item is already in the container
 * we still insert it at the end. If the
 * <code>item</code> is <code>null</code> we
 * do nothing.
 *
 * @param item 	The <code>IContainerItem</code> to be added as an item.
 */
public void addItem(IContainerItem item) {
	if (item != null) {
		fItems.addElement(item);
		ULCLabel ulcLabel= asULCLabel(item);
		internalAdd(ulcLabel); // sends "add" to the UI side
		if (fContext != null)
			fIdToItemMapping.put(new Integer(ulcLabel.getId()), item);
	}
}
/**
 * Registers the given listener to begin receiving notifications
 * when (de-) selecting items.
 *
 * @param listener	The object interested in my selectionChangedEvents.
 */
public void addSelectionChangedListener(ISelectionChangedListener listener) {
	internalAddListener("selectionChanged", listener);
}
/**
 * Return the label as ULCLabel. Construct
 * a new ULCLabel only if needed
 *
 * @param 	label		The <code>IContainerItem</code> to be converted to a <code>ULCLabel</code>
 * @return	The label as <code>ULCLabel</code>
 */
protected ULCLabel asULCLabel(IContainerItem label) {
	if (label instanceof ULCLabel)
		return (ULCLabel) label;
	ULCLabel ulcLabel= new ULCLabel(label.getIcon());
	ulcLabel.setLabel(label.getLabel());
	ulcLabel.setToolTipText(label.getToolTipText());
	ulcLabel.setPopupMenu(label.getPopupMenu());
	return ulcLabel;
}
/**
 * This method checks if the items conform
 * to <code>IContainerItem</code> interface.
 *
 * @param	items	The items to be checked
 * @throws IllegalArgumentException if <code>items</code> is null
 * or an item is null or an item is not of type <code>IContainerItem</code>.
 * @return <code>true</code> if items is <code>null</code> or all elements can be casted to <code>IContainerItem</code>
 */
protected boolean checkItems(Vector items) {
	if (items == null)
		throw new IllegalArgumentException("Vector is null");
	Object dummy= null;
	for (Enumeration enum= items.elements(); enum.hasMoreElements();)
		try {
			dummy= (IContainerItem) enum.nextElement();
			if (dummy == null)
				throw new IllegalArgumentException("Vector item is null");
		} catch (ClassCastException e) {
			throw new IllegalArgumentException("Vector item has illegal type");
		}
	return true;
}
/**
 * Gets the height that has or will
 * be uploaded.
 *
 * @return	The initial height as <code>int</code>
 */
public int getHeight() {
	return fInitialHeight;
}
/**
 * Return the horizontal gap between two icons
 *
 * @return <code>int</code> The horizontal gap between icons in pixels
 */
public int getHorizontalGap() {
	return fHorizontalGap;
}
/**
 * Return the number of elements in the container
 *
 * @return	The number of elements as an <code>int</code>
 */
public int getItemCount() {
	return fItems.size();
}
/**
 * Return the items that are in the container.
 * @return		The items of the container in a <code>Vector</code>
 */
public Vector getItems() {
	return (Vector) fItems.clone();
}
/**
 * Return the label position relative to the icon
 * of the <code>ULCContainer</code>
 *
 * @return <code>int</code> The position of the label relative to the icon:
 * <pre>
 *  					IDefaults.BOX_LEFT
 *  					IDefaults.BOX_RIGHT
 *  					IDefaults.BOX_TOP
 *  					IDefaults.BOX_BOTTOM
 * </pre>
 * Default is <code>IDefaults.BOX_BOTTOM</code>
 */
public int getLabelPosition() {
	return fLabelPosition;
}
/**
 * There are 3 ways in which the scroll bars can be displayed.
 * This integer will represent one of these 3 constants:
 * <pre>
 *  					IDefaults.SCROLLBARS_ALWAYS
 *  					IDefaults.SCROLLBARS_AS_NEEDED
 *  					IDefaults.SCROLLBARS_NEVER
 * </pre>
 * @return	Scrollbar policy as an <code>int</code>
 */
public int getScrollbarDisplayPolicy() {
	return fScrollbarDisplayPolicy;
}
/**
 * This method returns the index of the selected
 * item. If no OR MORE THAN ONE element is
 * selected <code>-1</code> will be returned.
 * @return <code>int</code>	The index of the one and only selected item
 */
public int getSelectedIndex() {
	if (fSelectedOids.size() != 1)
		return -1;
	else
		return fItems.indexOf(fIdToItemMapping.get(fSelectedOids.firstElement()));
}
/**
 * This method returns the indices of the selected
 * items in the items list. If no item is
 * is selected an empty vector is returned.
 * @return	The indices of the selected items in a <code>Vector</code>
 */
public Vector getSelectedIndices() {
	Vector indices= new Vector();
	for (Enumeration enum= fSelectedOids.elements(); enum.hasMoreElements();)
		indices.addElement(new Integer(fItems.indexOf(fIdToItemMapping.get(enum.nextElement()))));
	return indices;
}
/**
 * This method returns the selected
 * item. If no OR MORE THAN ONE item is
 * selected <code>null</code> will be returned.
 * @return	<code>null</code> or the one and only selected item (<code>IContainerItem</code>)
 */
public IContainerItem getSelectedItem() {
	if (fSelectedOids.size() != 1)
		return null;
	else
		return (IContainerItem) fIdToItemMapping.get(fSelectedOids.firstElement());
}
/**
 * This method returns the selected items. If no
 * item is selected an empty vector is returned.
 * @return	The selected items as <code>Vector</code>
 */
public Vector getSelectedItems() {
	Vector items= new Vector();
	for (Enumeration enum= fSelectedOids.elements(); enum.hasMoreElements();)
		items.addElement(fIdToItemMapping.get(enum.nextElement()));
	return items;

}
private Anything getSelectedOidsAsAnything() {
	Anything oidsAny = new Anything();
	for (Enumeration enum= fSelectedOids.elements(); enum.hasMoreElements();)
		oidsAny.append(new Anything(((Integer) enum.nextElement()).intValue()));
	return oidsAny;	
}
/**
 * Return the vertical gap between two icons
 *
 * @return <code>int</code> The vertical gap between icons in pixels
 */
public int getVerticalGap() {
	return fVerticalGap;
}
/**
 * Gets the width that has or will
 * be uploaded.
 *
 * @return	The initial width as <code>int</code>
 */
public int getWidth() {
	return fInitialWidth;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		Integer oidObj= null;
		int index= -1;
		
		String type = args.get("type", "???");

		if (type.equals("action")) {
			distributeToListeners("action", new ULCActionEvent(this, args.get("cmd", "???")));
			return;
		}
		if (type.equals("selectionCleared")) {
			fSelectedOids.removeAllElements();
			distributeToListeners("selectionChanged", new ULCSelectionChangedEvent(this, type, -1, -1));
			return;
		}
		if (type.equals("selectedItems")) {
			fSelectedOids.removeAllElements();
			Anything oids= args.get("selectedOids");
			Vector indices= new Vector(oids.size());
			int minIndex= Integer.MAX_VALUE;
			int maxIndex= Integer.MIN_VALUE;			
			for (int i= 0; i < oids.size(); i++) {
				oidObj= new Integer(oids.get(i).asInt(-1));
				if (oidObj.intValue() > -1) {
					index= fItems.indexOf(fIdToItemMapping.get(oidObj));
					if (index > -1) {
						fSelectedOids.addElement(oidObj);
						indices.addElement(new Integer(index));
					}
				}
			}
			distributeToListeners("selectionChanged", new ULCSelectionChangedEvent(this, type, indices));
			return;
		}
		else {
			oidObj= new Integer(args.get("oid", -1));
			if (oidObj.intValue() > -1)
				index= fItems.indexOf(fIdToItemMapping.get(oidObj));
			if (index < 0)
				return;
				
			if (type.equals("selectedItem")) {
				if (!isMultiSelectionAllowed())
					fSelectedOids.removeAllElements();
				fSelectedOids.addElement(oidObj);
				distributeToListeners("selectionChanged", new ULCSelectionChangedEvent(this, type, index, index));
				return;
			}
			if (type.equals("unselectedItem")) {
				fSelectedOids.removeElement(oidObj);
				distributeToListeners("selectionChanged", new ULCSelectionChangedEvent(this, type, index, index));
				return;
			}
			if (type.equals("singleItemSelected")) {
				fSelectedOids.removeAllElements();
				fSelectedOids.addElement(oidObj);
				distributeToListeners("selectionChanged", new ULCSelectionChangedEvent(this, type, index, index));
				return;
			}
		}
	}
	super.handleRequest(conn, request, args);
}
/**
 * Set items of the container.
 * If items is <code>null</code> we create a new Vector.
 *
 * @throws IllegalArgumentException if <code>items</code> don't conform to <code>IContainerItem</code>
 * @param	items	The items to be set as container items
 */
protected void internalSetItems(Vector items) {
	checkItems(items);
	fSelectedOids.removeAllElements();
	fItems= (Vector) items.clone();
	fList= new Vector();
	Vector ulcLabels= new Vector();
	for (Enumeration enum= items.elements(); enum.hasMoreElements();)
		ulcLabels.addElement(asULCLabel((IContainerItem) enum.nextElement()));
	internalAdd(ulcLabels); // sends "addMany" to the UI side
	if (fContext != null)
		for (int i= 0; i < fItems.size(); i++)
			fIdToItemMapping.put(new Integer(((ULCLabel) fList.elementAt(i)).getId()), fItems.elementAt(i));
}
/**
 * Answer if icons are auto-arranged.
 *
 * @return if icons are auto-arranged <code>true</code>
 * or not <code>false</code>.
 */
public boolean isAutoArranging() {
	return fAutoArranging;
}
/**
 * Answer if dragging of icons is allowed or not.
 * The default is false
 *
 * Note: Private because it is not part of the API [yet]
 *
 * @return if dragging of icons is allowed
 */
private boolean isDragIconsAllowed() {
	return fDragIconsAllowed;
}
/**
 * Answer if multi-selection is allowed or not.
 * The default is <code>true</code>
 *
 * @return if multi-selection is allowed
 */
public boolean isMultiSelectionAllowed() {
	return fMultiSelectionAllowed;
}
/**
 * For debugging only - print the selection state
 */
public void printSelectionState() {
	System.out.println("  " + fItems);
	System.out.println("  " + fSelectedOids);
	System.out.println("  " + getSelectedItems());	
	System.out.println("  " + getSelectedIndices());
	System.out.println("  " + getSelectedIndex());
	System.out.println("  " + getSelectedItem());
	System.out.println("------------------------------------------------");
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive action events.
 *
 * @param listener	The widget that was registered to receive my actionEvents.
 */
public void removeActionListener(IActionListener listener) {
	internalRemoveListener("action",listener);
}
/**
 * Removes the first item in the container
 * that is equal to <code>item</code>.
 * If there is no such item in the container
 * or <code>item</code> is <code>null</code> we do nothing.
 *
 * @param item 	The <code>IContainerItem</code> to be removed
 * @return <code>true</code> if an item has been removed
 * 			<code>false</code> otherwise.
 */
public boolean removeItem(IContainerItem item) {
	boolean result= false;
	if (item != null) {
		int index= fItems.indexOf(item);
		if (index >= 0) {
			result= fItems.removeElement(item);
			ULCLabel ulcLabel= (ULCLabel) fList.elementAt(index);
			fList.removeElementAt(index);
			if (fInitiallySelectedItems	!= null)
				fInitiallySelectedItems.removeElement(ulcLabel);
			if (fList != null && fContext != null) {
				Integer id= new Integer(ulcLabel.getId());
				fIdToItemMapping.remove(id);
				fSelectedOids.removeElement(id);
				sendUI("remove", ulcLabel);
			}
		}
	}
	return result;
}
/**
 * Unregisters the given observer from the notification list
 * so it will no longer receive events. 
 *
 * @param listener	The object that was registered for my selectionChanged events
 */
public void removeSelectionChangedListener(ISelectionChangedListener listener) {
	internalRemoveListener("selectionChanged", listener);
}
/**
 * @param a	The <code>Anything</code> into which we store the state
 */
protected void saveState(Anything a) {
	super.saveState(a);

	if (fInitiallySelectedItems != null) {
		// Build the mapping
		for (int i= 0; i < fItems.size(); i++)
			fIdToItemMapping.put(new Integer(((ULCLabel) fList.elementAt(i)).getId()), fItems.elementAt(i));
		// Build fSelectedOids
		for (Enumeration enum= fInitiallySelectedItems.elements(); enum.hasMoreElements();)
			fSelectedOids.addElement(new Integer(((ULCLabel) enum.nextElement()).getId()));
		fInitiallySelectedItems= null;
	}
		
	if (fInitialWidth != 0)
		a.put("w", fInitialWidth);
	if (fInitialHeight != 0)
		a.put("h", fInitialHeight);
	if (fLabelPosition != BOX_BOTTOM)
		a.put("labelPosition", fLabelPosition);
	if (!fMultiSelectionAllowed)
		a.put("multiSelectionAllowed", fMultiSelectionAllowed);
	if (fDragIconsAllowed)
		a.put("dragIconsAllowed", fDragIconsAllowed);
	if (fHorizontalGap != 5)
		a.put("hg", fHorizontalGap);	
	if (fVerticalGap != 5)
		a.put("vg", fVerticalGap);
	if (!fSelectedOids.isEmpty())
		a.put("selectedOids", getSelectedOidsAsAnything());
	if (!fAutoArranging)
		a.put("autoArranging", fAutoArranging);
	if (fScrollbarDisplayPolicy != SCROLLBARS_AS_NEEDED)
		a.put("scrollbarDisplayPolicy", fScrollbarDisplayPolicy);
//	super.saveState(a);
}
/**
 * Set if auto-arrangement of icons is turned
 * on (<code>true</code>) or off (<code>false</code>).
 * This will also force a re-arrangement of the
 * icons.
 *
 * @param state Tells if auto-arranging is turned on or off
 */
public void setAutoArranging(boolean state) {
	if (fAutoArranging != state) {
		fAutoArranging= state;
		if (fAutoArranging)
			fDragIconsAllowed= false;
		sendUI("setAutoArranging", new Anything(state));			
	}
}
/**
 * Set if it is allowed to drag the icons or not.
 * If we change from allowed to not-allowed then
 * we re-arrange the icons.
 *
 * Note: Private because it is not part of the API [yet]
 *
 * @param state tells if dragging icons is allowed
 */
private void setDragIconsAllowed(boolean state) {
	if (fDragIconsAllowed != state) {
		fDragIconsAllowed= state;
		if (fDragIconsAllowed)
			fAutoArranging= false;
		sendUI("setDragIconsAllowed", new Anything(state));
	}
}
/**
 * Sets the height of this widget.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @throws IllegalArgumentException if <code>height</code> is negative
 * @param height	The <code>int</code> height
 */
public void setHeight(int height) {
	if (height < 0)
		throw new IllegalArgumentException("Height must be positive");
	fInitialHeight= height;
}
/**
 * Set the horizontal gap between two icons
 *
 * @throws IllegalArgumentException if <code>gap</code> is negative
 * @param gap The vertical gap between icons in pixels (<code>int</code>)
 */
public void setHorizontalGap(int gap) {
	if (gap < 0)
		throw new IllegalArgumentException("Horizontal gap can't be negative");
	if (fHorizontalGap != gap) {
		fHorizontalGap= gap;
		sendUI("setHGap", new Anything(fHorizontalGap));
	}
}
/**
 * Set items of the container.
 * If items is null we create a new Vector
 *
 * @throws IllegalArgumentException if not all items are
 *		   <code>IContainerItem</code>s
 * @param items	The items to be set as the container's items
 */
public void setItems(Vector items) {
	if (items == null) {
		fIdToItemMapping.clear();
		fItems= new Vector();
		fList= new Vector();	// the internal list with ULCLables
		sendUI("allItemsRemoved");		
	}
	else
		internalSetItems(items);
}
/**
 * Set the label position relative to the icon
 * of the <code>ULCContainer</code>
 *
 * @throws IllegalArgumentException if <code>labelPosition</code> is wrong
 * @param labelPosition	The position of the label relative to the icon:
 * <pre>
 *  					IDefaults.BOX_LEFT
 *  					IDefaults.BOX_RIGHT
 *  					IDefaults.BOX_TOP
 *  					IDefaults.BOX_BOTTOM
 * </pre>
 * Default is <code>IDefaults.BOX_BOTTOM</code>
 */
public void setLabelPosition(int labelPosition) {
	if (labelPosition != BOX_TOP && labelPosition != BOX_BOTTOM
		&& labelPosition != BOX_LEFT && labelPosition != BOX_RIGHT)
		throw new IllegalArgumentException("Wrong label position");
	if (fLabelPosition != labelPosition) {
		fLabelPosition= labelPosition;
		sendUI("setLabelPosition", new Anything(fLabelPosition));
	}
}
/**
 * Set if multi-selection is allowed or not.
 * If more than one item is selected then
 * only the item that has been first selected
 * remains selected.
 *
 * @param state	Tells if multiselection is allowed
 */
public void setMultiSelectionAllowed(boolean state) {
	if (fMultiSelectionAllowed != state) {
		fMultiSelectionAllowed= state;
		sendUI("setMultiSelectionAllowed", new Anything(state));			
	}
}
/**
 * There are 3 ways in which the scroll bars can be displayed.
 * @throws IllegalArgumentException if <code>policy</code> is wrong
 * @param	policy	The scrollbar policy:
 * <pre>
 *  					IDefaults.SCROLLBARS_ALWAYS
 *  					IDefaults.SCROLLBARS_AS_NEEDED
 *  					IDefaults.SCROLLBARS_NEVER
 * </pre>
 */
public void setScrollbarDisplayPolicy(int policy) {
	if (policy != SCROLLBARS_ALWAYS && policy != SCROLLBARS_AS_NEEDED
		&& policy != SCROLLBARS_NEVER)
		throw new IllegalArgumentException("Wrong scrollbar display policy");
	if (fScrollbarDisplayPolicy != policy) {
		fScrollbarDisplayPolicy= policy;
		sendUI("setScrollbarDisplayPolicy", new Anything(fScrollbarDisplayPolicy));
	}
}
/**
 * Select the item at <code>index</code>
 *
 * @throws IllegalArgumentException if no object at <code>index</code>
 * @param index The index of the row to be selected.
 */
public void setSelectedIndex(int index) {
	Vector indicesToSelect= new Vector();
	indicesToSelect.addElement(new Integer(index));
	setSelectedIndices(indicesToSelect);
}
/**
 * Set the selection according to the <code>indices</code>
 *
 * @throws IllegalArgumentException if more than one object
 * 			should be selected in single-select mode
 *			or <code>indices</code> is <code>null</code>
 *			or one of the indices doesn't point to an item
 *			of the container
 * @param indices The vector of indices to be selected.
 */
public void setSelectedIndices(Vector indices) {
	ULCLabel item= null;
	if (indices == null || (!isMultiSelectionAllowed() && indices.size() > 1))
		throw new IllegalArgumentException("multiselection not allowed");
	fSelectedOids.removeAllElements();
	for (Enumeration enum= indices.elements(); enum.hasMoreElements();) {
		try {
			item= (ULCLabel) fList.elementAt(((Integer) enum.nextElement()).intValue());
		}
		catch (ArrayIndexOutOfBoundsException e) {
			throw new IllegalArgumentException("item not in items list");
		}
		if (fContext == null)
			fInitiallySelectedItems.addElement(item);
		else
			fSelectedOids.addElement(new Integer(item.getId()));
	}
	sendUI("setSelectedOids", getSelectedOidsAsAnything());
	distributeToListeners("select", new ULCSelectionChangedEvent(this, "selectionSet", -1, -1));
}
/**
 * Set the only selected item to be the object
 * specified by <code>item</code>
 * 
 * @throws  IllegalArgumentException if <code>item</code>
 			is <code>null</code> or not in the container
 * @param  item The <code>IContainerItem</code>  to be selected.
 */
public void setSelectedItem(IContainerItem item) {
	if (item == null)
		throw new IllegalArgumentException("value must not be null");
	int index= fItems.indexOf(item);
	if (index == -1)
		throw new IllegalArgumentException("item not in items list");
	Vector indicesToSelect= new Vector();
	indicesToSelect.addElement(new Integer(index));
	setSelectedIndices(indicesToSelect);
}
/**
 * Set the currently selected items to be the objects
 * specified in <code>items</code>
 *
 * @throws IllegalArgumentException if
 *			more than one object should be selected in single-select mode
 *			or <code>items</code> is <code>null</code>
 *			or one of the items is not in the container
 *	
 * @param  items The <code>Vector</code> of objects to be selected.
 */
public void setSelectedItems(Vector items) {
	if (items == null)
		throw new IllegalArgumentException("value must not be null");
	Vector indices= new Vector();
	int index= -1;
	for (Enumeration enum= items.elements(); enum.hasMoreElements();) {
		index= fItems.indexOf(enum.nextElement());
		if (index == -1)
			throw new IllegalArgumentException("item not in items list");
		indices.addElement(new Integer(index));
	}
	setSelectedIndices(indices);
}
/**
 * Set the vertical gap between two icons
 *
 * @throws  IllegalArgumentException if <code>gap</code> is negative
 * @param gap <code>int</code> The vertical gap between icons in pixels
 */
public void setVerticalGap(int gap) {
	if (gap < 0)
		throw new IllegalArgumentException("Vertical gap can't be negative");
	if (fVerticalGap != gap) {
		fVerticalGap= gap;
		sendUI("setVGap", new Anything(fVerticalGap));
	}
}
/**
 * Set the width of this widget.
 * Setting this value after the widget has been uploaded has no effect.
 *
 * @throws  IllegalArgumentException if <code>width</code> is negative
 * @param width	The <code>int</code> width
 */
public void setWidth(int width) {
	if (width < 0)
		throw new IllegalArgumentException("Width can't be negative");
	fInitialWidth= width;
}
}
