package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import com.ibm.ulc.util.Anything;

/**
 * A ULCColumn represents all the attributes of a column in a ULCTable,
 * such as the column name in the associated TableModel,
 * width, editability, and how values are rendered.
 */
public class ULCColumn extends ULCComponent {
	/**
	 * The attributeName of this column
	 * @serial
	 */
	private String fColId = null;
	/**
	 * The IDataType configured for this column
	 * @serial
	 */
	protected IDataType fDataType = null;
	/**
	 * The widget to be used as my renderer
	 * @serial
	 */
	private ULCLabel fHeaderRenderer = null;
	/**
	 * The widget to be used as my header renderer
	 * @serial
	 */
	private ULCComponent fRenderer = null;
	/**
	 * The width of this column in pixels
	 * @serial
	 */
	private int fWidth = 40;
	/**
	 * Allow this column to be editable if true.
	 * @serial
	 */
	private boolean fEditable = false;
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 * The column's header label is the same as the key.
 */
public ULCColumn() {
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 * The column's header label is the same as the key.
 */
public ULCColumn(String attributeName) {
	super(attributeName);
	fColId = attributeName;
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 * The column's header label is the same as the key.
 * @param width the width of the column.
 */
public ULCColumn(String attributeName, int width) {
	super(attributeName);
	fColId = attributeName;
	fWidth = width;
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 * The column's header label is the same as the key.
 * @param width the width of the column.
 */
public ULCColumn(String attributeName, int width, IDataType datatype) {
	this(attributeName, width);
	fDataType = datatype;
}
/**
 * Constructs a table column which extracts the values with the given key from the 
 * TableModel.
 * The column's header label is the same as the key.
 *
 * @param 	attributeName 	
 *			The attribute name which will used to fetch data from the TableModel
 * @param datatype 
 *			The datatype associated with this column. The datatype is used to
 *			perform data validation or format conversion.
 * @see IDataType
 */
public ULCColumn(String attributeName, IDataType datatype) {
	this(attributeName);
	fDataType = datatype;
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 * The column's header label is the same as the key.
 *
 * @param 	attributeName 	
 *			The attribute name which will used to fetch data from the TableModel
 * @param renderer Render the value in this column by using this ULCComponent.
 *			For example if the renderer is a ULCCheckBox then the data retrieved
 *			using key from the ULCTableModel is interpreted and displayed by the
 *			ULCCheckBox.
 */
public ULCColumn(String attributeName, ULCComponent renderer) {
	super(attributeName);
	fColId = attributeName;
	fRenderer = renderer;
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 * The column's header label is the same as the key.
 *
 * @param 	attributeName 	
 *			The attribute name which will used to fetch data from the TableModel
 * @param renderer Render the value in this column by using this ULCComponent.
 *			For example if the renderer is a ULCCheckBox then the data retrieved
 *			using key from the ULCTableModel is interpreted and displayed by the
 *			ULCCheckBox.
 * @param datatype 
 *			The datatype associated with this column. The datatype is used to
 *			perform data validation or format conversion.
 * @see IDataType
 */
public ULCColumn(String attributeName, ULCComponent renderer, IDataType datatype) {
	this(attributeName, renderer);
	fDataType = datatype;
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 *
 * @param 	key 	
 *			The attribute name which will used to fetch data from the TableModel
 * @param headerValue The column's header label.
 */
public ULCColumn(String attributeName, String headerValue) {
	super(headerValue);
	fColId = attributeName;
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 *
 * @param 	key 	
 *			The attribute name which will used to fetch data from the TableModel
 * @param headerValue The column's header label.
 * @param width the width of the column.
 */
public ULCColumn(String attributeName, String headerValue, int width) {
	super(headerValue);
	fColId = attributeName;
	fWidth = width;
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 *
 * @param 	attributeName 	
 *			The attribute name which will used to fetch data from the TableModel
 * @param headerValue The column's header headerValue.
 * @param width the width of the column.
 * @param datatype 
 *			The datatype associated with this column. The datatype is used to
 *			perform data validation or format conversion.
 * @see IDataType
 */
public ULCColumn(String attributeName, String headerValue, int width, IDataType datatype) {
	this(attributeName, headerValue, width);
	fDataType = datatype;
}
/**
 * Constructs a table column which extracts the values with the given key from the TableModel.
 *
 * @param 	attributeName 	
 *			The attribute name which will used to fetch data from the TableModel
 * @param headerValue The column's header headerValue.
 * @param renderer Render the value in this column by using this ULCComponent.
 *			For example if the renderer is a ULCCheckBox then the data retrieved
 *			using key from the ULCTableModel is interpreted and displayed by the
 *			ULCCheckBox.
 */
public ULCColumn(String attributeName, String headerValue, ULCComponent renderer) {
	super(headerValue);
	fColId = attributeName;
	fRenderer = renderer;
}
/**
 * Gets the columns's identifier.
 *
 * @return The string identifier for this column.
 */
public String getAttributeName() {
	return fColId;
}
/**
 * Gets the formatter and validator object for this field.
 *
 * @return datatype The frmatter/validator associated with this column
 */
public IDataType getDataType() {
	return fDataType;
}
/**
 * Gets the ULCLabel that is configured as my header renderer
 *
 * @return ULCLabel The widget that is used as my header renderer
 */
public ULCLabel getHeaderRenderer() {
	return fHeaderRenderer;
}
/**
 * Gets the columns's label.
 *
 * @return The string label for this column.
 */
public String getLabel() {
	return fLabel;
}
/**
 * Gets the ULCComponent that is configured as my renderer
 *
 * @return ULCComponent The widget that is used as my renderer
 */
public ULCComponent getRenderer() {
	return fRenderer;
}
/**
 * Gets this column's width
 *
 */
public int getWidth() {
	return fWidth;
}
/**
 * Return true if this column is editable .
 *
 * @return editable if true allows direct edit in this column.
 */
public boolean isEditable() {
	return fEditable;
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	if (fColId != null)
		a.put("colId", fColId);
	if (fDataType != null)
		a.put("dataType", fDataType.getRef(fContext));
	if (fWidth != 40)
		a.put("w", fWidth);
	if (fRenderer != null)
		a.put("renderer", fRenderer.getRef(fContext));
	if (fHeaderRenderer != null)
		a.put("hrenderer", fHeaderRenderer.getRef(fContext));
	if (fEditable)
		a.put("editable", fEditable);
	super.saveState(a);
}
/**
 * Sets the columns's identifier.
 *
 * @param The string identifier for this column.
 */
public void setAttributeName(String attributeName) {
	if ((fColId == null) || (!fColId.equals(attributeName))) {
		fColId = attributeName;
		sendUI("setColId", new Anything(fColId));
	}
}
/**
 * Sets the formatter and validator object for this field.
 *
 * @param datatype The frmatter/validator associated with this column
 */
public void setDataType(IDataType datatype) {
	fDataType = datatype;
	sendUI("setDataType", fDataType);
}
/**
 * Set the column's editability.
 *
 * @param editable if true allows direct edit in this column.
 */
public void setEditable(boolean editable) {
	if (editable != fEditable) {
		fEditable = editable;
		sendUI("setEditable", new Anything(fEditable));
	}
}
/**
 * Sets the ULCLabel that is configured as my header renderer
 *
 * @param ULCLabel The widget that is used as my header renderer
 */
public void setHeaderRenderer(ULCLabel renderer) {
	fHeaderRenderer = renderer;
	sendUI("setHeaderRenderer", fHeaderRenderer);
}
/**
 * Sets the widgets's <code>String</code> label.
 *
 * @param label the label <code>String</code> of the widget.
 */
public void setLabel(String label) {
	internalSetLabel(label);
}
/**
 * Sets the ULCComponent that is configured as my renderer
 *
 * @param ULCComponent The widget that is used as my renderer
 */
public void setRenderer(ULCComponent renderer) {
	fRenderer = renderer;
}
/**
 * Sets this column's width to <I>newWidth</I>.  If <I>newWidth</I>
 * exceeds the minimum or maximum width, it's adjusted to the
 * appropriate limiting value in the UI Engine.
 *
 * @param newWidth The width to be set
 */
public void setWidth(int newWidth) {
		fWidth = newWidth;
		sendUI("setWidth", new Anything(fWidth));
}
/**
 * Convenience method to switch the icon of the column header during a sort request on a column header.
 *
 * @param lastSortedColumn 		The last column that has been sorted in this table or null.
 * @param currentSortedColumn 	The current column that is being sorted.
 * @param ascendingIcon 		The <code>ULCIcon</icon> that should be displayed when the sort is an ascending sort.
 * @param descendingIcon 		The <code>ULCIcon</icon> that should be displayed when the sort is a descending sort.
 * @param ascending 			The flag indicating if the sort is ascending or descending.
 */
public static void swapHeaderIcon(ULCColumn lastSortedColumn, ULCColumn currentSortedColumn, ULCIcon ascendingIcon, ULCIcon descendingIcon, boolean ascending) {
	if (lastSortedColumn != null) {
		ULCLabel label = lastSortedColumn.getHeaderRenderer();
		if ((label != null) && (lastSortedColumn != currentSortedColumn))
			label.setIcon(null);
	}
	ULCLabel label = currentSortedColumn.getHeaderRenderer();
	if (label == null) {
		label = new ULCLabel(currentSortedColumn.getLabel());
		label.setHorizontalAlignment(BOX_CENTER);
		label.setHorizontalTextPosition(BOX_LEFT);
	}
	if (ascending)
		label.setIcon(ascendingIcon);
	else
		label.setIcon(descendingIcon);
	currentSortedColumn.setHeaderRenderer(label);
}
}
