package com.ibm.ulc.application;

/*
 * Copyright (c) 1999-2000 Object Technology International Inc.
 */
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.base.*;
/**
 * Implement an ITrigger that can block all input in the UI whenever 
 * an action or selection occurs in the UI.
 */
public class ULCBlockInputTrigger extends ULCTrigger {
	protected int fMode = TRIGGER_ALL_SHELLS;
	protected int fTriggerOn = TRIGGER_ON_ACTION;
	protected boolean fAutoReset = true;
/**
 * Creates a trigger that will block input of all shell widgets within this ULCApplication in the UI.
 *
 */
 public ULCBlockInputTrigger() {
	this(TRIGGER_ALL_SHELLS);
}
/**
 * Creates a trigger that will block input of its widgets in the UI based on the mode.
 *
 * @param mode  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>TRIGGER_ALL_SHELLS</code> (the default),
 *           <code>TRIGGER_SHELL</code>
 *
 */
public ULCBlockInputTrigger(int mode) {
	this(TRIGGER_ALL_SHELLS, TRIGGER_ON_ACTION);
}
/**
 * Creates a trigger that will block input of its widgets in the UI based on the mode.
 *
 * @param mode  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>TRIGGER_ALL_SHELLS</code> (the default),
 *           <code>TRIGGER_SHELL</code>
 *
 * @param triggerOn  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>TRIGGER_ON_ACTION</code> (the default),
 *           <code>TRIGGER_ON_SELECTION</code>,
 *           <code>TRIGGER_ON_ACTION_AND_SELECTION</code>.
 *
 * Note: widgets that only have valueChanged listeners 
 * like MenuItems, ToggleMenuItems, ToggleButtons will get triggered
 * both when the triggerOn property is set to ACTION and/or SELECTION.
 */
public ULCBlockInputTrigger(int mode, int triggerOn) {
	super();
	fMode = mode;
	fTriggerOn = triggerOn;
}
/**
 * Gets the mode of this trigger.
 * Either block input on all shells within the UI or the owning widgets shell.
 *
 * @return mode  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>TRIGGER_ALL_SHELLS</code> (the default),
 *           <code>TRIGGER_SHELL</code>
 *
 */
public int getMode() {
	return fMode;
}
/**
 * Returns when this trigger is activated in the UI.
 *
 * @return triggerOn  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>TRIGGER_ON_ACTION</code> (the default),
 *           <code>TRIGGER_ON_SELECTION</code>,
 *           <code>TRIGGER_ON_ACTION_AND_SELECTION</code>.
 *
 * Note: widgets that only have valueChanged listeners 
 * like MenuItems, ToggleMenuItems, ToggleButtons will get triggered
 * both when the triggerOn property is set to ACTION and/or SELECTION.
 */
public int getTriggerOn() {
	return fTriggerOn;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If the call chain has reached this object it means that the request is an unknown request.
 * Write the received request to the console.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		if (args.get("type", "").equals("triggered"))
			reset();
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Returns true if this trigger will automatically reset itself.
 * If false it is the application developers responsibility to call 
 * reset on this trigger.
 *
 * @return boolean autoReset if true this trigger will reset itself automatically
 *
 */
public boolean isAutoReset() {
	return fAutoReset;
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fMode != TRIGGER_ALL_SHELLS)
		a.put("mode", new Anything(fMode));
	if (fTriggerOn != TRIGGER_ON_ACTION)
		a.put("triggerOn", new Anything(fTriggerOn));
	if (!fAutoReset)
		a.put("autoReset", new Anything(fAutoReset));
}
/**
 * Returns true if this trigger will automatically reset itself.
 * If false it is the application developers responsibility to call 
 * reset on this trigger.
 *
 * @param boolean autoReset if true this trigger will reset itself automatically
 *
 */
public void setAutoReset(boolean autoReset) {
	if (fAutoReset != autoReset) {
		fAutoReset = autoReset;
		sendUI("setAutoReset", new Anything(fAutoReset));
	}
}
/**
 * Sets the mode of this trigger.
 * Either block input on all shells within the UI or the owning widgets shell.
 *
 * @return mode  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>TRIGGER_ALL_SHELLS</code> (the default),
 *           <code>TRIGGER_SHELL</code>
 *
 */
public void setMode(int mode) {
	if (fMode != mode) {
		fMode = mode;
		sendUI("setMode", new Anything(fMode));
	}
}
/**
 * Sets when this trigger is activated in the UI.
 *
 * @param triggerOn  One of the following constants
 *           defined in <code>IDefaults</code>:
 *           <code>TRIGGER_ON_ACTION</code> (the default),
 *           <code>TRIGGER_ON_SELECTION</code>,
 *           <code>TRIGGER_ON_ACTION_AND_SELECTION</code>.
 *
 * Note: widgets that only have valueChanged listeners 
 * like MenuItems, ToggleMenuItems, ToggleButtons will get triggered
 * both when the triggerOn property is set to ACTION and/or SELECTION.
 */
public void setTriggerOn(int triggerOn) {
	if (fTriggerOn != triggerOn) {
		fTriggerOn = triggerOn;
		sendUI("setTriggerOn", new Anything(fTriggerOn));
	}
}
}
