package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */
import java.util.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;
import com.ibm.ulc.comm.ORBConnection;

/**
 * Subclasses of this widget is used for representing an hierarchical relationship
 * between objects. The tree has a 'root  node' which has 'children';
 * each child in turn may have children of it's own. 
 * The tree is also associated with an adapter (ULCTreeModel) for mapping
 * onto the underlying hierarchical business objects. 
 * Note that the business objects do not need to be modelled as 'nodes'
 * in order to be used in the tree - for example, a Directory structure 
 * where a directory contains other directories and files, can also be
 * displayed easily as a tree.
 *
 * Also, a tree can have columns associated with it, in the same way
 * as a table can. Thus, each business object can display it's attributes 
 * (in addition to it's icon and label) in the tree.
 *
 * @see ULCTreeModel
 * @see ULCColumn 
 */
abstract public class ULCAbstractTree extends ULCAbstractList {
	/**
	 * Determines whether the root of the tree is displayed or not.
	 * @serial	 	 
	 */
	protected boolean fRootVisible = true;
	/**
	 * Determine whether a given node is currently expanded or not.
	 * @serial
	 */
	protected boolean fIsNodeExpanded;
/**
 * Constructs a new instance of the receiver.
 */
public ULCAbstractTree() {
}
/**
 * Constructs a new instance of the receiver, which displays values from the 
 * given ULCAbstractTableModel.
 *
 * @param model 	The ULCAbstractTableModel that serves as the data source.
 */
public ULCAbstractTree(ULCAbstractTableModel model) {
	super(model);
}
/**
 * Constructs a new instance of the receiver, which displays values from the 
 * given ULCAbstractTableModel.
 *
 * @param model 	The ULCAbstractTableModel that serves as the data source.
 * @param selectionMode 	Specifies the selection policy for the tree.
 * 							<pre>
 * 							Valid values are :
 *								TREE_SINGLE_SELECTION
 * 								TREE_CONTIGUOUS_SELECTION
 *								TREE_DISCONTIGUOUS_SELECTION
 * 							</pre>
 */
public ULCAbstractTree(ULCAbstractTableModel model, int selectionMode) {
	this(model);
	fSelectionMode = selectionMode;
}
/**
 * Constructs a new instance of the receiver, which displays values from the 
 * given ULCAbstractTableModel.
 *
 * @param model 	The ULCAbstractTableModel that serves as the data source.
 * @param selectionMode 	Specifies the selection policy for the tree.
 * 							<pre>
 * 							Valid values are :
 *								TREE_SINGLE_SELECTION
 * 								TREE_CONTIGUOUS_SELECTION
 *								TREE_DISCONTIGUOUS_SELECTION
 * @param rowHeight		The height specified as number of visible rows for this tree.
 * 							</pre>
 */
public ULCAbstractTree(ULCAbstractTableModel model, int selectionMode, int rowHeight) {
	super(model, selectionMode, rowHeight);
}
/**
 * Registers the given listener to begin receiving notifications when 
 * a node in the tree is collapsed.
 *
 * @param listener	The object interested in my treeEvents.
 */
public void addTreeCollapseActionListener(ITreeListener listener) {
	enableOptionalEvent("nodeCollapsed");
	internalAddListener("nodeCollapsed", listener);
}
/**
 * Registers the given listener to begin receiving notifications when a 
 * (double-click) is performed on a node.
 *
 * @param listener	The object interested in my treeEvents.
 */
public void addTreeDefaultActionListener(ITreeListener listener) {
	enableOptionalEvent("defaultAction");
	internalAddListener("defaultAction", listener);
}
/**
 * Registers the given listener to begin receiving notifications when 
 * a node in the tree is expanded.
 *
 * @param listener	The object interested in my treeEvents.
 */
public void addTreeExpandActionListener(ITreeListener listener) {
	enableOptionalEvent("nodeExpanded");
	internalAddListener("nodeExpanded", listener);
}
/**
 * Registers the given listener to begin receiving notifications when 
 * there is a change in the selected nodes of the tree.
 *
 * @param listener	The object interested in my treeEvents.
 */
public void addTreeSelectionChangedListener(ITreeListener listener) {
	internalAddListener("select", listener);
}
/**
 * Programmatically collapse the tree at the given node
 *
 * @param object	The node to be collapsed.
 */
public void collapseNode(Object object) {
	sendUI("collapse", new Anything(internalGetModel().getRowIdFor(object)));
}
/*
 * The selection has changed on the UI. Notify all listerners of this fact.
 *
 * @param type 	: The type of event, namely "select".
 */
public void distributeSelectionChangeEventToListeners(String type) {
	Vector selectedNodes = getSelectedItems();
	distributeToListeners("select", new ULCTreeEvent(this, "selectionChanged", selectedNodes));
}
/**
 * Requests my UI proxy to ensure that the object at <code>object</code> is visible.
 *
 * @param object	The object of the table to be made visible.
 */
public void ensureItemIsVisible(Object object) {
	ensureNodeIsVisible(object);
}
/**
 * Requests my UI proxy to ensure that the tree node at <code>node</code> is visible.
 *
 * @param node	The node of the tree to be made visible.
 */
public void ensureNodeIsVisible(Object node) {
	int id = internalGetModel().getRowIdFor(node);
	if (id != -1) { // skip if node unavailable
		Vector rowIds= new Vector();
		rowIds.add(new Integer(id));
		Vector nodes= getSelectedNodesFrom(rowIds);
		fContext.startBatchingRequests();
		getTreeItemList().sendPaths(nodes);
		sendUI("ensureItemIsVisible", new Anything(id));		
		fContext.stopBatchingRequests();
	}
}
/**
 * Programmatically expand the tree at the given node
 *
 * @param object	The node to be expanded.
 */
public void expandNode(Object object) {
	sendUI("expand", new Anything(internalGetModel().getRowIdFor(object)));
}
/**
 * Returns the current list of all ULCColumns in this tree.
 *
 * @return	The <code>Vector</code> containing a list of <code>ULCColumn</code> objects.
 * @deprecated with R3.0		Tree with columns is now supported in ULCTreeTable
 */
public Vector getColumns() {
	trouble("addColumn", "ULCTree no longer supports columns. Use TreeTable instead");
	return new Vector();
}
/**
 * Answer the default item list for the receiver, 
 * which is the default HierarchicalItemList of the receiver's model.
 * If the receiver's model is null, answer null.
 *
 */
protected IItemList getDefaultItemList() {
	IItemList answer = null;
	try {
		ULCAbstractTableModel model = internalGetModel();
		if (model != null)
			answer = getTreeModel().getHierarchicalItemList();
	} catch (ClassCastException cce) {
	}
	return answer;
}
/**
 * Answer the receiver's treeModel
 */
public IHierarchicalTableModel getModel() {
	return (IHierarchicalTableModel)internalGetModel();
}
/**
 * Answer the object identified by the given parameter.
 *
 * @param args	Anything	Contains the identifier to locate the
 * 							required object.
 * @return the Object identified by this id.
 *
 */
protected Object getNodeFrom(Anything args) {
	int nodeId = args.asInt(-1);
	return fModel.getRow(nodeId);
}
/**
 * Return the selected node of the tree or null if nothing is selected
 *
 * @return node The node selected.
 * @deprecated with R3.0, use #getSelectedItem instead.
 */
public Object getSelectedNode() {
	return getSelectedItem();
}
/**
 * Return the selected nodes of the tree
 *
 * @return nodes The vector of nodes to be selected.
 * @deprecated with R3.0, use #getSelectedItems instead.
 */
public Vector getSelectedNodes() {
	return getSelectedItems();
}
/**
 * Answer the vector of objects identified by the objectIdentifiers.
 *
 * @param args	oids	Contains the objectIdentifiers to locate the
 * 							required objects.
 *
 */
protected Vector getSelectedNodesFrom(int[] oids) {
	Vector nodes = new Vector(oids.length);
	for (int i = 0; i < oids.length; i++) {
		Object nextNode = internalGetModel().getRow(oids[i]);
		nodes.addElement(nextNode);
	}
	return nodes;
}
/**
 * Answer the vector of objects identified by the given parameter.
 *
 * @param args	Anything	Contains the identifiers to locate the
 * 							required objects.
 *
 */
protected Vector getSelectedNodesFrom(Anything a) {
	Anything selectedNodesAny= a.get("s");
	Vector nodes = new Vector(selectedNodesAny.size());
	for (int i = 0; i < selectedNodesAny.size(); i++) {
		Object nextNode= getNodeFrom(selectedNodesAny.get(i));
		nodes.addElement(nextNode);
	}
	return nodes;
}
/**
 * Answer the vector of objects identified by the given parameter.
 *
 * @param args	Anything	Contains the objectIdentifiers to locate the
 * 							required objects.
 *
 */
protected Vector getSelectedNodesFrom(Vector oids) {
	Vector nodes = new Vector(oids.size());
	for (int i = 0; i < oids.size(); i++) {
		Object nextNode = internalGetModel().getRow(((Integer) oids.elementAt(i)).intValue());
		nodes.addElement(nextNode);
	}
	return nodes;
}
/**
 * Answer the itemList for the receiver tree.
 */
public IHierarchicalItemList getTreeItemList() {
	return (IHierarchicalItemList) getItemList();
}
/**
 * Answer the receiver's treeModel
 *
 * @deprecated with R3.0, use #getModel() instead
 */
public IHierarchicalTableModel getTreeModel() {
	return getModel();
}
/**
 * Handle the incoming defaultAction event.
 *
 * @param args	Anything	Contains the node(s) on which the default
 *							action was performed.
 */
protected void handleDefaultAction(Anything args) {
	Vector nodes = (args.get("rows")).get("rowids").toCollection();
	if (nodes == null)
		nodes = new Vector();
	else
		nodes = getSelectedNodesFrom(nodes);
	distributeToListeners("defaultAction", new ULCTreeEvent(this, "defaultAction", nodes));
}
/**
 * Handle the incoming event which indicates that a node of the tree
 * has been collapsed.
 *
 * @param collapsedNode		Object	The collapsed node.
 */
protected void handleNodeCollapsed(Object collapsedNode) {
	distributeToListeners("nodeCollapsed", new ULCTreeEvent(this, "nodeCollapsed", collapsedNode));
}
/**
 * Handle the incoming event which indicates that a node of the tree
 * has been expanded.
 *
 * @param expandedNode	Object	The expanded node.
 */
protected void handleNodeExpanded(Object expandedNode) {
	distributeToListeners("nodeExpanded", new ULCTreeEvent(this, "nodeExpanded", expandedNode));
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("setSelectedNodes")) {
			handleSetSelectedNodes(args);
		} else
			if (type.equals("action")) {
				handleDefaultAction(args);
			} else
				if (type.equals("nodeExpanded")) {
					handleNodeExpanded(getNodeFrom(args.get("id")));
				} else
					if (type.equals("nodeCollapsed")) {
						handleNodeCollapsed(getNodeFrom(args.get("id")));
					} else
						super.handleRequest(conn, request, args);
	} else
		if (request.equals("isExpanded")) {
			fIsNodeExpanded = args.asBoolean(false);
			return;
		} else
			System.out.println("Invalid Request: " + request + " to ULCTree");
}
/**
 * Handle the incoming event which indicates that some nodes need
 * to be set as 'selected'.
 *
 * @param args	Anything	Contains the node(s) to be selected.
 */
protected void handleSetSelectedNodes(Anything args) {
	setSelectedOids("select", (args.get("rows")).get("rowids").toCollection());
	distributeToListeners("selectionChanged", new ULCTreeEvent(this, "selectionChanged", getSelectedItems()));
}
/**
 * Programmatically check whether the given node is expanded in the tree.
 * Note : This is a synchronous call.
 *
 * @param object 	Object (node) which is checked for expansion.
 * @answer boolean, true if the given node is expanded in the tree.
 */
public boolean isNodeExpanded(Object object) {
	sendUI("isExpanded", new Anything(internalGetModel().getRowIdFor(object)));
	getContext().synch(0);
	return fIsNodeExpanded;
}
/**
 * Returns true if this tree has been configured to display its root.
 */
public boolean isRootVisible() {
	return fRootVisible;
}
/**
 * Internal method for collecting the currently selected nodes
 * into an Anything. Return the resulting Anything.
 */
protected Anything prepareSelectedNodes() {
	return new Anything(getSelectedOids());
}
/**
 * Unregisters the given listener from receiving notifications when a node 
 * in the tree is collapsed.
 *
 * @param listener	The object interested in my treeEvents.
 */
public void removeTreeCollapseActionListener(ITreeListener listener) {
	disableOptionalEvent("nodeCollapsed");
	internalRemoveListener("nodeCollapsed", listener);
}
/**
 * UnRegisters the given listener from receiving notifications when a 
 * (double-click) is performed on a node.
 *
 * @param listener	The object interested in my treeEvents.
 */
public void removeTreeDefaultActionListener(ITreeListener listener) {
	disableOptionalEvent("defaultAction");
	internalRemoveListener("defaultAction", listener);
}
/**
 * UnRegisters the given listener from receiving notifications when a 
 * node in the tree is expanded.
 *
 * @param listener	The object interested in my treeEvents.
 */
public void removeTreeExpandActionListener(ITreeListener listener) {
	disableOptionalEvent("nodeExpanded");
	internalRemoveListener("nodeExpanded", listener);
}
/**
 * UnRegisters the given listener from receiving notifications when there 
 * is a change in the selected nodes of the tree.
 *
 * @param listener	The object interested in my treeEvents.
 */
public void removeTreeSelectionChangedListener(ITreeListener listener) {
	internalRemoveListener("select", listener);
}
/**
 * Save the reference of the receiver's itemList to the specified
 * Anything. Do nothing if the itemlist is null.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveItemList(Anything a) {
	if (fModel != null || fItemList != null)
		a.put("itemList", getItemList().getRef(fContext));
}
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	if (fSelectedOids.length > 0)
		a.put("selectedNodes", prepareSelectedNodes());
	if (!fRootVisible)
		a.put("rootVisible", new Anything(fRootVisible));
}
/**
 *  Update the UI with the newly selected items.
 */
protected void sendSelectedItems() {
	Vector nodes = getSelectedNodesFrom(getSelectedOids());
	fContext.startBatchingRequests();
	getTreeItemList().sendPaths(nodes);
	sendUI("setSelectedNodes", prepareSelectedNodes());
	fContext.stopBatchingRequests();
}
/**
 * Set the <code>IHierarchicalItemList</code> that will serve as intermediary between
 * the receiver and its model.
 *
 * IMPORTANT: If the itemList passed does not know its tableModel, this method will throw
 * an exception. When connecting widgets to an itemList visually the connection 
 * between the itemList and its tableModel must be fired prior to the connection to the
 * widget. The sequence of connection execution can be modified in the builder.
 *
 * if <code>itemList</code> is null, the default itemList of the receiver's model
 * will be used by default.
 *
 * @see #getItemList
 *
 * @param itemList	The <code>IHierarchicalItemList</code> to be used from now on
 */
public void setItemList(IHierarchicalItemList itemList) {
	super.setItemList(itemList);
}
/**
 * Set the <code>IItemList</code> that will serve as intermediary between
 * the receiver and its ULCTableModel.
 *
 * IMPORTANT: If the itemList passed does not know its model, this method will throw
 * an exception. When connecting widgets to an itemList visually the connection 
 * between the itemList and its tableModel must be fired prior to the connection to the
 * widget. The sequence of connection execution can be modified in the builder.
 *
 * if <code>itemList</code> is null, the default itemList of the receiver's model
 * will be used by default.
 *
 * @see #getItemList
 *
 * @param itemList	The <code>IItemList</code> to be used from now on
 */
public void setItemList(IItemList itemList) {
	setItemList((IHierarchicalItemList) itemList);
}
/**
 * Set the <code>ULCTreeModel</code> that will serve as my data source.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * @param model	The <code>ULCTreeModel</code>
 */
public void setModel(IHierarchicalTableModel model) {
	internalSetModel((ULCAbstractTableModel) model);
}
/**
 * Sets whether the root is visible or not.
 *
 * @param rootVisible : set to true if you want the root to be visible
 *						on the tree.
 */
public void setRootVisible(boolean rootVisible) {
	if (rootVisible != fRootVisible) {
		fRootVisible = rootVisible;
		sendUI("setRootVisible", new Anything(fRootVisible));
	}
}
/**
 * Set the selected node at <code>obj</code>
 *
 * @param obj 	The node to be selected. obj = null will result in de-selection.
 * @deprecated with R3.0, use #setSelectedItem instead.
 */
public void setSelectedNode(Object obj) {
	setSelectedItem(obj);
}
/**
 * Set the selected nodes of the tree.
 * If any of these nodes have not yet been uploaded to the UI, they are uploaded right
 * away, so that the selection can be set correctly.
 *
 * @param nodes : The vector of nodes to be selected. nodes= null will result in de-selection.
 * @deprecated with R3.0, use #setSelectedItems instead.
 */
public void setSelectedNodes(Vector nodes) {
	setSelectedItems(nodes);
}
/**
 * Set the current selection mode of the receiver
 *
 * @param selectionMode	The selection mode which can be one of:
 * <pre>
 *  					TREE_SINGLE_SELECTION
 *  					TREE_CONTIGUOUS_SELECTION
 *  					TREE_DISCONTIGUOUS_SELECTION
 * </pre>
 */
public void setSelectionMode(int mode) {
	if (mode != fSelectionMode) {
		fSelectionMode = mode;
		sendUI("setSelectionMode", new Anything(mode));
	}
}
/**
 * Set the <code>ULCTreeModel</code> that will serve as my data source.
 * If the model being set is already uploaded to the UI this call can 
 * be used to switch the data source for the widget without requiring additional
 * round trips to the application to retrieve the data.
 *
 * @param model	The <code>ULCTreeModel</code>
 */
public void setTreeModel(IHierarchicalTableModel model) {
	setModel(model);
}
}
