package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.util.*;
import com.ibm.ulc.util.*;
import com.ibm.ulc.comm.*;
import java.beans.*;

/**
 * A ULCTableModel is a default implementation for a 'remote' model, that is a model which is split
 * across the UIEngine and the ULC application side. It implements the protocol defined by the UIEngine
 * for synchronizing two model halfs in terms of a simple getValue/setValue API.
 * Clients have to subclass from ULCTableModel to at least implement the <code>getValueAt</code> and
 * <code>getRowCount</code> methods.
 * Typically subclasses are not storing the data itself but are adapters which redirect the getValueAt and
 * setValueAt calls to some other model or data structure.
 */
abstract public class ULCAbstractTableModel extends ULCAbstractModel implements IEnabler, PropertyChangeListener, ITableModel {
	/**
	 * Internal use only.
	 * @serial	 
	 */
	private static final boolean DEBUG = false;
	/**
	 * The number of rows of data to be transmitted in a single request.
	 */
	private int fPrefetchCount;
	/**
	 * Internal use only.
	 * @serial	 	 
	 */
	protected boolean fGotReply;
	/**
	 * Determines whether changes are sent to the application for confirmation. 
	 */
	private boolean fVeto;
	/**
	 * The current notification policy.
	 * <pre>
	 * Valid values are :
	 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
	 * 	TABLE_EDIT_UPDATE_ON_REQUEST
	 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
	 * </pre>
	 * @serial	 	 
	 */
	/**
	 * @serial	 
	 */
	protected int fNotificationPolicy;
	/**
	 * The row cache which stores the attribute data for the receiver.	 
	 */
	private IRowCache fRowCache;
	/**
	 * The list of columns to be preloaded to the UI 
	 */
	private Vector fPreloadCols;
	/**
	 * The last oid assigned for rows in the receiver 
	 */
	protected int fLastOid;
	/**
	 * The set of itemLists which can maintain different row-orderings on the
	 * receiver's data.
	 * The default itemList maintains the order of rows the way the receiver
	 * did in releases prior to R1.6, and is not part of this list.
	 */
	protected IItemList[] fItemLists;
/**
 * Constructs a new instance of the receiver
 */
public ULCAbstractTableModel() {
	super();
}
/**
 * Constructs a new instance of the receiver with the specified number of prefetch rows.
 *
 * @param prefetch 	int	determines how many rows of data are transmitted in a single request.
 */
public ULCAbstractTableModel(int prefetch) {
	this();
	fPrefetchCount = prefetch;
}
/**
 * Constructs a new instance of the receiver with the specified number of prefetch rows 
 * and notification policy
 *
 * @param prefetch determines how many rows of data are transmitted in a single request.
 * @param policy	The notification policy
 * <pre>
 * Valid values are :
 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
 * 	TABLE_EDIT_UPDATE_ON_REQUEST
 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
 * </pre>
 */
public ULCAbstractTableModel(int prefetch, int policy) {
	this(prefetch);
	fNotificationPolicy = policy;
}
/**
 * Constructs a new instance of the receiver with the specified number of prefetch rows and 
 * notification policy.
 * If veto is true then all updates are sent to the application for confirmation.
 *
 * @param prefetch 	determines how many rows of data are transmitted in a single request.
 * @param veto 		determines whether changes of a cell modify the UI Engine's cache
 *					directly or whether they are reported back to the application via the setValueAt calls.
 * @param policy	The notification policy
 * <pre>
 * Valid values are :
 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
 * 	TABLE_EDIT_UPDATE_ON_REQUEST
 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
 * </pre>
 */
public ULCAbstractTableModel(int prefetch, boolean veto, int policy) {
	this(prefetch);
	fVeto = veto;
	fNotificationPolicy = policy;
}
/**
 * Constructs a new instance of the receiver
 *
 * @param boolean veto ; Determines whether changes of a cell modify the UI Engine's 
 *							cache directly or whether they are reported back to the 
 *							application via the setValueAt calls.
 */
public ULCAbstractTableModel(boolean veto) {
	this();
	fVeto = veto;
}
/**
 * Constructs a new instance of the receiver with the specified veto and notification policy.
 * If veto is true then all updates are sent to the application for confirmation.
 *
 * @param veto 		determines whether changes of a cell modify the UI Engine's cache
 *					directly or whether they are reported back to the application via the setValueAt calls.
 * @param policy	The notification policy
 * <pre>
 * Valid values are :
 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
 * 	TABLE_EDIT_UPDATE_ON_REQUEST
 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
 * </pre>
 */
public ULCAbstractTableModel(boolean veto, int policy) {
	this(veto);
	fNotificationPolicy = policy;
}
/**
 * Add a new itemList to be served by the receiver.
 *
 * The receiver informs all itemLists of any change in the receiver's data.
 *
 * @param itemList	IItemList	The itemList being added.
 *
 * @see ULCItemListAbstract
 */
public void addItemList(IItemList itemList) {
	for (int i = 0; i < fItemLists.length; i++) {
		if (fItemLists[i] == itemList)
			return;
	}
	IItemList[] newList = new IItemList[fItemLists.length + 1];
	if (fItemLists.length > 0)
		System.arraycopy(fItemLists, 0, newList, 0, fItemLists.length);
	fItemLists = newList;
	fItemLists[fItemLists.length - 1] = itemList;
}
/**
 * Add all attributes that should be preloaded to the UI to the specified Vector.
 *
 * @param vectorOfPreloadAttributes	The <code>Vector</code> into which the receiver's
 									preloadAttributes should be added.
 */
protected void addPreloadAttributesInto(Vector vectorOfPreloadAttributes) {
	if (!getUploadAllAttributes())
		return;
	addPreloadTableAttributesInto(vectorOfPreloadAttributes);
}
/**
 * Preload the data of column with the specified attributeName to the UI.
 *
 * @param attributeName	The key of the attribute to be retrieved.
 */
public void addPreloadColumn(String attributeName) {
	String attr = attributeName;
	if (attr == null)
		attr = "null";
	if (!getPreloadColumns().contains(attr))
		getPreloadColumns().addElement(attr);
}
/**
 * Add any preloaded rows to anything.
 *
 * The rows actually preloaded are defined by the receiver's itemLists.
 *
 * @param 	anything	the <code>Anything</code> sent to the UI
 */
protected void addPreloadRows(Anything anything) {
	int[] rowIds = computePreloadedRowIds();
	if (rowIds.length == 0)
		return;
	Anything attrNames = new Anything(getPreloadAttributesWithColumns());
	Vector ranges = UlcRange.createFromIntArray(rowIds);
	Vector tables = new Vector(ranges.size());
	if (ranges.size() < (rowIds.length / 2)) {
		for (int i = 0; i < ranges.size(); i++) {
			Anything table = new Anything();
			UlcRange range = (UlcRange) ranges.elementAt(i);
			tables.addElement(table);
			table.put("rowids", range.toArray());
			table.put("of", range.fStartIndex);
			table.put("ot", range.fEndIndex);
			table.put("an", attrNames);
		}
		getData(new Anything(tables));
		for (int i = 0; i < tables.size(); i++) {
			((Anything) tables.elementAt(i)).remove("rowids");
		}
	} else {
		Anything table = new Anything();
		table.put("rowids", rowIds);
		table.put("an", attrNames);
		tables.addElement(table);
		getData(new Anything(tables));
	}
	for (int i = 0; i < tables.size(); i++) {
		((Anything) tables.elementAt(i)).remove("an");
	}
	anything.put("an", attrNames);
	anything.put("data", tables);
}
/**
 * Add all table attributes that should be preloaded to the UI to the specified Vector.
 * Get preloadAttributes from all my item lists, and andd them to the given vector.
 *
 * @param vectorOfPreloadAttributes	Vector	into which the receiver adds the
 *											table attributes it needs to preload.
 */
public void addPreloadTableAttributesInto(Vector vectorOfPreloadAttributes) {
	super.addPreloadTableAttributesInto(vectorOfPreloadAttributes);
	IItemList[] itemLists = allItemLists();
	for (int i = 0; i < itemLists.length; i++) {
		String[] listAttributes = itemLists[i].getPreloadedAttributes();
		for (int j = 0; j < listAttributes.length; j++) {
			if (!vectorOfPreloadAttributes.contains(listAttributes[j]))
				vectorOfPreloadAttributes.addElement(listAttributes[j]);
		}
	}
}
/**
 * Answer an array with all the receiver's itemLists, including the default itemList,
 * which is not contained in the <code>fItemLists</code>.
 */
protected IItemList[] allItemLists() {
	Vector lists = new Vector();
	allItemLists(lists);
	IItemList[] answer = new IItemList[lists.size()];
	lists.copyInto(answer);
	return answer;
}
/**
 * Add all itemLists of the receiver into the given vector.
 *
 * @param lists	Vector	The vector into which I add my itemLists.
 *
 * Note that this process does not add my defaultItemList, since that
 * is not part of my fItemLists state.
 */
protected void allItemLists(Vector lists) {
	for (int i = 0; i < fItemLists.length; i++)
		lists.addElement(fItemLists[i]);
}
/**
 * Store all attributes for the given objectId in a new Anything
 *
 * Only attriubutes included in <code>attributes</code> that have not previously been uploaded are
 * uploaded. If no attributes need upload, the receiver will answer an empty <code>Anything</code>.
 *
 * @param oid			int			The objectId whose attributes are to be uploaded.
 * @param attributes	String[]	The attributes to be uploaded
 */
protected Anything attributesForUpload(int oid, String[] attributes) {
	Anything answer = new Anything();
	if (oid > -1) {
		Anything values = new Anything();
		for (int i = 0; i < attributes.length; i++) {
			values.put(attributes[i], convert(fContext, getValueAtRowId(attributes[i], oid)));
		}
		if (values.size() > 0) {
			answer.put("oid", oid);
			answer.put("a", values);
		}
	}
	return answer;
}
/**
 * Send a request to the Table Model to throw away any current edit values.
 */
public void cancelInput() {
	sendUI("cancelInput");
}
/**
 * Send a request to the Table Model to throw away the current edit values of
 * the changes in the model's row.
 *
 * @param model ULCAbstractRowModel	The row model which should cancel current inputs.
 *
 * see ULCRowModel#cancelInput
 */
public void cancelInput(ULCAbstractRowModel model) {
	sendUI("cancelInput", model.getRef(fContext));
}
/**
 * Answer an array with all oids that should be preloaded.
 */
private int[] computePreloadedRowIds() {
	IItemList[] allItemLists = allItemLists();
	UlcHashtable oids = new UlcHashtable();
	if (getPreloadColumns().size() == 0) {
		for (int i = 0; i < allItemLists.length; i++) {
			int[] initialOids = allItemLists[i].initialOids();
			for (int j = 0; j < initialOids.length; j++) {
				oids.put(new Integer(initialOids[j]), new Integer(initialOids[j]));
			}
		}
	} else {
		for (int i = 0; i < getRowCount(); i++) {
			Integer oid = new Integer(getOidForIndex(i));
			oids.put(oid, oid);
		}
	}
	Enumeration o = oids.elements();
	int[] rowIds = new int[oids.size()];
	int counter = 0;
	while (o.hasMoreElements()) {
		rowIds[counter++] = ((Integer) o.nextElement()).intValue();
	}
	return rowIds;
}
/**
 * Answer an array holding all current oids of the receiver
 */
protected int[] getAllRowIds() {
	return getRowCache().getAllRowIds();
}
/**
 * Internal method to marshall the requested attributes into the specified Anything.
 */
void getAttributes(Anything args) {
	boolean usingRange = false;
	Anything table = new Anything();
	Anything attrNames = args.get("attrNames");
	table.put("attrNames", attrNames == null ? new Anything() : attrNames);
	Anything identifiers = args.get("rowids");
	if (identifiers == null) {
		int from = args.get("of").asInt(-1);
		if (from > -1) {
			usingRange = true;
			UlcRange range = new UlcRange(from, args.get("ot").asInt(-1));
			identifiers = Common.convertCollectionToAnything(range.toVector());
		} else {
			identifiers = new Anything();
		}
		table.put("rowids", identifiers);
		Anything data = new Anything();
		data.append(table);
		getData(data);
		if (usingRange) {
			table.remove("rowids");
		}
	}
	sendUI("setData", table);
}
/**
 * Internal method to marshall the requested attributes into the specified Anything.
 *
 * @param args		The <code>Anything</code> specifying the
 *					data required by the UI.
 */
void getData(Anything argCollection) {
	for (int ri = 0; ri < argCollection.size(); ri++) {
		Anything args = argCollection.get(ri);
		Anything attrArgs = args.get("an");
		if (attrArgs == null)
			attrArgs = new Anything();
		String[] attrNames = getPreloadAttributes(attrArgs);
		if (attrNames.length > 0) {
			Anything rowids = args.get("rowids");
			Anything attributeNames = new Anything(attrNames);
			Anything attributeValues = new Anything();
			for (int i = 0; i < rowids.size(); i++) {
				int oid = rowids.get(i).asInt(-1);
				Anything attrVal = new Anything();
				for (int j = 0; j < attrNames.length; j++) {
					if (oid >= 0) {
						Object val = getValueAtRowId(attrNames[j], oid);
						attrVal.append(convert(fContext, val));
					} else
						attrVal.append(new Anything());
				}
				attributeValues.append(attrVal);
			}
			args.put("an", new Anything(attrNames));
			args.put("av", attributeValues);
			args.put("rowids", rowids);
		}
	}
}
/**
 * Answer the receiver's current index based itemList. This API is used to provide a default itemlist
 * for list widgets and combobox.
 *
 * The receiver's <code>ULCItemlist</code> is responsible for the order in which
 * the receiver's rows are displayed.
 *
 */
abstract public IIndexedItemList getIndexedItemList();
/**
 * Return the index for the rowId specified. Answer -1 if the oid was not found.
 *
 * @param oid	int	The rowId whose index is required.
 */
protected int getIndexForOid(int oid) {
	return getRowIndexForOid(oid);
}
/**
 * Answer the receiver's current index based itemList. This API is used to provide a default itemlist
 * for list widgets and combobox.
 *
 * The receiver's <code>ULCItemlist</code> is responsible for the order in which
 * the receiver's rows are displayed.
 *
 */
protected IIndexedItemList getItemList() {
	return getIndexedItemList();
}
/**
 * Answer an array with all the receiver's itemLists, including the default itemList,
 * which is not contained in the <code>fItemLists</code>.
 */
protected int getNextOid() {
	return ++fLastOid;
}
/**
 * Returns the edit notification policy.
 *
 * @return policy The new notification policy.
 * <pre>
 * Valid values are :
 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
 * 	TABLE_EDIT_UPDATE_ON_REQUEST
 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
 * </pre>
 */
public int getNotificationPolicy() {
	return fNotificationPolicy;
}
/**
 * Return the row id for the index or <code>INVALID_OID</code> if not found
 *
 * @param index	int	The index for which the oid is required
 */
protected int getOidForIndex(int index) {
	return getIndexedItemList().getOidForIndex(index);
}
/**
 * Return the number of rows to prefetch.
 */
public int getPrefetchCount() {
	if (getPreloadColumns().size() == 0)
		return (fPrefetchCount);
	else {
		return getRowCount();
	}
}
/**
 * Answer the array of attributeNames that will be uploaded with the next bunch of
 * rows going to the UI.
 *
 * In addition to the attributes requested by the UI, additional attributeNames may be 
 * defined by the receiver, and any of its itemLists.
 *
 * @see ULCItemListAbstract#getPreloadedAttributes
 *
 * @param 	attributes	the <code>Anything</code> received from the UI
 */
private String[] getPreloadAttributes(Anything attributes) {
	Vector vector = attributes.toCollection();
	Vector preload = getPreloadAttributes();
	for (int i = 0; i < preload.size(); i++) {
		String attr = preload.elementAt(i).toString();
		if (!vector.contains(attr)) {
			vector.addElement(attr);
			attributes.append(new Anything(attr));
		}
	}
	String[] answer = new String[vector.size()];
	vector.copyInto(answer);
	return answer;
}
/**
 * Append the list of attributeNames that will be preloaded into the specified Anything.
 *
 *
 * @see ULCItemListAbstract#getPreloadedAttributes
 *
 */
private Anything getPreloadAttributesNames() {
	Vector vector = getPreloadAttributesWithColumns();
	Anything attributes = new Anything();
	for (int i = 0; i < vector.size(); i++) {
		attributes.append(new Anything((String) vector.elementAt(i)));
	}
	return attributes;
}
/**
 * Answer the array of attributeNames that will be uploaded with the next bunch of
 * rows going to the UI.
 *
 * In addition to the attributes requested by the UI, additional attributeNames may be 
 * defined by the receiver, and any of its itemLists.
 *
 * @see ULCItemListAbstract#getPreloadedAttributes
 */
protected Vector getPreloadAttributesWithColumns() {
	Vector vector = getPreloadAttributes();
	Vector preloadColumns = getPreloadColumns();
	for (int i = 0; i < preloadColumns.size(); i++) {
		if (!vector.contains(preloadColumns.elementAt(i)))
			vector.addElement(preloadColumns.elementAt(i));
	}
	return vector;
}
/**
 * Answer the set of columns which are preloaded for this receiver
 */
private Vector getPreloadColumns() {
	return fPreloadCols;
}
/**
 * Answer the application object defined by <code>rowId</code>.
 *
 * @param rowId	int	The identfier of the row being accessed.
 */
protected Object getRow(int rowId) {
	return getRowCache().getRowAtRowId(rowId);
}
/*
 * Answer the adapter for the receiver's rows. It is used to access the values of the rows.
 */
abstract protected IRowAdapter getRowAdapter();
/**
 * Override this method to return the requested row.
 *
 * @param rowIndex	int	The index of the row being accessed.
 */
public Object getRowAt(int rowIndex) {
	return getRow(getOidForIndex(rowIndex));
}
/**
 * Answer the object that handles the row caching and accessing for the receiver.
 */
protected IRowCache getRowCache() {
	return fRowCache;
}
/**
 * Override this method to return the number of rows of this table.
 */
abstract public int getRowCount();
/**
 * Return the row id for the given object, or -1 if not found
 *
 * @param object	Object	The object whose rowId is required
 */
protected int getRowIdFor(Object object) {
	if (object == null)
		return -1;
	else
		return getRowCache().getRowIdFor(object);
}
/**
 * Return the row ids for the given indices.
 * Answer -1 if an index is not found
 *
 * @param objects	Object[]	The objects whose rowIds are required
 */
protected int[] getRowIdsFor(Object[] objects) {
	int[] ids = new int[objects.length];
	for (int i = 0; i < objects.length; i++) {
		ids[i] = getRowIdFor(objects[i]);
	}
	return ids;
}
/**
 * Return the row index for the specified oid or -1 if not found
 *
 * @param oid	int	The identfier whose row index is required
 */
public int getRowIndexForOid(int oid) {
	return getIndexedItemList().getIndexForOid(oid);
}
/**
 * Answer the value of the value for attributeName of the row specified by rowIndex.
 *
 * @param attributeName	The key of the attribute to be retrieved.
 * @param rowIndex		The index of the row being accessed.
 */
public Object getValueAt(String attributeName, int rowIndex) {
	return getValueAtRowId(attributeName, getOidForIndex(rowIndex));
}
/**
 * Answer the value of the value for attributeName of the row specified by rowid.
 *
 * @param attributeName	The key of the attribute to be retrieved.
 * @param rowId			The oid of the row being accessed.
 */
protected Object getValueAtRowId(String attributeName, int rowId) {
	Object row = getRow(rowId);
	if (row == null)
		return null;
	else
		return getRowAdapter().getValue(attributeName, getRow(rowId));
}
/**
 * Returns true if changes are sent to the application for confirmation.
 *
 * @return veto 	determines whether changes of a cell modify the UI Engine's cache
 *					directly or whether they are reported back to the application via the setValueAt calls.
 */
public boolean getVeto() {
	return fVeto;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
void handleGetDataRequest(Anything args) {
	boolean usingRange = false;
	Anything table = new Anything();
	table.put("an", args.get("an"));
	if (!args.isDefined("rowids")) {
		usingRange = true;
		int from = args.get("of", -1);
		int to = args.get("ot", -1);
		if (from > -1 && to > -1) {
			UlcRange range = new UlcRange(from, to);
			table.put("rowids", new Anything(range.toArray()));
			table.put("ot", to);
			table.put("of", from);
		} else {
			table.put("rowids", new Anything());
		}
	} else {
		table.put("rowids", args.get("rowids"));
	}
	Anything coll = new Anything();
	coll.append(table);
	getData(coll);
	if (usingRange) {
		table.remove("rowids");
	}
	Anything answer = new Anything();
	answer.put("data", coll);
	answer.put("an", table.get("an"));
	table.remove("an");
	sendUI("setData", answer);
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("setData")) {
		setData(args);
		fGotReply = true;
		return;
	}
	if (request.equals("requestAttributes")) {
		handleGetDataRequest(args);
		return;
	}
	super.handleRequest(conn, request, args);
}
/**
 * Notification that the receiver's DefaultItemList has mapped index to an oid.
 * Default is to ignore this message.
 *
 * @param	index	int		The index which was mapped
 * @param	oid		int		The oid to which the index was mapped
 */
protected void indexWasMapped(int index, int oid) {
}
/**
 * Initialize the receiver.
 *
 */
protected void initialize() {
	super.initialize();
	fPrefetchCount = 20;
	fGotReply = false;
	fVeto = false;
	fNotificationPolicy = TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE;
	fPreloadCols = new Vector(0);
	fLastOid = 0;
	fItemLists = new IItemList[0];
	setRowCache();
}
/**
 * Notify UI TableModel that the contents of the receiver has changed.
 *
 * Every itemList may cause the receiver to upload a number of rows. The receiver does not
 * upload any rows on its own.
 *
 */
protected void internalNotifyContentsChanged() {
	startBatchingRequests();
	Anything args = new Anything();
	addPreloadRows(args);
	sendUI("contentsChanged", args);
	IItemList[] itemLists = allItemLists();
	for (int i = 0; i < itemLists.length; i++) {
		itemLists[i].notifyContentsChanged();
	}
	stopBatchingRequests();
}
/**
 * Notify UI TableModel that some rows have been added to the receiver.
 *
 * Have the itemlists uploade their information. Data is uploaded only if some widget
 * actually needs it.
 *
 * @param	rowIds	int[]	The oid's of the rows which were added
 *
 */
protected void internalNotifyRowsAdded(int[] rowIds) {
	startBatchingRequests();
	IItemList[] itemLists = allItemLists();
	for (int i = 0; i < itemLists.length; i++) {
		itemLists[i].notifyRowsAdded(rowIds);
	}
	stopBatchingRequests();
}
/**
 * Notify the UI that the rows identified by rowIds have changed.
 *
 * Only those rows whose uploaded attributes includes <code>attributeName</code> are uploaded.
 * If <code>attributeName</code> is null, the row will be refreshed completely, if it had previously
 * uploaded at least one attribute.
 *
 * @param	rowIds			int[]		The oid's of the rows which were added
 * @param	attributeNames	String[]	The attributeNames which need to be uploaded
 *
 */
protected void internalNotifyRowsChanged(int[] rowIds, String[] attributeNames) {
	startBatchingRequests();
	IItemList[] lists = allItemLists();
	for (int i = 0; i < lists.length; i++) {
		lists[i].notifyRowsChanged(rowIds, attributeNames);
	}
	sendRowsChanged(rowIds, attributeNames);
	stopBatchingRequests();
}
/**
 * Notify the UI that the rows identified by rowIds have changed.
 *
 * Only those rows whose uploaded attributes includes <code>attributeName</code> are uploaded.
 * If <code>attributeName</code> is null, the row will be refreshed completely, if it had previously
 * uploaded at least one attribute.
 *
 * @param	rowIds			int[]	The oid's of the rows which were added
 * @param	attributeName	String	The attributeName which need to be uploaded
 */
protected void internalNotifyRowsChanged(int[] rowIds, String attributeName) {
	if (attributeName == null)
		internalNotifyRowsChanged(rowIds, new String[0]);
	else
		internalNotifyRowsChanged(rowIds, new String[] {attributeName});
}
/**
 * Notify the UI that the rows identified by rowIds have been removed.
 *
 * Inform every itemList of the event.
 *
 * @param	rowIds	int[]	The oid's of the rows which were removed
 */
protected synchronized void internalNotifyRowsRemoved(int[] rowIds) {
	startBatchingRequests();
	Anything args = new Anything();
	args.put("rowids", new Anything(rowIds));
	sendUI("removeRows", args);
	IItemList[] lists = allItemLists();
	for (int i = 0; i < lists.length; i++) {
		lists[i].notifyRowsRemoved(rowIds);
	}
	for (int i = 0; i < rowIds.length; i++)
		getRowCache().resetRowId(rowIds[i]);
	stopBatchingRequests();
}
/**
 * Answer true if the receiver has been set up to listen for property changes of its rows.
 * Default is to answer null. Subclasses may override this method.
 *
 * @return boolean
 */
boolean isListeningForPropertyChange() {
	return false;
}
/**
 * Notify UI TableModel that either contents or size of
 * ULC TableModel has changed.
 *
 * @param type 	The kind of change.
 * <pre>
 * type can be one of:
 *	TABLE_MODEL_CONTENTS_CHANGED;
 *	TABLE_MODEL_ROWS_ADDED;
 *	TABLE_MODEL_ROWS_REMOVED;
 *	TABLE_MODEL_ROWS_CHANGED;
 * </pre>
 * @param attributeNames An array of all attributes that have changed.
 * @param objects		An array of all objects that have changed
 */
 void notify(int type, String[] attributeNames, Object[] objects) {
	if (type == TABLE_MODEL_CONTENTS_CHANGED) {
		internalNotifyContentsChanged();
		return;
	}
	Vector rowIds = new Vector();
	for (int i = 0; i < objects.length; i++) {
		int oid = getRowCache().getRowIdFor(objects[i]);
		if (oid > -1)
			rowIds.addElement(new Integer(oid));
	}
	int[] oids = new int[rowIds.size()];
	for (int i = 0; i < rowIds.size(); i++)
		oids[i] = ((Integer) rowIds.elementAt(i)).intValue();
	if (type == TABLE_MODEL_ROWS_ADDED) {
		internalNotifyRowsAdded(oids);
	}
	if (type == TABLE_MODEL_ROWS_REMOVED) {
		internalNotifyRowsRemoved(oids);
	}
	if (type == TABLE_MODEL_ROWS_CHANGED || type == TABLE_MODEL_CELL_CHANGED) {
		internalNotifyRowsChanged(oids, attributeNames);
	}
}
/**
 * Notify the receiver that the specified objects have been added.
 *
 * @param objects		An array of all objects that have changed
 */
public void notifyAdd(Object[] objects) {
	if (isListeningForPropertyChange()) {
		try {
			for (int i = 0; i < objects.length; i++) {
				((IPropertyChanger) objects[i]).addPropertyChangeListener(this);
			}
		}
		catch (ClassCastException ce) {
		}
	}
	notify(TABLE_MODEL_ROWS_ADDED, new String[0], objects);
}
/**
 * Notify the receiver that the specified object has been added.
 *
 * @param object		The object that has changed
 */
public void notifyAdd(Object object) {
	if (object != null)
		notifyAdd(new Object[] {object});
}
/**
 * Notify the receiver that the entire contents has changed. Subclasses using the propertyChange mechanism should
 * take care to remove themselves from the propertyChangeListeners of the existing rows.
 *
 */
public void notifyChange() {
	notify(TABLE_MODEL_CONTENTS_CHANGED, null, null);
}
/**
 * Notify the receiver that the specified objects have been changed.
 *
 * @param objects		An array of all objects that have changed
 */
public void notifyChange(Object[] objects) {
	notify(TABLE_MODEL_ROWS_CHANGED, new String[0], objects);
}
/**
 * Notify the receiver that the specified attributes in objects have been changed.
 *
 * @param attributeNames 	An array of all attributes that have changed.
 * @param objects			An array of all objects that have changed
 */
public void notifyChange(String[] attributesNames, Object[] objects) {
	notify(TABLE_MODEL_ROWS_CHANGED, attributesNames, objects);
}
/**
 * Notify the receiver that the specified object has been changed.
 *
 * @param object		The object that has changed
 */
public void notifyChange(Object object) {
	if (object != null)
		notifyChange(new Object[] {object});
}
/**
 * Notify the receiver that the specified object has been changed at the
 * given attribute
 *
 * @param attributeName 	The name of the attribute that has changed.
 * @param object			The object that has changed
 */
public void notifyChange(String attributesName, Object object) {
	if (object != null)
		notifyChange(new String[] {attributesName}, new Object[] {object});
}
/**
 * Notify the receiver that the specified objects have been deleted.
 *
 * @param objects		An array of all objects that have changed
 */
public void notifyDelete(Object[] objects) {
	if (isListeningForPropertyChange()) {
		try {
			for (int i = 0; i < objects.length; i++) {
				((IPropertyChanger) objects[i]).removePropertyChangeListener(this);
			}
		}
		catch (ClassCastException ce) {
		}
	}
	notify(TABLE_MODEL_ROWS_REMOVED, new String[0], objects);
}
/**
 * Notify the receiver that the specified object has been deleted.
 *
 * @param object		The object that has been deleted
 */
public void notifyDelete(Object object) {
	if (object != null)
		notifyDelete(new Object[] {object});
}
/**
 * Preload the data of column with the specified attributeName to UI.
 *
 * @param attributeName	The key of the attribute to be retrieved.
 * @deprecated since R1.7 use addPreloadColumn
 */
public void preload(String attributeName) {
	addPreloadColumn(attributeName);
}
/**
 * Prepare the data for upload for the rows and attributes specified.
 *
 * @param rowIds		the identifiers of rows to upload
 * @param attributes	the attributes to upload (in addition to the ones specified by the receiver and its itemLists)
 */
private Anything prepareRows(Anything rowIds, String[] attributes) {
	Anything answer = new Anything();
	for (int i = 0; i < rowIds.size(); i++) {
		int rowId = rowIds.get(i).asInt(-1);
		getRowCache().setUploaded(rowId);
		Anything rowAny = attributesForUpload(rowId, attributes);
		if (rowAny.size() > 0)
			answer.append(rowAny);
	}
	return answer;
}
/**
 * This method gets called when a bound property is changed.
 *
 * @param event 	A PropertyChangeEvent object describing the event source 
 *   				and the property that has changed.
 */

public void propertyChange(PropertyChangeEvent event) {
	int oid = getRowCache().getRowIdFor(event.getSource());
	if (oid == -1)
		return;
	internalNotifyRowsChanged(new int[] {oid}, event.getPropertyName());
}
/**
 * Remove an itemList from the receiver's itemLists.
 *
 * @param 	itemList	IItemList	The itemList to be removed
 * 
 * @see #addItemList
 */
public void removeItemList(IItemList itemList) {
	int index = -1;
	for (int i = 0; i < fItemLists.length; i++) {
		if (fItemLists[i] == itemList) {
			index = i;
			break;
		}
	}
	if (index > -1) {
		IItemList[] newList = new IItemList[fItemLists.length - 1];
		System.arraycopy(fItemLists, 0, newList, 0, index);
		if (index < (fItemLists.length - 1))
			System.arraycopy(fItemLists, index + 1, newList, index, (fItemLists.length - index));
		fItemLists = newList;
	}
}
/**
 * Reset the receiver's default indexed itemList.
 *
 * @param 	itemList	IIndexedItemList	The itemList to be reset
 */
abstract protected void resetDefaultItemList(IIndexedItemList itemList);
/**
 * answer true, if the two objects are the same.. 
 *
 * @param left	first object
 * @param right	second object
 */
protected boolean sameValue(Object left, Object right) {
	if (left == null || right == null)
		return (left == right);
	return left.equals(right);
}
/**
 * Synchronously request any changes of the FormModel.
 * Updated values are reported via the setValueAt method on this model 
 * while blocking in this method.
 * Calling this method will ensure all values changed in the UI are received
 * in the application.
 *
 * @returns true if the values have been recieved. Returns false if the connection failed
 * 			while waiting for the reply.
 */
public boolean saveInput() {
	sendUI("flush", new Anything());
	fGotReply = false;
	while ((!fGotReply) && !fContext.processNextRequest(0)) {
		try {
			Thread.sleep(100);
		}
		catch (Exception e) {
		}
	};
	return fGotReply;
}
/**
 * Synchronously request the changes currently in the model's row.
 *
 * Updated values are reported via the setValueAt method on this model 
 * while blocking in this method.
 * Calling this method will ensure all values of the specified row changed in 
 * the UI are received in the application.
 *
 * This methods is internally called by ULCAbstractRowModel#saveInput.
 *
 * @param model ULCAbstractRowModel	The row model whose changes are to be saved
 * @see ULCAbstractRowModel#saveInput
 *
 * @returns true if the values have been recieved. Returns false if the connection failed
 * 			while waiting for the reply.
 */
public boolean saveInput(ULCAbstractRowModel model) {
	Anything args = new Anything();
	args.put("oid", model.getRef(fContext));
	sendUI("flush", args);
	fGotReply = false;
	while ((!fGotReply) && !fContext.processNextRequest(0)) {
		try {
			Thread.sleep(100);
		}
		catch (Exception e) {
		}
	};
	return fGotReply;
}
/**
 * Add the receiver's default item lists to anything.
 *
 * @param anything	Anything	The object into which my state should be saved.
 */
abstract protected void saveItemLists(Anything anything);
/**
 * Save the state of this object on the supplied Anything.
 * Every ULCProxy object that needs to send state to the UI must 
 * override this method to save its state in the Anything and then
 * call the super class implementation.
 *
 * @param a	Anything	The object into which my state should be saved.
 */
protected void saveState(Anything a) {
	super.saveState(a);
	a.put("rows", getRowCount());
	if (fVeto)
		a.put("veto", fVeto);
	saveItemLists(a);
	addPreloadRows(a);
	a.put("preloadColumns", getPreloadAttributesNames());
	if (fNotificationPolicy != TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE)
		a.put("notificationPolicy", fNotificationPolicy);
}
/**
 * Notify the UI that the rows identified by rowIds have changed.
 *
 * Only those rows whose uploaded attributes includes <code>attributeName</code> are uploaded.
 * If <code>attributeName</code> is null, the row will be refreshed completely, if it had previously
 * uploaded at least one attribute.
 */
protected void sendRowsChanged(int[] rowIds, String[] attributeNames) {
	Anything rowIdAny = new Anything();
	for (int i = 0; i < rowIds.length; i++) {
		if (getRow(rowIds[i]) != null) {
			rowIdAny.append(new Anything(rowIds[i]));
		}
	}
	Anything args = new Anything();
	args.put("rowids", rowIdAny);
	Anything attributeNameAny = new Anything();
	for (int i = 0; i < attributeNames.length; i++) {
		attributeNameAny.append(new Anything(attributeNames[i]));
	}
	Anything data;
	if (attributeNames.length == 0)
		data = prepareRows(rowIdAny, getPreloadAttributes(attributeNameAny));
	else
		data = prepareRows(rowIdAny, attributeNames);
	args.put("data", data);
	args.put("clearAll", (attributeNames.length == 0));
	args.put("a", attributeNameAny);
	if (args.size() > 0)
		sendUI("rowsChanged", args);
}
/**
 * The UI has sent new data for storing in the associated rows.
 *
 * The data is set using the <code>setValueAt</code> API of the receiver regardless of the <code>fVeto</code> flag.
 * Subclasses must handle the veto case.
 *
 * @see #setValueAt
 *
 * @param args	the rowIds and associated data to be stored in the application objects.
 */
protected void setData(Anything args) {
	Anything rowIds = args.get("rowids");
	if (rowIds != null) {
		for (int i = 0; i < rowIds.size(); i++) {
			Anything changes = args.get("changes");
			Anything aa = changes.get(i);
			int oid = rowIds.get(i).asInt(-1);
			if (oid >= 0) {
				Anything columns = aa.get("cols");
				for (int j = 0; j < columns.size(); j++) {
					Anything column = columns.get(j);
					String attributeName = column.get("colId", null);
					Anything value = column.get("value");
					if (attributeName != null && value != null) {
						setValueAtRowId(convert(value), attributeName, oid);
					}
				}
			}
		}
	}
}
/**
 * Set the edit notification policy.
 *
 * @param policy The new notification policy.
 * <pre>
 * Valid values are :
 *	TABLE_EDIT_UPDATE_ON_FOCUS_CHANGE
 * 	TABLE_EDIT_UPDATE_ON_REQUEST
 *	TABLE_EDIT_UPDATE_ON_ROW_CHANGE
 * </pre>
 */
public void setNotificationPolicy(int policy) {
	if (fNotificationPolicy != policy) {
		fNotificationPolicy = policy;
		sendUI("setNotificationPolicy", new Anything(fNotificationPolicy));
	}
}
/**
 * Set the number of rows to prefetch. This number may be overridden 
 * by the receiver's itemlists
 *
 * @param prefetchCount	int 	The number of rows
 */
public void setPrefetchCount(int prefetchCount) {
	fPrefetchCount = prefetchCount;
}
/**
 * Set the receiver's RowCache.
 */
abstract protected void setRowCache() ;
/**
 * Set the receiver's RowCache.
 *
 * @param cache	IRowCache	The new row cache for the receiver
 */
protected void setRowCache(IRowCache cache) {
	fRowCache = cache;
}
/**
 * The default behavior causes all this formModel to send all 
 * attributes known to be used to the UI during the saveState.
 * Users can disable this behavior by setting this flag to false.
 *
 * Note: Once this attribute has been set to true we cannot allow
 * it to be set to false since the UI will not request attributes
 * once marked as preload.
 *
 * @param uploadAllAttributes	boolean	Determines whether all attributes
 *										should be uploaded.
 */
public void setUploadAllAttributes(boolean uploadAllAttributes) {
	if (!getUploadAllAttributes())
		super.setUploadAllAttributes(uploadAllAttributes);
}
/**
 * Override this method to store the given value in the specified cell.
 * The default implementation prints the given argument on the console.
 *
 * @param value 		The new <code>Object</code> value to be set.
 * @param attributeName	The key of the attribute that has changed.
 * @param rowIndex		The index of the row that has changed.
 * 
 */
public void setValueAt(Object value, String attributeName, int rowIndex) {
	setValueAtRowId(value, attributeName, getOidForIndex(rowIndex));
}
/**
 * Override this method to store the given value in the specified cell.
 * The default implementation prints the given argument on the console.
 *
 * @param value 		The new <code>Object</code> value to be set.
 * @param attributeName	The key of the attribute that has changed.
 * @param rowId			The oid of the row that has changed.
 * 
 */
public void setValueAtRowId(Object value, String attributeName, int rowId) {
	getRowAdapter().setValue(attributeName, value, getRow(rowId));
}
/**
 * Determines whether changes are sent to the application for confirmation.
 *
 * If <code>fVeto</code> is set to <code>true</code>, changes originating with the UI must be
 * stored in the rows by the application, and the UI must be updated also.
 *
 * @param veto 		determines whether changes of a cell modify the UI Engine's cache
 *					directly or whether they are reported back to the application via the setValueAt calls.
 */
public void setVeto(boolean veto) {
	if (veto != fVeto) {
		fVeto = veto;
		sendUI("setVeto", new Anything(fVeto));
	}
}
/**
 * After this method is called all requests sent to the UI are kept pending till 
 * the stopBatchingRequests is called.
 */
protected void startBatchingRequests() {
	if (fContext != null)
		fContext.startBatchingRequests();
}
/**
 * After this method is called any pending requests are sent to the UI and the batching 
 * of requests is stopped.
 */
protected void stopBatchingRequests() {
	if (fContext != null)
		fContext.stopBatchingRequests();
}
/**
 * returns the fully qualified class name of my UI class
 */
protected String typeString() {
	return "TableModel";
}
}
