package com.ibm.ulc.application;

/**
 * The ULCAbstractModel class is the abstract superclass for all list, form and
 * row models. It implements the default preload attributes behavior for these ULC 
 * model classes
 */
import java.util.Vector;
import com.ibm.ulc.comm.ORBConnection;
import com.ibm.ulc.util.Anything;
public abstract class ULCAbstractModel extends ULCSharedProxy {
	/**
	 * The default behavior causes this model to not send any
	 * attributes with requested rows. If enabled, all known attributes 
	 * of the receiver's owners are uploaded with every row.
	 * Users can enable this behavior by setting this flag to true.
	 * @serial	 
	 */
	private boolean fUploadAllAttributes;
	/**
	 * The collection of attributes that are uploaded even if not requested when 
	 * a row is uploaded to the UI.
	 */
	protected Vector fPreloadAttributes;
/**
 * ULCAbstractModel subclasses act as model for list widgets
 * (in the ULCAbstractList hierarchy).
 */
public ULCAbstractModel() {
	super();
	initialize();
}
/**
 * Add a listener to be notified when this form model has been edited in the UI.
 * This method replaces addHasChangedListener(IValueChangedListener) because it
 * collided with the valueChangedEvent.
 *
 * @param	listener The <code>IChangedListener</code> object that will be notified.
 * @since	R3.1
 */
public void addHasChangedListener(IChangedListener listener) {
	enableOptionalEvent("hasChanges");
	internalAddListener("hasChanges", listener);
}
/**
 * Add the specified attribute to the list of attributes that will be uploaded to the
 * UI even if not requested by the UI when a row is uploaded for the first time. 
 *
 *
 * @param 	attributeName	the <code>String</code> attributeName 
 */
public void addPreloadAttribute(String attributeName) {
	if (fPreloadAttributes == null)
		fPreloadAttributes = new Vector();
	if (!fPreloadAttributes.contains(attributeName))
		fPreloadAttributes.addElement(attributeName);
}
/**
 * Add all attributes that should be preloaded to the UI to the specified Vector.
 *
 * @param vectorOfPreloadAttributes	The <code>Vector</code> into which the receiver's
 									preloadAttributes should be added.
 */
abstract protected void addPreloadAttributesInto(Vector vectorOfPreloadAttributes);
/**
 * Add all form attributes that should be preloaded to the UI to the specified Vector.
 *
 * @param vectorOfPreloadAttributes Vector : The vector into which the receiver's
 *						form preloadAttributes should be added.
 */
public void addPreloadFormAttributesInto(Vector vectorOfPreloadAttributes) {
	Vector o = getOwners();
	for (int i = 0; i < o.size(); i++) {
		try {
			IModelOwner mo = (IModelOwner) o.elementAt(i);
			mo.addPreloadFormAttributesInto(vectorOfPreloadAttributes);
		} catch (Exception e) {
		}
	}
}
/**
 * Add all table attributes that should be preloaded to the UI to the specified Vector.
 * Ask all my owners to add their table preload attributes into the given vector.
 *
 * @param vectorOfPreloadAttributes	Vector	into which the receiver adds the
 *											table attributes it needs to preload.
 */
public void addPreloadTableAttributesInto(Vector vectorOfPreloadAttributes) {
	Vector o = getOwners();
	for (int i = 0; i < o.size(); i++) {
		try {
			IModelOwner mo = (IModelOwner) o.elementAt(i);
			mo.addPreloadTableAttributesInto(vectorOfPreloadAttributes);
		} catch (Exception e) {
		}
	}
}
/**
 * Distributes ULCEvents to listeners implementing the corresponding listener interface.
 *
 * @param	eventKey 	The <code>String</code> key of the event.
 * @param	state		The <code>Boolean</code> state if the event.
 * @since	R3.1
 */
protected void distributeHasChangedToListeners(String eventKey, Boolean state) {
	ULCValueChangedEvent oldEvent= new ULCValueChangedEvent(this, state);
	ULCHasChangedEvent newEvent= new ULCHasChangedEvent(this, state);
	if (fListeners != null) {
		Object value = fListeners.get(eventKey);
		if (value == null)
			return;
		Vector list = (Vector) value;
		java.util.Enumeration e = list.elements();
		while (e.hasMoreElements()) {
			IChangedListener listener = (IChangedListener) e.nextElement();
			if (listener != null)
				if (listener instanceof IValueChangedListener)
					oldEvent.dispatch(listener);
				else if (listener instanceof IHasChangedListener)
					newEvent.dispatch(listener);
		}
	}
}
/**
 * Answer the array of attributeNames that will be uploaded with the next bunch of
 * rows going to the UI.
 *
 * In addition to the attributes requested by the UI, additional attributeNames may be 
 * defined by the receiver, and any of its itemLists.
 *
 * @see ULCItemListAbstract#getPreloadedAttributes
 */
protected Vector getPreloadAttributes() {
	Vector vector = new Vector();
	addPreloadAttributesInto(vector);
	if (fPreloadAttributes != null) {
		for (int i = 0; i < fPreloadAttributes.size(); i++) {
			String attr = (String) fPreloadAttributes.elementAt(i);
			if (!vector.contains(attr))
				vector.addElement(attr);
		}
	}
	return vector;
}
/**
 * The default behavior causes all this formModel to send all 
 * attributes known to be used to the UI during the saveState.
 * Users can disable this behavior by setting this flag to false.
 */
public boolean getUploadAllAttributes() {
	return fUploadAllAttributes;
}
/**
 * The UI has sent a request to this object. Do all processing necessary.
 * If this object does not handle this request call super.handleRequest.
 *
 * @param conn		ORBConnection	The connection on which the reply should be sent.
 * @param request 	String			The string that identifies this request.
 * @param args		Anything		The arguments associated with this request.
 */
public void handleRequest(ORBConnection conn, String request, Anything args) {
	if (request.equals("event")) {
		String type = args.get("type", "???");
		if (type.equals("hasChanges")) {
			distributeHasChangedToListeners(type, new Boolean(args.get("c").asBoolean(false)));
			return;
		}
	}
	super.handleRequest(conn, request, args);
}
/**
 * Initialize the receiver's state. Default is do nothing.
 */
protected void initialize() {
	fUploadAllAttributes = false;
	fPreloadAttributes = null;
}
/**
 * Remove the listener that was notified when this form model has been edited in the UI.
 * This method replaces addHasChangedListener(IValueChangedListener) because it
 * collided with the valueChangedEvent. 
 *
 * @param	listener The <code>IChangedListener</code> object that will be notified.
 * @since	R3.1
 */
public void removeHasChangedListener(IChangedListener listener) {
	disableOptionalEvent("hasChanges");
	internalRemoveListener("hasChanges", listener);
}
/**
 * Remove the specified attribute from the list of attributes that will be uploaded to the
 * UI even if not requested by the UI when a row is uploaded for the first time. 
 *
 *
 * @param 	attributeName	the <code>String</code> attributeName 
 */
public void removePreloadAttribute(String attributeName) {
	if (fPreloadAttributes == null)
		return;
	fPreloadAttributes.removeElement(attributeName);
}
/**
 * The default behavior causes all this formModel to send all 
 * attributes known to be used to the UI during the saveState.
 * Users can disable this behavior by setting this flag to false.
 *
 * Note: Once this attribute has been set to true we cannot allow
 * it to be set to false since the UI will not request attributes
 * once marked as preload.
 *
 * @param uploadAllAttributes	boolean	Determines whether all attributes
 *										should be uploaded.
 */
public void setUploadAllAttributes(boolean uploadAllAttributes) {
	fUploadAllAttributes = uploadAllAttributes;
}
}
