package com.ibm.ulc.application;

/*
 * Copyright (c) 1997,1998 Object Technology International Inc.
 */

import java.net.*;
import com.ibm.ulc.base.IDefaults;
import com.ibm.ulc.comm.*;
import com.ibm.ulc.util.Anything;
import com.ibm.ulc.util.Assert;

/**
 * ULC's static <code>run</code> method
 * is the entry point for running a ULC application.
 * ULC should not be subclassed.
 * ULC supports two modes: a 'client' and a 'server' mode.
 * In 'client' mode, the UI-engine is the server
 * waiting for the application (the ULC class) to connect to it.
 * In the 'server' mode the application (the ULC class)
 * starts up as a server and awaits connections from UI-engines.
 *
 * @see ULC#run(String[], Class)
 * @see ULC#startServer()
 */
public class ULC extends Server implements IDefaults {
	public static Class fgContextClass = null;
	public static boolean fgServerMode = false; // whether we should run as a server
	private static String fgUrlString = null; // the url to connect to or to serve on
	private static int fgServerPort = 0; // port to listen to when run as a server
	private static String fgServerProtocol = null; // the protocol to be used in server mode
	private static String fgUserParameter = null; // the optional userparameter specifed on the command line
	/**
	 * Toggle this flag to enable debugging output.
	 */
	public static boolean fgDebug = false; // flag for generating debug output

public ULC() {
	super("Application Server");
}
/**
 * Called when a new connection to this server is requested.
 * Hook method from class Server.
 */
public void createConnection(UlcTransport transport) {
	ULCContext context = createContext();
	createConnection(transport, context);
}
/**
 * Called when a new connection to this server is requested.
 * Hook method from class Server.
 */
public void createConnection(UlcTransport transport, ULCContext context) {
	if (context != null) {
		context.setAutoReconnect(false);
		try {
			context.connect(transport, new ThreadPool(1, true)); // as daemon
			if (!context.waitForInit())
				context.start();
		}
		catch (UlcTransportException ex) {
			if (fgDebug)
				System.out.println("ULC.createConnection(). Could not connect to socket!");
		}
	}
}
/**
 * Called when a new connection to this server is requested.
 * Hook method from class Server.
 * @deprecated
 */
public void createConnection(Socket socket) {
	System.out.println("ULC.createConnection(socket) : Should not be called. use createConnection(UlcTransport) instead");
}
/**
 * Tries to instantiate an instance of the class stored in fContextClass.
 * Returns null in case of failure.
 */
public static ULCContext createContext() {
	ULCContext context = null;
	if (fgContextClass != null) {
		try {
			Object o = fgContextClass.newInstance();
			if (o instanceof ULCContext)
				context = (ULCContext) o;
			initializeContext(context);
		} catch (InstantiationException e) {
			System.out.println("InstantiationException in createConnection");
		} catch (IllegalAccessException e) {
			System.out.println("IllegalAccessException in createConnection");
		}
	}
	return context;
}
public static Server createCorbaServer() {
	String serverClass = "com.ibm.ulc.corba.CorbaServer";
	try {
		Class cl = Class.forName(serverClass);
		if (cl != null)
			return (Server) cl.newInstance();
		System.out.println("ULC.createCorbaServer(): could not find " + serverClass);
	}
	catch (ClassNotFoundException e) {
		System.out.println("ULC.createCorbaServer(): ClassNotFoundException for " + serverClass + " - " + e.toString());
	}
	catch (InstantiationException e) {
		System.out.println("ULC.createCorbaServer(): InstantiationException for " + serverClass + " - " + e.toString());
	}
	catch (IllegalAccessException e) {
		System.out.println("ULC.createCorbaServer(): IllegalAccessException for " + serverClass + " - " + e.toString());
	}
	catch (ClassCastException e) {
		System.out.println("ULC.createCorbaServer(): ClassCastException for " + serverClass + " - " + e.toString());
	}
	return null;
}
/**
 * Returns the default URL for the UI Engine running as a server 
 * on the local machine on the default  port
 *
 */
static String getDefaultUIUrl() {
	return "ulc://localhost:2327";
}
/**
 * Parse and remember arguments.
 * Starts a server as a side effect if the application is run in 'server' mode.
 * Normally called from <code>run</code> convenience method,
 * but can be called directly if <code>run</code> is not used.
 */
public static boolean init(String args[]) {
	Common.printULCVersionAndProtocolVersion();
	boolean argsOK = true;
	for (int i = 0; i < args.length; i++) {
		String arg = args[i];
		if (arg.equals("-url")) {
			i++;
			if (i < args.length) {
				fgUrlString = args[i];
				// check if valid
				if (UlcConnection.getHostFromUrlString(fgUrlString) == null) {
					System.out.println("invalid URL for option -url");
					argsOK = false;
				} else {
					fgServerPort = UlcConnection.getPortFromUrlString(fgUrlString);
					fgServerProtocol = UlcConnection.getProtocolFromUrlString(fgUrlString);
				}
			} else {
				System.out.println("expecting URL for option -url");
				argsOK = false;
			}
			continue;
		}
		if (arg.equals("-genServerUrlFile")) {
			i++;
			if (i < args.length) {
				setServerUrlFileName(args[i]);
			} else {
				System.out.println("expecting file name for option -genServerUrlFile");
				argsOK = false;
			}
			continue;
		}
		if (arg.equals("-userparm")) {
			i++;
			if (i < args.length) {
				fgUserParameter = args[i];
			} else {
				System.out.println("expecting user parameter for option -userparm");
				argsOK = false;
			}
			continue;
		}
		if (arg.equals("-server")) {
			i++;
			if (i < args.length) {
				fgServerMode = true;
				try {
					// argument could be a port number
					fgServerPort = Integer.parseInt(args[i]);
					if (fgServerPort <= 0) {
						System.out.println("invalid port number for option -server");
						argsOK = false;
					} else {
						fgServerProtocol = "ulc";
						fgUrlString = "ulc://" + UlcConnection.getLocalHost() + ":" + fgServerPort;
					}
				} catch (NumberFormatException e) {
					// no port number -> argument must be a URL
					fgUrlString = args[i];
					// check if valid
					if (UlcConnection.getHostFromUrlString(fgUrlString) == null) {
						System.out.println("invalid URL for option -server");
						i--;
						argsOK = false;
					} else {
						fgServerPort = UlcConnection.getPortFromUrlString(fgUrlString);
						fgServerProtocol = UlcConnection.getProtocolFromUrlString(fgUrlString);
					}
				}
			} else {
				System.out.println("expecting port number or URL for option -server");
				argsOK = false;
			}
			continue;
		}
		if (arg.equals("-debug")) {
			fgDebug = true;
			com.ibm.ulc.comm.UlcConnection.fgDebug = true;
			com.ibm.ulc.comm.UlcTransport.fgDebug = true;
			continue;
		}
		if (arg.equals("-ipaddr")) {
			UlcConnection.fgIPAddr = true;
			continue;
		}
	}
	if (!argsOK)
		return false;
	if (fgUrlString == null) {
		fgServerPort = com.ibm.ulc.comm.Common.SERVER_PORT;
		fgServerProtocol = "ulc";
		fgUrlString = fgServerProtocol + "://" + UlcConnection.getLocalHost() + ":" + fgServerPort;
	}
	return true;
}
/**
 * Initialize the context after it has been created.
 */
public static void initializeContext(ULCContext context) {
	if (fgUserParameter != null)
		context.setUserServerParameter(fgUserParameter);
}
/**
 * This isused for indicating the 'mode'
 * in which the receiver is executing. In the
 * 'client' mode, the UI-engine is the 'server'
 * waiting for the application to connect to it;
 * in the 'server' mode it is the application 
 * which starts up and awaits connections from
 * UI-engines.
 */
protected static boolean isServerMode() {
	return fgServerMode;
}
/**
  * Returns true if the UIEngine is running on the localmachine
  * and default port. 
  */
public static boolean isUIEngineActive() {
	try {
		UlcTransport transport = UlcTransport.create(getDefaultUIUrl());
		transport.start();
		transport.close();
		return true;
	} catch (UlcTransportException ex) {
		return false;
	}
}
/**
 * Convenience method: kill UI Engine on a given host.
 */
public static void killUI(String host) {
	ULCContext context = new DefaultContext();
	try {
		context.connect("ulc://" + host, new ThreadPool(1, false));
	}
	catch (UlcTransportException ex) {
		if (fgDebug)
			System.out.println("ULC.killUI(). Could not connect!");
		return;
	}
	
	context.send(Common.FACTORY_ID, "kill", null);
}
/**
 * Register a factory class for creating the ULCContext object.
 */
public static void register(Class cl) {
	Assert.isTrue(fgContextClass == null);
	fgContextClass = cl;
}
/**
 * Convenience method, for initializing ULC, registering the class which implements
 * the <code>main</code> method, and depending on ULC mode starting up the server
 * ('server' mode) or connecting to the UI Engine ('client' mode).
 *
 * @param	args[] 		
 *			The arguments passed in for starting a ULC application.
 * @param	contextClass	
 *			This is the 'main' class in the ULC application,
 *			typically inheriting from ULCContext.
 * @return The ULCContext that has been created or null if an error occurred or is running InServerMode
 * @see ULCContext#start()
 */
public static ULCContext run(String args[], Class contextClass) {
	String vers = System.getProperty("java.version");
	if (vers.compareTo(MIN_JVM_VERSION) < 0)
		System.out.println("ERROR: ULC server must be run with a " + MIN_JVM_VERSION + " or higher VM version!");
	else {
		register(contextClass);
		if (init(args))
			return startServer();
	}
	return null;
}
/**
  * If running in 'client' mode, this method 
  * tries to connect to the given server.
  * Also, it provides the link back into the
  * ULC Application using the start() method,
  * which allows the ULCApplication to begin
  * building up it's widgets etc.
  *
  * @see ULCContext#start()
  */
public static ULCContext startServer() {
	ULCContext context = null;
	if (isServerMode()) {
		Server server = new ULC();
		server.start(fgUrlString); // start reverse server on this url
	} else {
		context = createContext();
		if (context != null) {
			ThreadPool reqProc = new ThreadPool(1, false);
			if (context.connectWithRetries(fgUrlString, reqProc)) {
				// send version information
				final ULCContext ctx = context; //1FG237U
				Request r = new Request() {
					public void dispatch() {
						ctx.sendInit();
						ctx.start();
					}
				};
				reqProc.addRequest(r);
			} else {
				if (fgDebug)
					System.out.println("ULC.startServer(). Could not connect to: " + fgUrlString);
				reqProc.shutdown();
				reqProc = null;
			}
		}
	}
	return context;
}
}
