package com.ibm.ulc.application;

import java.util.*;
import com.ibm.ulc.application.*;
import com.ibm.ulc.util.*;
/**
 * Implementation of a generic ULCTreeNode that can be used with the GenericTreeModel.
 *
 * @see GenericTreeModel
 * @see ULCTreeModel
 * @see ULCTreeTableModel
 */
public class GenericTreeNode extends UlcObject implements IGenericTreeNode, ITableRow {
	protected Object fUserObject = null;
	protected Vector fChildren = null;
	protected GenericTreeNode fParentNode = null;
	protected ULCIcon fIcon = null; //Used for ULCTreeModel only
	protected String fLabel = null; //Used for ULCTreeModel only
	protected GenericTreeModel fTreeModel = null;
	protected ITableRow fColumnData = null;
/**
 * Construct a new GenericTreeNode
 */
public GenericTreeNode() {
	fChildren = new Vector();
}
/**
 * Construct a new GenericTreeNode using the given GenericTreeModel.
 *
 * @param treeModel GenericTreeModel	The treeModel wherein the receiver is used
 */
public GenericTreeNode(GenericTreeModel treeModel) {
	this();
	fTreeModel = treeModel;
}
/**
 * Construct a new GenericTreeNode as child of the given parent node, and
 * using the given GenericTreeModel.
 *
 * @param treeModel GenericTreeModel	The treeModel wherein the receiver is used
 * @param parent	GenericTreeNode		The parent for the receiver
 */
public GenericTreeNode(GenericTreeModel treeModel, IGenericTreeNode parent) {
	this(treeModel);
	parent.addNode(this, false);
}
/**
 * Construct a new using the given GenericTreeModel, and with the given icon and label.
 *
 * Note : This method is used only when constructing the receiver for a ULCTree.
 *
 * @see ULCTree
 * @param treeModel GenericTreeModel	The treeModel wherein the receiver is used
 * @param label		String				The label to be set for the receiver
 * @param icon		ULCIcon				The icon to be set for the receiver
 */
public GenericTreeNode(GenericTreeModel treeModel, String label, ULCIcon icon) {
	this();
	fTreeModel = treeModel;
	fLabel = label;
	fIcon = icon;
}
/**
 * Add the given node to the receiver.
 * Notify the UI.
 *
 * @param treeNode	IGenericTreeNode	The node to be added.
 */ 
public void addNode(IGenericTreeNode treeNode) {
	addNode(treeNode, true);
}
/**
 * Add the given node to the receiver. Notify the UI of
 * this fact based on the notify flag.
 *
 * @param treeNode		IGenericTreeNode	The node to be added.
 * @param notify	boolean				Whether to notify the model after the
 *										addition, or not.
 */
public void addNode(IGenericTreeNode treeNode, boolean notify) {
	fChildren.addElement(treeNode);
	treeNode.setParent(this);
	if (notify)
		fTreeModel.notifyAdd(treeNode);
}
/**
 * Add the given nodes to the receiver.
 * Notify the UI.
 *
 * @param nodes		Vector		The nodes to be added.
 */
public void addNodes(Vector vectorOfTreeNodes) {
	Enumeration e = vectorOfTreeNodes.elements();
	while (e.hasMoreElements()) {
		addNode((IGenericTreeNode) e.nextElement(), false);
	}
	fTreeModel.notifyAdd(vectorOfTreeNodes);
}
/**
 * Answer the number of children in the receiver
 */
public int getChildCount() {
	return fChildren.size();
}
/**
 * Answer the receiver's children
 */
public Vector getChildren() {
	return fChildren;
}
/**
 * Answer the column data for the receiver.
 */
public ITableRow getColumnData() {
	if (fColumnData == null)
		fColumnData = new UlcHashtable();
	return fColumnData;
}
/**
 * Answer the icon corresponding to the receiver.
 * Note : This method will get called only if you are using the receiver
 * inside a ULCTree. However, it is recommended that R3.0 onwards, you use
 * the ULCTreeTable widget instead.
 *
 * @see ULCTree
 * @see ULCTreeTable
 */
public ULCIcon getIcon() {
	return fIcon;
}
/**
 * Answer the label corresponding to the receiver.
 * Note : This method will get called only if you are using the receiver
 * inside a ULCTree. However, it is recommended that R3.0 onwards, you use
 * the ULCTreeTable widget instead.
 *
 * @see ULCTree
 * @see ULCTreeTable
 */
public String getLabel() {
	return fLabel;
}
/**
 * Answer the parent node for the receiver.
 */
public IGenericTreeNode getParent() {
	return fParentNode;
}
/**
 * Answer the treeModel wherein I am being used.
 */
public GenericTreeModel getTreeModel() {
	return fTreeModel;
}
/**
 * Return the object identified by the key attributeName or null if not found
 *
 * @return java.lang.Object
 * @param attributeName java.lang.String
 */
public Object getValueAt(String attributeName) {
	return getColumnData().getValueAt(attributeName);
}
/**
 * Insert a ITreeNode after the specified ITreeNode object and notify the UI.
 */
public void insertNodeAfter(IGenericTreeNode treeNode) {
	int index = fChildren.indexOf(treeNode);
	if (index >= 0)
		insertNodeAtIndex(index + 1, treeNode, true);
}
/**
 * Insert a ITreeNode at the specified node index and notify the UI.
 */
public void insertNodeAtIndex(int index, IGenericTreeNode treeNode) {
	insertNodeAtIndex(index, treeNode, true);
}
/**
 * Insert a ITreeNode at the specified row index and notify the UI if notify is true.
 */
public void insertNodeAtIndex(int index, IGenericTreeNode treeNode, boolean notify) {
	fChildren.insertElementAt(treeNode, index);
	if (notify)
		fTreeModel.changed(this, true);
}
/**
 * Insert a ITreeNode before the specified ITreeNode object and notify the UI.
 */
public void insertNodeBefore(IGenericTreeNode treeNode) {
	int index = fChildren.indexOf(treeNode);
	if (index >= 0)
		insertNodeAtIndex(index, treeNode, true);
}
/**
 * Remove the specified ITreeNode and notify the UI.
 */
public boolean removeNode(IGenericTreeNode treeNode) {
	return removeNode(treeNode, true);
}
/**
 * Remove the specified ITreeNode from my children. Answer true if the
 * deletion was successful. If the given node is not my child, answer
 * false.
 * Notify the UI if the notify flag is true.
 */
public boolean removeNode(IGenericTreeNode treeNode, boolean notify) {
	int i = fChildren.indexOf(treeNode);
	if (i >= 0) {
		fChildren.removeElement(treeNode);
		if (notify)
			fTreeModel.notifyDelete(treeNode);
		treeNode.setParent(null);
		return true;
	}
	return false;
}
/**
 * Remove the specified ITreeNode collection and notify the UI.
 */
public void removeNodes(Vector vectorOfTreeNodes) {
	removeNodes(vectorOfTreeNodes, true);
}
/**
 * Remove the specified ITreeNode collection and notify the UI if
 * so specified.
 */
public void removeNodes(Vector vectorOfTreeNodes, boolean notify) {
	Enumeration e = vectorOfTreeNodes.elements();
	while (e.hasMoreElements()) {
		removeNode((IGenericTreeNode) e.nextElement(), false);
	}
	if (notify)
		fTreeModel.notifyDelete(vectorOfTreeNodes);
}
/**
 * Set the column data for the receiver.
 *
 * @param columnData	ITableRow	The values for the receiver's attributes
 */
public void setColumnData(ITableRow columnData) {
	fColumnData = columnData;
}
/**
 * Set the icon for the receiver.
 * Note : This method should be called only if you are using the receiver
 * inside a ULCTree. However, it is recommended that R3.0 onwards, you use
 * the ULCTreeTable widget instead.
 *
 * @see ULCTree
 * @see ULCTreeTable
 *
 * @param icon	ULCIcon		The icon to be set for the receiver
 */
public void setIcon(ULCIcon icon) {
	fIcon = icon;
}
/**
 * Set the label for the receiver.
 * Note : This method should be called only if you are using the receiver
 * inside a ULCTree. However, it is recommended that R3.0 onwards, you use
 * the ULCTreeTable widget instead.
 *
 * @see ULCTree
 * @see ULCTreeTable
 *
 * @param label		String		The label to be set for the receiver
 */
public void setLabel(String label) {
	fLabel = label;
}
/**
 * Set the receiver's parent
 *
 * @param parent	IGenericTreeNode	The node to set as my parent.
 */
public void setParent(IGenericTreeNode parent) {
	fParentNode = (GenericTreeNode) parent;
}
/**
 * Set the treeModel wherein I am being used.
 *
 * @param treeModel GenericTreeModel	The treeModel wherein the receiver is used
 */
public void setTreeModel(GenericTreeModel treeModel) {
	fTreeModel = treeModel;
}
/**
 * Set the object identified by the key attributeName to the specified value.
 *
 * @param attributeName java.lang.String
 * @param value java.lang.Object
 */
public void setValueAt(Object value, String attributeName) {
	getColumnData().setValueAt(value, attributeName);
}
}
