/* */
/********************************************************************

NAME: do-build.cmd driverName [ debug | beta ]

SAMPLE INVOCATION:
  rexx do-build.cmd 2001-03-09 debug

PURPOSE:
 This is the top level script to build the application "myApp".

NOTES:
 * A temporary file named "do-ver.out" is used.

CUSTOMIZATION NOTES:
 * See the sections labeled "CUSTOMIZATION REQUIRED".
 * This example has 3 variables that need to be added to the master.list
   file, because they change often and/or are set at build time.
   + They need to be defined in Section "Definition" and
   + They need to be added to the master.list in Section "Adding to List".
   The variables are:
    DRIVER_NAME      -> Identifier which might be related to the build date.
    BUILD_MODE       -> Normal (default), or Debug or Beta
    OPERATING_SYSTEM -> Using "uname -a" to find out the name.
 * The file "level.list" has the serviceability strings that do not change
   often. The contents of this file is copied into the "master.list".
 * The master.list file is a temporary file that contains all the
   serviceability strings. This file is generated every single time
   this build script is invoked and it is used inside the main make file
   (myApp-w.mak) which in turn, calls do-what.cmd.

*********************************************************************/

say "do-build.ksh: Starting a build of the myApp application"
say "date: " 
"@date /t"
say ""

/* Verify the input parameters */

parse arg driverName option
usage = "do-build.cmd driverName [ debug | beta ]"

/* CUSTOMIZATION REQUIRED: Section "Definition"
   Set up the variables that change frequently:
*/
if (driverName = "") then
do
  say "*** Error: need to specify the driver name."
  say usage
  exit 1
end
else
do
  DRIVER_NAME=driverName
end

BUILD_MODE='Normal'

/* Issue the "ver" command and store the output in a temporary file named
   "do-ver.out".
   The variable 'OPERATING_SYSTEM' will have the version of the operating system.
*/
verout = "do-ver.out"

"@ver >" verout
 do while lines(verout)
    parse value linein(verout) with OPERATING_SYSTEM
 end
 call lineout verout
 "@del" verout

/* Remove files */

/* CUSTOMIZATION REQUIRED:
   You need to list the files to be removed before doing a build.
*/

'@del what-info.h'       /* This is the common include header file. */
'@del what-info.c'       /* This is the generic include source file. */
'@del what-info-level.c' /* This is the include source file for level application. */
'@del master.list'       /* This is a temporary file, with the complete list of strings. */
'@del *.obj'
'@del *.exe'

/* CUSTOMIZATION REQUIRED:
   You need to handle the appropriate processing of input arguments.
   For example, the DEBUG and BETA options are set.
*/

/* Do the processing for the option */

if (option = "") then
do
  FLAGS=""
end
else
do
  if (option = "debug") then
  do
     FLAGS="/Ti+ /DDEBUG"
     BUILD_MODE='DEBUG'
  end
  else
  do
     if (option = "beta") then
     do
        FLAGS="/DBETA"
        BUILD_MODE='BETA'
     end
     else
     do
       say "*** Error: wrong option. If specified, it must be: debug or beta"
       say usage
       exit 1
     end
  end
end

/* CUSTOMIZATION REQUIRED:
   Specify the permanent file "level.list" with the variables that do not
   change often. This is the input file.
   Specify the temporary file "master.list" that will have all the
   strings. This is the output file.
*/

/* Open input file */
inputFile='level.list'
rcFile = stream( inputFile, C, 'open read' )
if ( rcFile \= "READY:" ) then
do
   say "*** Error while trying to open file: " inputFile
   say "    The return code is: " rcFile
   say "    Exiting now."
   exit 1
end

/* Open output file */
outputFile='master.list'
rcFile = stream( outputFile, C, 'open write' )
if ( rcFile \= "READY:" ) then
do
   say "*** Error while trying to open file: " outputFile
   say "    The return code is: " rcFile
   say "    Exiting now."
   exit 1
end

/* CUSTOMIZATION REQUIRED: Section "Adding to List"
   It is necessary to create the master.list file with the variables
  defined in this script. Follow the format below, which is also used
  in the file with those serviceability strings that do not change often,
  such as level.list. The format is a variableName (such as driverInfo)
  followed by a value which can be several strings.
*/

outLine="   driverInfo Driver: " DRIVER_NAME
temp = LINEOUT(outputFile, outLine);
outLine="buildModeInfo Build Mode: " BUILD_MODE
temp = LINEOUT(outputFile, outLine);
outLine=" platformInfo This code was generated in " OPERATING_SYSTEM
temp = LINEOUT(outputFile, outLine);

DO UNTIL ( LINES( inputFile ) = 0 )   /* Read until the end of file */
  outLine = LINEIN( inputFile )
  temp = LINEOUT(outputFile,outLine)
END  /* do until */

rc = STREAM(outputFile, C, 'close');
rc = STREAM(inputFile, C, 'close');

/* Invoke the make file */

say 'nmake -f myApp-w.mak all FLAGS='FLAGS
    'nmake -f myApp-w.mak all FLAGS="'FLAGS'"'
/* end of file */
