/********************************************************************/
/*                                                                  */
/* Licensed Materials - Property of IBM                             */
/*                                                                  */
/* US Government Users Restricted Rights - Use, duplication or      */
/* disclosure restricted by GSA ADP Schedule Contract with IBM Corp.*/
/*                                                                  */
/* OCO Source Materials                                             */
/*                                                                  */
/* IBM IMS TCP/IP OTMA Connection, Version 2, Release 1             */
/* (C) Copyright IBM Corp. 1997, 1998.  All Rights Reserved.        */
/*                                                                  */
/* The source code for this program is not published or otherwise   */
/* divested of its trade secrets, irrespective of what has been     */
/* deposited with the U.S. Copyright Office.                        */
/*                                                                  */
/********************************************************************/
#ifndef HWSCSTAT_HPP
#define HWSCSTAT_HPP

#include <sys/timeb.h>
#include "hws.h"
#include "hwscstat.h"

class HWSError;

/*--------------------------------------------------------------------*/
/*  HWSConvState is a base class.  It represents the state of a       */
/*  conversational transaction.                                       */
/*--------------------------------------------------------------------*/
class DllExport HWSConvState {

   // Public access
   public:

      // Member functions
      HWSConvState( char *hwsPath );                      // Construct object with
                                                          // null state data.
      HWSConvState( char *hwsPath, char *datFileName, HWSError *errObj );
                                                          // Construct object from
                                                          // contents of .DAT file.
      ~HWSConvState();                                    // Destructor.

      void setConvID( char *cID, HWSError *errObj );      // Max 8 bytes.
      void setConvIDP1( void *cIDP1 );                    // Always 8 bytes.
      void setConvIDP2( void *cIDP2 );                    // Always 8 bytes.
      void setRUName( char *RUNm, HWSError *errObj );     // Max 8 bytes.
      void setDateTime( long dT );                        // time in time_b is type
                                                          // time_t, a long.
      void setMillisecs( unsigned
                         short msecs );                   // Milliseconds.
      void setInLPageName( char *inLPNm, HWSError *errObj );
                                                          // Variable # of bytes.
      void setHWSTerm( char *termVal, HWSError *errObj ); // Continue or end conversation.
      void setHostName( char *hName, HWSError *errObj );  // Max 8 bytes.
      void setPort( unsigned
                         short pt );                      // Port.
      void setDataStore( char *dStore, HWSError *errObj  );
                                                          // Max 8 bytes.

      inline char       *getConvID()       { return convID; };
      inline void       *getConvIDP1()     { return convIDP1; };
      inline void       *getConvIDP2()     { return convIDP2; };
      inline char       *getRUName()       { return RUName; };
      inline long       getDateTime()      { return dateTime; };
      inline unsigned short
                        getMillisecs()     { return millisecs; };
      inline char       *getInLPageName()  { return inLPageName; };
      inline char       *getHWSTerm()      { return HWSTerm; };
      inline char       *getHostName()     { return hostName; };
      inline unsigned short
                        getPort()          { return port; };
      inline char       *getDataStore()    { return dataStore; };

      /*--------------------------------------------------------------*/
      /* Write attribute data of HWSConvState object to fn.DAT,       */
      /* where fn is formed from this RUName.                         */
      /*                                                              */
      /* Writes new fn.DAT file.  Returns error if already exists.    */
      /*--------------------------------------------------------------*/
      void saveState( HWSError *errObj );

      /*--------------------------------------------------------------*/
      /* Write attribute data of HWSConvState object to fn.DAT,       */
      /* where fn is formed from this RUName.                         */
      /*                                                              */
      /* Updates existing fn.DAT file.                                */
      /*--------------------------------------------------------------*/
      void updateState( HWSError *errObj );

      /*--------------------------------------------------------------*/
      /* Delete .DAT file corresponding to this state object.         */
      /*--------------------------------------------------------------*/
      void deleteState( HWSError *errObj );

      /*--------------------------------------------------------------*/
      /* Compare state data of HWSConvState object with state data    */
      /* in fn.DAT, where fn is formed from this RUName.              */
      /*                                                              */
      /* Returns true if they are the same; false otherwise.  If      */
      /* "out of sync", conversation ID and RUName from DAT file are  */
      /* returned.                                                    */
      /*--------------------------------------------------------------*/
      HWSBool isInSync( HWSError *errObj, char **retID, char **retRUNM );



   // Private access
   private:

      // Data members
      char       *convID;         // 8 character conversation ID - also
                                  // name of .DAT file.
      void       *convIDP1;       // Part 1 of conversation ID - a 64-bit
                                  // binary number (STCK value).  Stays in
                                  // host format.
      void       *convIDP2;       // Part 2 of conversation ID - a binary
                                  // number (host format) of the form
                                  // yyyydddFhhmmssth - used to form name
                                  // of DAT file.
      char       *RUName;         // Request Unit Name.
      long       dateTime;        // Date and time.
      unsigned
         short   millisecs;       // Milliseconds.
      char       *inLPageName;    // Name of next Input LPage.
      char       *HWSTerm;        // Conversation to be continued or ended.
      char       *hostName;       // Name of IMS host machine.
      unsigned
          short  port;            // Port of IMS host machine.
      char       *dataStore;      // Name of IMS (datastore).

      char       *HWSPath;        // IMS Web installation path.

      // Methods
      char *prepareState( HWSError *errObj );
      char *buildFName( char *datFileName, HWSError *errObj );
};

#endif
