;--------------------------------------------------------------
; BEAT.ASM converts the system time into the BEAT time, which
; represents the day in 1/1000 units. BEAT time is supposed to
; become the virtual world time, that is, everywhere on the
; world is exactly the same BEAT time. This is important to
; make appointments all over the world.
;
; You may adjust BEAT.ASM to your local timezone by filling
; in the appropriate MEWZ_DIFF value in this source code.
;
; - How to create BEAT.COM
;
; A86 Assembler:
;     a86 beat.asm              ;delete beat.sym
;
; Turbo Assembler:
;     tasm beat.asm
;     tlink /t/x beat.obj       ;delete beat.obj
;
; MASM Assembler:
;     masm beat.asm;
;     link beat.obj;
;     exe2bin beat.exe beat.com ;delete beat.exe and beat.obj
;
;--------------------------------------------------------------

MEWZ_DIFF EQU 0         ;fill in your timezone difference to
                        ;MEWZ = middle european winter time
                        ;(range -12 to +12) and recompile

cseg   segment 'code'
assume cs:cseg,ds:cseg,es:cseg
org    100h

  start:
jmp    begin            ;jmp across data

;---DATA-------------------------------------------------------

hour    db 0
minute  db 0
second  db 0
sec100  db 0
beat    dw 0
beat100 dw 0
crtmode db 0
cmdflag db 0
msw_sec dw 0
lsw_sec dw 0
base    dw 10           ;convert to decimal
modell  dw 0
curpos  dw 0
mewzdif dw MEWZ_DIFF    ;timezone
string  db 'hh:mm:ss = @xxx:xx$'
help    db 'BEAT V1.0 (c) 98 Stefan Peichl',13,10
        db 'Optional  cmd line parameters:',13,10
        db '+-hh     ;adjust your timezone',13,10
        db '@xxx:xx  ;convert beat to time',13,10
        db 'hh:mm:ss ;convert time to beat  $'

;---CODE-------------------------------------------------------

  begin: 

call   cmdpar           ;scan command line

;check for HPx00LX palmtop

mov    ax,4dd4h
int    15h              ;get model information
cmp    bx,'HP'
jne    no_hp            ;no HP
cmp    cx,0102h
jne    no_hp            ;no HP100LX or HP200LX
mov    modell,bx        ;remember
mov    curpos,(8*160)+(11*2) ;use this cursor position(8,11)

xor    ax,ax
mov    es,ax
mov    al,es:[049fh]    ;read bios data area extended crtmode
mov    crtmode,al       ;and remember
push   ds
pop    es               ;restore es
mov    ax,85h
int    10h              ;set 40x16 zoom mode
jmp    short no_cursor

  no_hp:    
mov    ah,0fh
int    10h              ;read screen mode
mov    crtmode,al       ;remember
mov    ax,3
int    10h              ;set 80x25 color CGA on desktops
mov    curpos,(11*160)+(30*2) ;use this cursor position (11,30)

  no_cursor:
mov    dx,1850h         ;hide cursor at line 24, col 80
xor    bx,bx
mov    ah,2
int    10h

  endless:              ;loop endless

;read and save system time

mov    ah,2ch
int    21h              ;get system time
mov    hour,ch          ;and store it
mov    minute,cl
mov    second,dh
mov    sec100,dl
call   writetime        ;write time into string
call   time2beat        ;convert time to beat
mov    si,offset string ;poiter to string
mov    di,curpos        ;pointer where to write in CGA memory
call   print            ;write string at cursor position

;check for abort

mov    ah,1
int    16h              ;check for any key
jz     endless          ;no key pressed
xor    ax,ax            ;otherwise
int    16h              ;through away key

xor    ax,ax
mov    al,crtmode
cmp    modell,'HP'
jne    reset_crt        ;no palmtop
xor    bx,bx
mov    es,bx
mov    es:[049fh],al    ;restore extended crtmode
  reset_crt:
int    10h              ;and set it

mov    ah,4ch
int    21h              ;quit

;---SUBROUTINE TIME2BEAT---------------------------------------

  time2beat:

;Convert time into 1/10 seconds since midnight

push   cx
push   di
xor    ax,ax
mov    al,hour
call   do_mewz
xor    dx,dx
mov    bx,36000
mul    bx               ;convert hours
mov    msw_sec,dx       ;most significant word
mov    lsw_sec,ax       ;least significant word
xor    ax,ax
mov    al,minute
mov    bx,600
mul    bx               ;convert minutes
add    lsw_sec,ax       ;add to converted hours
jnc    seconds          ;no overflow
inc    msw_sec          ;otherwise
  seconds:
xor    ax,ax
mov    al,second
mov    bx,10
mul    bx               ;convert seconds
add    lsw_sec,ax
jnc    second10
inc    msw_sec
  second10:
xor    ax,ax
mov    al,sec100
xor    dx,dx
div    bx               ;convert 1/100 seconds
add    lsw_sec,ax
jnc    in_beat
inc    msw_sec

;convert 1/10 seconds since midnight into beat

  in_beat:
mov    dx,msw_sec
mov    ax,lsw_sec
mov    cx,864           ;one day has 864000 1/10 seconds
div    cx               ;dx:ax/cx=1/1000 of one day
mov    bx,3             ;convert into 3 digits
mov    di,offset string + 12 ;pointer into string
call   wan              ;write to string
inc    di               ;skip over :
dec    bx               ;convert into 2 digits
mov    ax,dx            ;rest of dx:ax/cx
mov    cx,100           ;add some precision
mul    cx
mov    cx,864
div    cx               ;dx:ax/cx=1/100000 of a day
call   wan              ;write to string
pop    di
pop    cx
ret

;---SUBROUTINE DO_MEWZ-----------------------------------------

; adjusts hour to local time according to mewzdif

  do_mewz:
add    ax,mewzdif
or     ax,ax
jl     do_plus
cmp    ax,23
jle    do_ok
sub    ax,24
jmp    short do_ok
  do_plus:
add    ax,24
  do_ok:
ret

;---SUBROUTINE WRITETIME---------------------------------------

; writes the time values in the string

  writetime:
push   di
xor    ax,ax
mov    al,hour
mov    di,offset string ;point di to output string
mov    bx,2             ;convert into 2 digits
call   wan              ;write in string
inc    di               ;skip over :
mov    al,minute
call   wan              ;write in string
inc    di               ;skip over :
mov    al,second
call   wan              ;write in string
pop    di
ret

;---SUBROUTINE CMDPAR------------------------------------------

; scans the command line for valid input and acts upon it
; if the main program is not needed to be entered

  cmdpar:
mov    di,81h           ;tail begin
xor    cx,cx
mov    cl,[di-1]        ;tail length
or     cx,cx
je     cmd_ok           ;no tail
  cmd_loop:
mov    al,[di]          ;load next char
cmp    al,'@'
jne    cmd_1
call   cmd_beat         ;valid @
jmp    short cmd_check
  cmd_1:
xor    dx,dx            ;flag
cmp    al,'+'
jne    cmd_2
call   cmd_mewz         ;valid +
jmp    short cmd_check
  cmd_2:
inc    dx 
cmp    al,'-'
jne    cmd_3
call   cmd_mewz         ;valid -
jmp    short cmd_check
  cmd_3:
cmp    al,'0'
jl     cmd_next         ;invalid
cmp    al,'9'
jg     cmd_next         ;invalid
call   cmd_time         ;valid first number means time
  cmd_check:
jc     cmd_help         ;error in command line
  cmd_next:             ;continue on cmd line
inc    di
dec    cx
jg     cmd_loop         ;process next char
cmp    cmdflag,0
je     cmd_help         ;no valid parameter
cmp    cmdflag,1
je     cmd_ok           ;do main if only +- command found
mov    dx,offset string
jmp    short cmd_out
  cmd_help:             ;invalid input
mov    dx,offset help   ;pointer to help text
  cmd_out:
mov    ah,9
int    21h              ;print text
mov    ah,4ch           ;and exit program
int    21h
  cmd_ok:
ret

;---CMD_TIME ROUTINE-------------------------------------------
  
  cmd_time:
dec    di
inc    cx               ;get back one step
call   get_1or2         ;and get hour in bx
jc     cmd_time_stc     ;invalid input
cmp    bx,23            ;range check
jg     cmd_time_stc
mov    hour,bl
inc    di
dec    cx               ;skip over separator
call   get_1or2         ;try get minutes
jc     cmd_time1        ;no minutes entered
cmp    bx,59
jg     cmd_time_stc
mov    minute,bl
inc    di
dec    cx               ;skip over separator
call   get_1or2         ;try get seconds
jc     cmd_time1        ;no seconds there
cmp    bx,59
jg     cmd_time_stc
mov    second,bl
  cmd_time1:
call   writetime        ;write time into string
call   time2beat        ;and convert it to beat
or     cmdflag,4
clc
ret
  cmd_time_stc:
stc
ret

;---CMD_BEAT ROUTINE-------------------------------------------

  cmd_beat:
call   get_1or2         ;get beat (2 chars)
jc     cmd_beat_stc
inc    di
dec    cx
jle    cmd_beat1
xor    ax,ax
mov    al,[di]
call   aschex           ;get 3.char
jc     cmd_beat1        ;ignore invalid 3.char
push   ax
mov    ax,bx            ;put beat into right decimal position
mov    bx,10
xor    dx,dx
mul    bx
pop    bx
add    bx,ax            ;and add 3.char
  cmd_beat1:
mov    beat,bx
inc    di
dec    cx               ;skip over separator
call   get_1or2         ;get 1/100 beat
jc     cmd_beat2        ;not there
mov    beat100,bx
  cmd_beat2:
call   beat2time        ;convert beat:beat100 to time
clc
or     cmdflag,2
ret
  cmd_beat_stc:
stc
ret

;---CMD_MEWZ ROUTINE-------------------------------------------

  cmd_mewz:
call   get_1or2         ;get timezone
jc     cmd_mewz_stc     ;invalid
cmp    bx,12
jg     cmd_mewz_stc
or     dx,dx
je     cmd_mewz1        ;flag '+'
neg    bx               ;flag '-'
  cmd_mewz1:
mov    mewzdif,bx       ;remember timezone
test   cmdflag,2
je     cmd_mewz2
call   beat2time
jmp    short cmd_mewz3
  cmd_mewz2:
test   cmdflag,4
je     cmd_mewz3
call   time2beat
  cmd_mewz3:
or     cmdflag,1
clc
ret
  cmd_mewz_stc:
stc
ret

;---SUBROUTINE GET_1OR2----------------------------------------

; reads 1 or 2 chars from the command line and checks for
; number or separator

  get_1or2:
inc    di
dec    cx
jle    get_stc
xor    ax,ax
mov    al,[di]
call   aschex           ;convert ascii number to hex
jc     get_stc
push   ax
mov    bl,10
mul    bl               ;adjust decimal position
mov    bx,ax
inc    di
dec    cx
jle    get_end          ;only 1 char available
pop    ax
mov    al,[di]
call   aschex
jc     get_stc
add    bx,ax            ;bx=2 digit value
jmp    short get_clc
  get_end:
pop    bx
  get_clc:
clc
ret
  get_stc:
stc
ret

;---SUBROUTINE ASCHEX------------------------------------------

;convert ascii al to hex al

  aschex:
cmp    al,'0'
jl     asc_err
cmp    al,'9'
jg     asc_err
sub    al,'0'
clc
ret
  asc_err:
stc
ret

;---SUBROUTINE BEAT2TIME---------------------------------------

; converts beat:beat100 into time

  beat2time:
push   cx
push   di
xor    dx,dx
mov    ax,beat
mov    bx,1000          ;precision=1/1 000 000 part of a day
mul    bx
mov    cx,ax
mov    ax,beat100
mov    bx,10
mul    bl
add    ax,cx            ;add converted 1/100 beats
jnc    b2t_1
inc    dx
  b2t_1:
mov    bx,41667         ;1 hour = 1 000 000/24
div    bx
call   do_mewz          ;adjust hour
mov    hour,al
mov    ax,dx            ;use rest
xor    dx,dx
mov    bx,695           ;1 min  = 1 000 000/24/60
div    bx
mov    minute,al
mov    ax,dx            ;use rest
xor    dx,dx
mov    bx,11            ;1 sec  = 1 000 000/24/60/60
div    bx
mov    second,al
call   writetime        ;write time into string
mov    di,offset string+12
mov    bx,3
mov    ax,beat
call   wan              ;write beat into string
inc    di
dec    bx
mov    ax,beat100
call   wan              ;write beat100 into string
pop    di
pop    cx
ret

;---SUBROUTINE WAN---------------------------------------------

; WAN converts and stores the value in AX into a [BASE] number
; with BX digits and leading zeros starting at ES:DI. On return
; DI points to the next position in the string
    
  wan:
push   ax
push   bx
push   cx
push   dx
push   bp
mov    bp,base
xor    cx,cx
mov    cx,bx
  wan_1:
xor    dx,dx
div    bp
push   dx
loop   wan_1
mov    cx,bx
  wan_2:
pop    ax
add    al,'0'
cmp    al,'9'
jna    wan_3
add    al,7
  wan_3:
stosb
loop   wan_2
pop    bp
pop    dx
pop    cx
pop    bx
pop    ax
ret

;---SUBROUTINE PRINT-------------------------------------------

; PRINT writes a DOS string from DS:SI to ES:DI direct into
; CGA video memory to prevent screen flicker on desktops.

  print:
push    ax
push    es
mov     ax,0b800h       ;CGA screen segment
mov     es,ax
mov     ah,7            ;attribut normal
  print_loop:
lodsb
cmp     al,'$'          ;are we at the string end?
je      print_end       ;yes
stosw                   ;write char + attribute
jmp     short print_loop
  print_end:
pop     es
pop     ax
ret

cseg    ends
        end start

