Unit DirCache;

{
 ***** version 2.0 ***** 
 (c) Pavel Zampach (zampach@volny.cz), 2005
 GNU GENERAL PUBLIC LICENSE 
 Source code for Borland Delphi Compiler (originally ver. 7.0)
}

interface

uses
  Filer;

procedure cacheInit;
procedure cacheReset (const Dir : string);
function  isInCache (const Dir : string) : boolean;
function  cacheRead (var Data : TDirEntry) : boolean;
procedure cacheWrite (const Dir : string; const Data : TDirEntry);
procedure cacheDelete (DelName : string);

{ ------------------------------------------------------------------ }

implementation

uses
  SysUtils;
  
const
  MAX_ITEM   = 1024;
  EMPTY_ITEM = -1;

var
  ReadPtr  : integer;
  ReadHash, WriteHash : longint;
  CacheArr : array [1..MAX_ITEM] of TDirEntry;
  HashArr  : array [1..MAX_ITEM] of longint;

{ ------------------------------------------------------------------ }

(*************************************************************************)
(*                                                                       *)
(*                   General Hash Functions Library                      *)
(*                                                                       *)
(* Author: Arash Partow - 2002                                           *)
(* URL: http://www.partow.net                                            *)
(*                                                                       *)
(* Copyright notice:                                                     *)
(* Free use of the General Hash Functions Library is permitted under the *)
(* guidelines and in accordance with the most current version of the     *)
(* Common Public License.                                                *)
(* http://www.opensource.org/licenses/cpl.php                            *)
(*                                                                       *)
(*************************************************************************)
 
function APHash (const Strg : string) : longint;

var i : integer;

begin
  Result := 0;
  for i := 1 to length (Strg) do
    if (i and 1) = 0 then
      Result := Result xor ((Result shl 7) xor ord (upCase (Strg[i])) xor (Result shr 3))
    else
      Result := Result xor (not ((Result shl 11) xor ord (upCase (Strg[i])) xor (Result shr 5)));
  Result := Result and $7FFFFFFF;
end; 

{ ------------------------------------------------------------------ }

procedure cacheInit;

var i : integer;

begin
  WriteHash := EMPTY_ITEM;
  for i := 1 to MAX_ITEM do
    HashArr[i] := EMPTY_ITEM;
end;

{ ------------------------------------------------------------------ }

procedure cacheReset (const Dir : string);

begin
  ReadPtr  := 0;
  ReadHash := APHash (Dir);
end;

{ ------------------------------------------------------------------ }

function  isInCache (const Dir : string) : boolean;

var
  HashDir : longint;
  i : integer;
  
begin
  HashDir := APHash (Dir);
  i := 0;
  repeat
    inc (i);
    Result := (HashDir = HashArr[i]);
  until Result or (i = MAX_ITEM);
end;  

{ ------------------------------------------------------------------ }

function cacheRead (var Data : TDirEntry) : boolean;

begin
  if ReadPtr = MAX_ITEM then begin
    Result := false;
    exit;
  end;  

  repeat  
    inc (ReadPtr);
    Result := (HashArr[ReadPtr] = ReadHash)
  until Result or (ReadPtr = MAX_ITEM);

  Data := CacheArr[ReadPtr];
end;

{ ------------------------------------------------------------------ }

procedure cacheWrite (const Dir : string; const Data : TDirEntry);

var
  ClearHash : longint;
  i, j, k : integer;

begin
  if Dir <> '' then
    WriteHash := APHash (Dir);

  i := 0;
  repeat
    inc (i);
  until (i = MAX_ITEM) or (HashArr[i] = EMPTY_ITEM);
  
  if HashArr[i] <> EMPTY_ITEM then begin
    j := 0;
    repeat
      inc (j);
      ClearHash := HashArr[j];
    until (j = MAX_ITEM) or (ClearHash <> WriteHash);
    for k := j to MAX_ITEM do
      if HashArr[k] = ClearHash then
        HashArr[k] := EMPTY_ITEM;
    i := j;    
  end;
  
  CacheArr[i] := Data;
  HashArr[i]  := WriteHash;
end;

{ ------------------------------------------------------------------ }

procedure cacheDelete (DelName : string);

var
  HashDir : longint;
  i : integer;
  OK : boolean;
  
begin
  HashDir := APHash (extractFilePath (DelName));
  DelName := upperCase (extractFileName (DelName));

  i := 0;
  repeat
    inc (i);
    OK := ((HashArr[i] = HashDir) and (DelName = string (CacheArr[i].fname)));
  until (i = MAX_ITEM) or OK;
  
  if OK then
    HashArr[i] := EMPTY_ITEM;
end;

{ ------------------------------------------------------------------ }

end.