/* MAKEEXM.C -- Make a .EXE File Into a .EXM File

	Written March 1992 by Craig A. Finseth
	Copyright 1992 by Craig A. Finseth

This program takes one or two arguments:

The first argument is always the name of the .EXM file.  If no suffix
is supplied, the suffix ".EXM" is used.  This file is updated in place.

The second argument is optional.  If present, it gives the name of a
.MAP file.  The extension is always forced to ".MAP".  If not present,
the first argument is used to name the .MAP file.
*/

#include <stdio.h>
#include "fgenlib.h"

char t_attrib;			/* for libasm.asm */

unsigned Find(char *map_file);
FLAG PSetExt(char *name, char *ext, FLAG force);
void Update(char *exm_file, unsigned address);

/* ------------------------------------------------------------ */

int
main(argc, argv)
	int argc;
	char *argv[];
	{
	char *map_file;
	char *exm_file;
	unsigned address;

	if (argc < 2 || argc > 3) {
usage:
		xprintf("Make .EXM file program.\r\n\
usage is:  makeexm <.exm file> <.map file>\r\n");
		exit(1);
		}

	exm_file = argv[1];
	if (argc == 3)
		map_file = argv[2];
	else	map_file = argv[1];

		/* find value */

	address = Find(map_file);

		/* update .exm file */

	Update(exm_file, address);

	xprintf("Converted %s, data segment %x\r\n", exm_file, address);
	exit(0);
	}


/* ------------------------------------------------------------ */

/* Find the start of the data segment.  This is the first of either
_INIT_ or _DATA. Print a message and abort if error. */

unsigned
Find(char *map_file)
	{
	char fname[FNAMEMAX];
	char buf[BUFFSIZE];
	unsigned val;
	FILE *fp;

	xstrcpy(fname, map_file);
	PSetExt(fname, "MAP", TRUE);

	if ((fp = fopen(fname, "r")) == NULL) {
		xeprintf("Unable to open map file '%s'.\r\n", fname);
		exit(1);
		}

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		if (strlen(buf) < 27) continue;
		if (!strnequ(&buf[22], "_DATA", 5) && 
			 !strnequ(&buf[22], "_INIT_", 6))
			continue;

		buf[5] = NUL;
		if (!SToN(&buf[1], &val, 16)) continue;
		fclose(fp);
		return(val);
		}

	fclose(fp);
	xeprintf("Unable to find start of data segment in %s.\r\n", fname);
	exit(1);
	}


/* ------------------------------------------------------------ */

/* Set the extension part of a name to EXT.  If FORCE is TRUE, set the
extension even if there is one.  Return TRUE if the extension was set
or FALSE if it wasn't.  EXT doesn't include the period!! */

FLAG
PSetExt(char *name, char *ext, FLAG force)
	{
	char *cptr;
	char *c1;

	cptr = c1 = name + strlen(name);
	while (cptr > name && *cptr != '.' && *cptr != '/') cptr--;
	if (*cptr != '.') cptr = c1;
	if (force || !*cptr) {
		*cptr++ = '.';
		xstrcpy(cptr, ext);
		return(TRUE);
		}
	return(FALSE);
	}


/* ------------------------------------------------------------ */

/* Find the start of the data segment.  This is the first of either
_INIT_ or _DATA. Print a message and abort if error. */

void
Update(char *exm_file, unsigned address)
	{
	char fname[FNAMEMAX];
	char buf[BUFFSIZE];
	int amt;
	int fd;

	xstrcpy(fname, exm_file);
	PSetExt(fname, "EXM", FALSE);

	if ((fd = open(fname, 2, 0666)) < 0) {
		xeprintf("Unable to open exm file '%s'.\r\n", fname);
		exit(1);
		}

	if ((amt = read(fd, buf, sizeof(buf))) <= 0x1b) {
		xeprintf("Unable to read exm file %s.\r\n", fname);
		close(fd);
		exit(1);
		}

/* set the new magic number */
	buf[0] = 0x44;
	buf[1] = 0x4c;

/* set the data segment start */
	buf[0x1a] = address;
	buf[0x1b] = address >> 8;

	lseek(fd, 0L, 0);
	if (write(fd, buf, amt) != amt) {
		xeprintf("Unable to update exm file %s.\r\n", fname);
		close(fd);
		exit(1);
		}

	close(fd);
	}


/* end of MAKEEXM.C -- Make a .EXE File Into a .EXM File */
